%Copyright (2005) Alexander Karaivanov
%code that computes the linear program for the MH case
%called by: mlestep2.m
%calls: linprog (Matlab LP routine - slow) or lpcplex (CPLEX routine - fast but commercial!)

%inputs: 
%eta - talent random variable, eta (see paper)
%AA - wealth grid, edu - education
%output: F - probability of business, weighted by normal pdf

function [F]=lpcomputeMH(eta,AA,edu)

global  g g2 ka qqmax ww r l del0 del1 del2

%grid bounds
kmax=5;         %max capital
zmax=5;         %max effort
zmin=10^-4;     %min effort
cmax=10;        %max consumption

%Grid sizes
nk=16;      %capital
nz=10;      %effort
nq=2;       %output
nc=10;      %consumption
n=nk*nz*nq*nc;      %overall dimension

t=exp(del0+del1*log(AA)+del2*log(edu+1)+eta);   %construct talent variable

%Construct the grids
gridk=[0,logspace(-4,log10(kmax),nk-1)];        %capital
gridz=linspace(zmin,zmax,nz);       %effort
gridc=linspace(10^-8,cmax,nc);      %consumption

qmax = qqmax*t;         %high output - ents
w = qqmax;              %high output - workers

options=optimset('Display','off','TolFun',10^-9,'MaxIter',150,'TolX',10^-8);

%output grids
gridqw=[0,w];
gridqe=[0,qmax];

%Create the vectors - see MHmatlab.pdf write-up
cc=kron(ones(1,nk*nz*nq),gridc)';  %consumptions

qq=[kron(ones(1,nz),kron(gridqw,ones(1,nc))),...
        kron(ones(1,nz*(nk-1)),kron(gridqe,ones(1,nc)))]';      %outputs

zz=kron(ones(1,nk),kron(gridz,ones(1,nc*nq)))';         %efforts
zz2=kron(ones(1,nk-1),gridz)';

kk=kron(gridk,kron(1,ones(1,nc*nq*nz)))';           %capital
kk2=kron(gridk(2:nk),kron(1,ones(1,nz)))';

uu=(1-g)^-1*cc.^(1-g)-ka*(g2^-1)*zz.^g2;  %utilities

prt=(kk2.^l.*zz2.^(1-l))./(1+kk2.^l.*zz2.^(1-l));       %prob. of success - ents
ppe=[1-prt,prt]';

ppw=[1-gridz./(1+gridz);gridz./(1+gridz)];         %prob. of success - workers

%construct the production function (probabilities)
pp=zeros(n,1);
pp(1:nq*nz*nc)=kron(ppw(:),ones(nc,1));
pp(nc*nz*nq+1:n)=kron(ppe(:),ones(nc,1));

%Constructing the constraints
%1. Zero profit constraint (equality)
zp=(cc-qq-r*(AA-kk))';    %coefs on the ZPC

%2. Adding up constraint (equality)
au=ones(1,n);

%3. Incentive constraints (nk*(nz-1)) (inequalities)

%preparation
tm0=reshape(nc*nq*[0:nz*nk-1]',nz,nk);
tm01=tm0(2:nz,:);			%each column is inv index
ppos1=tm01(:);				%positions of not lowest eff.

tm2=repmat(nc*nq*[0:nk*nz-1],nq,1);
ppos11=tm2(:);
ppos3=nc*[0:nq*nz*nk-1]';

%construct the ICC
for i=1:nk*(nz-1)
   pos1=ppos1(i);
   pos2=ppos1(i)-nc*nq;
   IC(i,:)=sparse([zeros(1,pos1),(uu(pos2+1:pos2+nc*nq).*pp(pos2+1:pos2+nc*nq)./...
         pp(pos1+1:pos1+nc*nq))'-(uu(pos1+1:pos1+nc*nq))',zeros(1,n-nc*nq-pos1)]);      %the ICC
end

%4. Mother nature constraints (nq*nz*nk) (equalities)
for j=1:nk*nz*nq
   pos3=ppos3(j);
   pos1=ppos11(j);
    MN(j,:)=sparse([zeros(1,pos3),ones(1,nc),zeros(1,n-nc-pos3)]-...
            [zeros(1,pos1),pp(pos3+1)*ones(1,nc*nq),zeros(1,n-nc*nq-pos1)]);    %the MN
end

%Constructing the vectors of free coefs b
b=[zeros((nk-1)*(nz-1),1)];     %inequality constraints
beq=[0;1;zeros(nk*nz*nq,1)];    %equality constraints

%Constructing the matrix of coefs on the pi's
Aeq=[zp;au;MN];         %equality constr.
A=IC((nz-1)+1:nk*(nz-1),:);         %inequalities - note, no ICC for workers

%Actual linear programming
%Can use either Matlab's linprog or CPLEX - results may differ a little
x0=zeros(n,1);

%matrices used by CPLEX
A1=[A;Aeq];
b1=[b;beq];
le=[1:size(A,1)];

%the objective (the LP problem is min)
f=-uu;

%if use liprog
%[x,fval,eflag,nnn,la]=linprog(f,A,b,Aeq,beq,zeros(n,1),ones(n,1),[],options);

%if use cplex
[v1,x,lambda,flag1,colstat,it] = lpcplex(f,A1,b1,zeros(n,1),ones(n,1),le); 
%the pi's (from the paper) are in the vector x (here)

%compute prob. of business implied by LP solution
prob=1-sum(x(1:nc*nq*nz));

F=prob.*normpdf(eta,0,1);       %weight by normal pdf for the quadrature
