%Copyright (2005) Alexander Karaivanov
%code that computes the linear program for the MH case and generates
%simulated data (expected values)
%called by: mainsim.m
%calls: linprog (Matlab LP routine - slow) or lpcplex (CPLEX routine - fast but commercial!)

%inputs: 
%eta - talent random variable, eta (see paper)
%AA - wealth, edu - education
%outputs: ek - expected investment, ez - expected effort, ec - exp. cons.,
%epr - exp. prob. of business, eu - exp. utility, euc - exp. utility (cons.
%part only), icc - "1" denotes the ICC binds

%NOTE: this program is very similar to lpcomputeMH.m - see there for more
%details

function [ek,ez,ec,epr,eu,euc,icc]=simcomputeMH(eta,AA,edu)

global  g g2 ka qqmax ww r l del0 del1 del2

%grid bounds
kmax=5;
zmax=5;
zmin=10^-4;
cmax=10;

%Grid sizes
nk=16;     
nz=10;     
nq=2;  
nc=10;      
n=nk*nz*nq*nc;

t=exp(del0+del1*log(AA)+del2*log(edu+1)+eta);       %talent

%grids
gridk=[0,logspace(-4,log10(kmax),nk-1)];
gridz=linspace(zmin,zmax,nz);
gridc=linspace(10^-8,cmax,nc);

qmax = qqmax*t;
w = qqmax;

options=optimset('Display','off','TolFun',10^-9,'MaxIter',150,'TolX',10^-8);

gridqw=[0,w];
gridqe=[0,qmax];

%Create the vectors
cc=kron(ones(1,nk*nz*nq),gridc)';  %consumptions
qq=[kron(ones(1,nz),kron(gridqw,ones(1,nc))),...
        kron(ones(1,nz*(nk-1)),kron(gridqe,ones(1,nc)))]';

zz=kron(ones(1,nk),kron(gridz,ones(1,nc*nq)))';
zz2=kron(ones(1,nk-1),gridz)';

kk=kron(gridk,kron(1,ones(1,nc*nq*nz)))';
kk2=kron(gridk(2:nk),kron(1,ones(1,nz)))';

uu=(1-g)^-1*cc.^(1-g)-ka*(g2^-1)*zz.^g2;    %utilities
uuc=(1-g)^-1*cc.^(1-g);


prt=(kk2.^l.*zz2.^(1-l))./(1+kk2.^l.*zz2.^(1-l));

ppe=[1-prt,prt]';
ppw=[1-gridz./(1+gridz);gridz./(1+gridz)];

pp=zeros(n,1);

pp(1:nq*nz*nc)=kron(ppw(:),ones(nc,1));
pp(nc*nz*nq+1:n)=kron(ppe(:),ones(nc,1));

%Constructing the constraints
%===================================
%1. Zero profit constraint (equality)
zp=(cc-qq-r*(AA-kk))';
		
%2. Adding up constraint (equality)
au=ones(1,n);

%3. Incentive constraints (nk*(nz-1)) (inequalities)

tm0=reshape(nc*nq*[0:nz*nk-1]',nz,nk);
tm01=tm0(2:nz,:);			            %each column is inv index
ppos1=tm01(:);				            %positions of not lowest eff.
tm2=repmat(nc*nq*[0:nk*nz-1],nq,1);
ppos11=tm2(:);
ppos3=nc*[0:nq*nz*nk-1]';

for i=1:nk*(nz-1)
   pos1=ppos1(i);
   pos2=ppos1(i)-nc*nq;
   IC(i,:)=sparse([zeros(1,pos1),(uu(pos2+1:pos2+nc*nq).*pp(pos2+1:pos2+nc*nq)./...
         pp(pos1+1:pos1+nc*nq))'-(uu(pos1+1:pos1+nc*nq))',zeros(1,n-nc*nq-pos1)]);      %ICCs
end

%4. Mother nature constraints (nq*nz*nk) (equalities)
for j=1:nk*nz*nq
   pos3=ppos3(j);
   pos1=ppos11(j);
    MN(j,:)=sparse([zeros(1,pos3),ones(1,nc),zeros(1,n-nc-pos3)]-...
            [zeros(1,pos1),pp(pos3+1)*ones(1,nc*nq),zeros(1,n-nc*nq-pos1)]);
end

%Constructing the vector of free coefs b
b=[zeros((nk-1)*(nz-1),1)];
beq=[0;1;zeros(nk*nz*nq,1)];

%Constructing the matrix of coefs on pis
Aeq=[zp;au;MN];
A=IC((nz-1)+1:nk*(nz-1),:);         %workers no MH

%Actual linear programming
x0=zeros(n,1);

%needed for CPLEX
A1=[A;Aeq];
b1=[b;beq];
le=[1:size(A,1)];

f=-uu;

%if use liprog
%[x,fval,eflag,nnn,lambda]=linprog(f,A,b,Aeq,beq,zeros(n,1),ones(n,1),[],options);

%if use cplex
[v1,x,lambda,flag1,colstat,it] = lpcplex(f,A1,b1,zeros(n,1),ones(n,1),le); 

%COMPUTE EXPECTED VALUES of simulated data (expectation taken over
%lotteries at lin.prog. solution - i.e. over the pi's)
%the pi's (from the paper) are in the vector x (here)
ek=kk'*x;           %investment
ez=zz'*x;           %effort
ec=cc'*x;           %consumption
epr=1-sum(x(1:nc*nq*nz));  %prob. of being entrepreneur
eu=uu'*x;           %expected utility
euc=uuc'*x;         %Eu(c)


%Determine binding ICCs at equil. path
kx=kk(x>0);
zx=zz(x>0);
indk=1:length(gridk);
indz=1:length(gridz);

for ii=1:length(x(x>0))
    kpos=indk(abs(gridk-kx(ii))<10^-7);
    zpos=indz(abs(gridz-zx(ii))<10^-3);
    
    
    if kpos>1
        iccv(ii)=lambda((kpos-2)*(nz-1)+zpos-1);
    else
        iccv(ii)=0;
    end
end

icc=any(iccv);





