#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "rasterfile.h"
#include "basic.h"
#include "structN.h"
#include "bmeN.h"

#define  BILINEAR  22
#define  Percent   0.3
#define  IBLOCK


void block2pixel2(float *mvx, float *mvy, vector_ptr fmv, int cx, int cy, int xblk, int yblk, int hor, int ver, enum BlockMode block_mode);


void rec_find_uncovered_block(vector_ptr fmv, enum PixelMode *cstate, int x, int y, int xblock, int yblock, int hor, int ver);

void position(int *px, int *py, float pfx, float pfy, float mvx, float mvy, int hor, int ver);

float interpolate(float fx, float fy, float *frame, int hor, int ver, int type);

int get_unconnect(vector_ptr fmv, videoinfo info)
{
  int        hor, ver, xnum, ynum, xblock, yblock;
  int        x, y, X, Y, pos, unconnect=0;
  float *ymvx, *ymvy;

//  float      *upframe = NULL;

  hor  = info.ywidth; ver = info.yheight;
  xblock = info.xblk;   yblock = info.yblk;
  xnum = info.xnum;   ynum = info.ynum;

  ymvx = (float *) getarray(hor*ver, sizeof(float), "ymvx");
  ymvy = (float *) getarray(hor*ver, sizeof(float), "ymvy");

  for(y=0, Y=0 ; Y<ynum ; y+=yblock, Y++){
    for(x=0, X=0 ; X<xnum ; x+=xblock, X++){
      block2pixel2(ymvx, ymvy, &fmv[Y*xnum+X], x, y, xblock, yblock, hor, ver, DEFAULT);
    }
  }

  for(y=0; y<ver; y++)
	for(x=0;x<hor;x++)
	  if(ymvx[y*hor+x] == Infinity && ymvy[y*hor+x] == Infinity)
		unconnect++;

//  if(half == 1)
//    free(upframe);
  free(ymvx); free(ymvy); 

  return unconnect;
}




/*
 *    Whether_do_MCTF()
 * Find the percentage of unconnected pixels
 * fmv   : frame motion vectors.
 * fr1   : current frame.
 * fr0   : next frame.
 * info  : video information
 */
enum FLAG Whether_do_MCTF(vector_ptr fmv, YUVimage fr1, YUVimage fr0, int t_level, videoinfo info)
{
  int  hor, ver, xnum, ynum, xblk, yblk, unconnect, thresh;
  //int  x, y, X, Y, pos;
  enum PixelMode *pstate, *cstate;

  hor  = info.ywidth; ver = info.yheight;
  xblk = info.xblk;   yblk = info.yblk;
  xnum = info.xnum;   ynum = info.ynum;


  thresh = (int)(hor*ver*Percent);   // 30% of 960x540 = 155520    30% of 352x240 = 25344
  pstate = (enum PixelMode *)getarray(hor*ver, sizeof(enum PixelMode), "pstate");
  cstate = (enum PixelMode *)getarray(hor*ver, sizeof(enum PixelMode), "cstate");

	  
  //unconnect = find_uncovered_pixel(fmv, fr1, fr0, pstate, cstate, info);
#ifdef IBLOCK  
  find_unmatched_block(fmv, fr1, fr0, t_level, info); 
#endif
  unconnect = get_unconnect(fmv, info);

  free(pstate);
  free(cstate);

  if(unconnect > thresh){ 
	//printf("No MCTF: unconnected pixels %d (FME.c)\n", unconnected);
	return NO;  // donot do MCTF 
  }
  else{
    return YES;
  }
}

/*
 *   find_uncovered_pixel()
 * find multi-connected pixels only from motion vectors in DEFAULT mode
 * fmv   : frame motion vectors.
 * fr1   : current frame.
 * fr0   : next frame.
 * pstate: reference frame pixel state.
 * cstate: current frame pixel state.
 * info  : video information
 *
 * return:unconnected pixel number including multi-connected and intra mode pixels in the current frame
 */
int find_uncovered_pixel(vector_ptr fmv, YUVimage fr1, YUVimage fr0, enum PixelMode *pstate, enum PixelMode *cstate, 
						  videoinfo info)
{
  int   i, x, y, X, Y, yhor, yver, connect, unconnect;
  int cx, cy, px, py, cpos, *ppos_of, ppos;
  float pfx, pfy;
  int   xnum, ynum, xblk, yblk;
  float *mvx, *mvy;
  float *min_dpd, *dpd, ptmp;
  unsigned char *used;

  yhor = info.ywidth; yver = info.yheight;
  xnum = info.xnum;   ynum = info.ynum;
  xblk = info.xblk;   yblk = info.yblk; 

  ppos_of = (int *)getarray(yhor*yver, sizeof(int), "ppos_of");
  dpd = (float *)getarray(yhor*yver, sizeof(float), "dpd");
  used = (unsigned char *)getarray(yhor*yver, sizeof(unsigned char), "used");
  min_dpd = (float *)getarray(yhor*yver, sizeof(float), "min_dpd");

  for(i=0; i<yhor*yver; i++){
    pstate[i]=UNDEF; 
    cstate[i]=UNDEF; 
  }

 /* allocate the memory and initialization */
  mvx = (float *) getarray(yhor*yver, sizeof(float), "mvx");
  mvy = (float *) getarray(yhor*yver, sizeof(float), "mvy");

  /* convert the block-based vectors to the pixel-based vectors */
  for(y=0, Y=0 ; Y<ynum ; y+=yblk, Y++){
    for(x=0, X=0 ; X<xnum ; x+=xblk, X++){
      block2pixel2(mvx, mvy, &fmv[Y*xnum+X], x, y, xblk, yblk, yhor, yver, DEFAULT);
    }
  }

  for(cy=0 ; cy<yver ; cy++){
    for(cx=0 ; cx<yhor ; cx++){
      cpos = cy*yhor+cx;
	  if(mvx[cpos] != Infinity && mvy[cpos] != Infinity ){
        pfx = cx-mvx[cpos]; pfy = cy-mvy[cpos];
        position(&px, &py, pfx, pfy, mvx[cpos], mvy[cpos], yhor, yver);

        ppos = py*yhor+px;
	    ppos_of[cpos] = ppos;
        
        ptmp = interpolate(pfx, pfy, fr0.Y, yhor, yver, TYPE);//printf("out\n");
	
	    dpd[cpos] = (float)fabs(fr1.Y[cpos]-ptmp);
        if(pstate[ppos]==UNDEF){
          pstate[ppos]=UNI;
          min_dpd[ppos]=dpd[cpos];
		}
        else{
          pstate[ppos] = MULTI;
          if(dpd[cpos]<min_dpd[ppos])
            min_dpd[ppos]=dpd[cpos];
		}
	  }//mvx[cpos] != Infinity && mvy[cpos] != Infinity
    }
  }
 

  connect = 0; unconnect = 0;
  for(i=0; i<yhor*yver; i++){
    used[i]=0;
  }
  for(cy=0 ; cy<yver ; cy++){
    for(cx=0 ; cx<yhor ; cx++){
      cpos = cy*yhor+cx;
	  if(mvx[cpos] != Infinity && mvy[cpos] != Infinity ){
	    ppos = ppos_of[cpos];
        if(pstate[ppos]==UNI){
          cstate[cpos]=UNI;
          connect++;
		}
        else{
          if(pstate[ppos] == MULTI){
            if(dpd[cpos]==min_dpd[ppos]&&used[ppos]==0){
              cstate[cpos]=UNI;
              used[ppos]=1;
              connect++;
			}
            else{
              cstate[cpos]=MULTI;/* multi-connected*/
              unconnect++;
			}
		  }
          else{
            printf("error in find_uncovered_pixel!\n");
            exit(0);
		  }
		}
	  }	//mvx[cpos] != Infinity && mvy[cpos] != Infinity	
	  else{             
		cstate[cpos]=INTRA;/* unmatched*/
		unconnect++;
	  }

    }
  }

  if( (connect+unconnect) != (yver*yhor) ){
    printf("error in find_uncovered_pixel!\n");
    exit(0);
  }

  free(ppos_of);
  free(dpd);
  free(used);
  free(min_dpd);
  free(mvx); free(mvy);

//  printf("unconnect = %d connect = %d\n", unconnect, connect); 
  return(unconnect);
}

/*
 * rec_find_uncovered_block
 * distinguish uncovered block based on multi-connected pixels for all nodes in the tree
 */
void rec_find_uncovered_block(vector_ptr fmv, enum PixelMode *cstate, int x, int y, int xblock, int yblock, int hor, int ver)
{
 int i, j, xblk, yblk, pos;
 int count;

 xblk = (x+xblock<=hor)? xblock : hor-x;
 yblk = (y+yblock<=ver)? yblock : ver-y;

  if(xblk<=0 || yblk<=0){
    return;
  }

  count=0;
  for(i=y; i<y+yblk; i++){
    for(j=x; j<x+xblk; j++){
      pos=i*hor+j;
      if(cstate[pos]==MULTI)
        count++;
    }
  }

  /*if(x==192 && y == 128 && xblock == 32 && yblock == 32)
    printf("count %d xblk %d yblk %d \n", count, xblk, yblk);*/

  if(count>((xblk*yblk)/2)){
    fmv->mode = REVERSE;


  }
  //else
	//fmv->mode = DEFAULT;

   /* should use xblock and yblock instead of xblk and yblk */

if(fmv->child){
   rec_find_uncovered_block(fmv->child0, cstate, x,            y,           xblock/2, yblock/2, hor, ver); // not xblk yblk

   rec_find_uncovered_block(fmv->child1, cstate, x + xblock/2, y,           xblock/2, yblock/2, hor, ver);

   rec_find_uncovered_block(fmv->child2, cstate, x,            y + yblock/2, xblock/2, yblock/2, hor, ver);

   rec_find_uncovered_block(fmv->child3, cstate, x + xblock/2, y + yblock/2, xblock/2, yblock/2, hor, ver);

 }

}

/*
 *    find_uncovered_block()
 * Find the percentage of unconnected pixels. If it is smaller than a threshold
 * go on to find uncovered blocks from multi-connected pixels and marked "REVERSE" mode
 * in all nodes of the tree
 */

enum FLAG find_uncovered_block(vector_ptr fmv, YUVimage fr1, YUVimage fr0, int t_level, videoinfo info)
{
  int  hor, ver, xnum, ynum, xblk, yblk, pos, unconnect, thresh;
  int  x, y, X, Y;
  enum PixelMode *pstate, *cstate;

  hor  = info.ywidth; ver = info.yheight;
  xblk = info.xblk;   yblk = info.yblk;
  xnum = info.xnum;   ynum = info.ynum;


  thresh = (int)(hor*ver*Percent);   // 30% of 960x540 = 155520    30% of 352x240 = 25344
  printf("hor %d ver %d thresh %d (FME.c)\n", hor, ver, thresh);
  pstate = (enum PixelMode *)getarray(hor*ver, sizeof(enum PixelMode), "pstate");
  cstate = (enum PixelMode *)getarray(hor*ver, sizeof(enum PixelMode), "cstate");

  find_uncovered_pixel(fmv, fr1, fr0, pstate, cstate, info);
  /* fixed size block matching */
  for(y=0, Y=0 ; Y<ynum ; y+=yblk, Y++){
    for(x=0, X=0 ; X<xnum ; x+=xblk, X++){
      pos = Y*xnum+X;

      rec_find_uncovered_block(&fmv[pos], cstate, x, y, xblk, yblk, hor, ver);
    }
  }

#ifdef IBLOCK 
  find_unmatched_block(fmv, fr1, fr0, t_level, info); 
#endif

  unconnect = get_unconnect(fmv, info);

  if(unconnect > thresh){
	printf("No MCTF: unconnected pixels %d (FME.c)\n", unconnect);

    free(pstate);
    free(cstate);
	return NO;  // donot do MCTF 
  }




  free(pstate);
  free(cstate);
  return YES; //can do MCTF
}


