/*
 * Decompiled with CFR 0.152.
 */
package org.jaudiolibs.audioservers.javasound;

import java.nio.FloatBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.DataLine;
import javax.sound.sampled.Line;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.Mixer;
import javax.sound.sampled.SourceDataLine;
import javax.sound.sampled.TargetDataLine;
import org.jaudiolibs.audioservers.AudioClient;
import org.jaudiolibs.audioservers.AudioConfiguration;
import org.jaudiolibs.audioservers.AudioServer;
import org.jaudiolibs.audioservers.javasound.AudioFloatConverter;

public class JavasoundAudioServer
implements AudioServer {
    private static final Logger LOG = Logger.getLogger(JavasoundAudioServer.class.getName());
    private static final int NON_BLOCKING_MIN_BUFFER = 16384;
    private int nonBlockingOutputRatio = 16;
    private int lineBitSize = 16;
    private boolean signed = true;
    private boolean bigEndian = false;
    private AtomicReference<State> state;
    private AudioConfiguration context;
    private Mixer inputMixer;
    private Mixer outputMixer;
    private AudioClient client;
    private TimingMode mode;
    private TargetDataLine inputLine;
    private SourceDataLine outputLine;
    private byte[] inputByteBuffer;
    private float[] inputFloatBuffer;
    private byte[] outputByteBuffer;
    private float[] outputFloatBuffer;
    private List<FloatBuffer> inputBuffers;
    private List<FloatBuffer> outputBuffers;
    private AudioFloatConverter converter;

    private JavasoundAudioServer(Mixer inputMixer, Mixer outputMixer, TimingMode mode, AudioConfiguration context, AudioClient client) {
        this.inputMixer = inputMixer;
        this.outputMixer = outputMixer;
        this.context = context;
        this.mode = mode;
        this.client = client;
        this.state = new AtomicReference<State>(State.New);
    }

    public void run() throws Exception {
        if (!this.state.compareAndSet(State.New, State.Initialising)) {
            throw new IllegalStateException();
        }
        try {
            this.initialise();
            this.client.configure(this.context);
        }
        catch (Exception ex) {
            this.state.set(State.Terminated);
            this.closeAll();
            this.client.shutdown();
            throw ex;
        }
        if (this.state.compareAndSet(State.Initialising, State.Active)) {
            this.runImpl();
        }
        this.closeAll();
        this.client.shutdown();
        this.state.set(State.Terminated);
    }

    public AudioConfiguration getAudioContext() {
        return this.context;
    }

    public boolean isActive() {
        State st = this.state.get();
        return st == State.Active || st == State.Closing;
    }

    public void shutdown() {
        State st;
        while ((st = this.state.get()) != State.Terminated && st != State.Closing && !this.state.compareAndSet(st, State.Closing)) {
        }
    }

    private void initialise() throws Exception {
        int byteBufferSize;
        float srate = this.context.getSampleRate();
        int buffersize = this.context.getMaxBufferSize();
        int inputChannels = this.context.getInputChannelCount();
        int outputChannels = this.context.getOutputChannelCount();
        if (inputChannels > 0) {
            AudioFormat inputFormat = new AudioFormat(srate, this.lineBitSize, inputChannels, this.signed, this.bigEndian);
            this.inputLine = (TargetDataLine)this.inputMixer.getLine(new DataLine.Info(TargetDataLine.class, inputFormat));
            this.inputFloatBuffer = new float[buffersize * inputChannels];
            byteBufferSize = buffersize * inputFormat.getFrameSize();
            this.inputByteBuffer = new byte[byteBufferSize];
            this.inputLine.open(inputFormat, byteBufferSize *= this.nonBlockingOutputRatio);
        }
        AudioFormat outputFormat = new AudioFormat(srate, this.lineBitSize, outputChannels, this.signed, this.bigEndian);
        this.outputLine = (SourceDataLine)this.outputMixer.getLine(new DataLine.Info(SourceDataLine.class, outputFormat));
        this.outputFloatBuffer = new float[buffersize * outputChannels];
        byteBufferSize = buffersize * outputFormat.getFrameSize();
        this.outputByteBuffer = new byte[byteBufferSize];
        if (this.mode != TimingMode.Blocking) {
            byteBufferSize *= this.nonBlockingOutputRatio;
            byteBufferSize = Math.min(byteBufferSize, 16384);
        }
        this.outputLine.open(outputFormat, byteBufferSize);
        this.converter = AudioFloatConverter.getConverter(outputFormat);
        ArrayList<FloatBuffer> ins = new ArrayList<FloatBuffer>(inputChannels);
        for (int i = 0; i < inputChannels; ++i) {
            ins.add(FloatBuffer.allocate(buffersize));
        }
        this.inputBuffers = Collections.unmodifiableList(ins);
        ArrayList<FloatBuffer> outs = new ArrayList<FloatBuffer>(outputChannels);
        for (int i = 0; i < outputChannels; ++i) {
            outs.add(FloatBuffer.allocate(buffersize));
        }
        this.outputBuffers = Collections.unmodifiableList(outs);
    }

    private void runImpl() {
        if (this.inputLine != null) {
            this.inputLine.start();
        }
        this.outputLine.start();
        long startTime = System.nanoTime();
        double bufferTime = (double)this.context.getMaxBufferSize() / (double)this.context.getSampleRate() * 1.0E9;
        long bufferCount = 0L;
        int bufferSize = this.context.getMaxBufferSize();
        try {
            while (this.state.get() == State.Active) {
                this.readInput();
                if (this.client.process(System.nanoTime(), this.inputBuffers, this.outputBuffers, bufferSize)) {
                    this.writeOutput();
                    switch (this.mode) {
                        case Estimated: {
                            while ((double)(System.nanoTime() - startTime) / bufferTime < (double)bufferCount) {
                                try {
                                    Thread.sleep(1L);
                                }
                                catch (Exception ex) {}
                            }
                            break;
                        }
                        case FramePosition: {
                            while (this.outputLine.getLongFramePosition() < bufferCount * (long)bufferSize) {
                                try {
                                    Thread.sleep(1L);
                                }
                                catch (Exception ex) {}
                            }
                            break;
                        }
                    }
                    ++bufferCount;
                    continue;
                }
                this.shutdown();
            }
        }
        catch (Exception ex) {
            Logger.getLogger(JavasoundAudioServer.class.getName()).log(Level.SEVERE, "", ex);
        }
    }

    private void readInput() {
        TargetDataLine tdl = this.inputLine;
        if (tdl != null) {
            int bsize = this.inputByteBuffer.length;
            if (tdl.available() < bsize) {
                int fsize = this.inputFloatBuffer.length;
                for (int i = 0; i < fsize; ++i) {
                    this.inputFloatBuffer[i] = 0.0f;
                }
            } else {
                tdl.read(this.inputByteBuffer, 0, bsize);
                this.converter.toFloatArray(this.inputByteBuffer, this.inputFloatBuffer);
            }
            int channels = this.inputBuffers.size();
            for (int channel = 0; channel < channels; ++channel) {
                FloatBuffer inBuf = this.inputBuffers.get(channel);
                float[] input = inBuf.array();
                int x = channel;
                for (int i = 0; i < input.length; ++i) {
                    input[i] = this.inputFloatBuffer[x];
                    x += channels;
                }
                inBuf.rewind();
            }
        }
    }

    private void writeOutput() {
        int channels = this.outputBuffers.size();
        for (int channel = 0; channel < channels; ++channel) {
            FloatBuffer outBuf = this.outputBuffers.get(channel);
            float[] output = outBuf.array();
            int x = channel;
            for (int i = 0; i < output.length; ++i) {
                this.outputFloatBuffer[x] = output[i];
                x += channels;
            }
            outBuf.rewind();
        }
        this.converter.toByteArray(this.outputFloatBuffer, this.outputByteBuffer);
        this.outputLine.write(this.outputByteBuffer, 0, this.outputByteBuffer.length);
    }

    private void closeAll() {
        TargetDataLine tdl;
        SourceDataLine sdl = this.outputLine;
        if (sdl != null) {
            sdl.close();
        }
        if ((tdl = this.inputLine) != null) {
            tdl.close();
        }
    }

    public static JavasoundAudioServer create(Mixer inputMixer, Mixer outputMixer, AudioConfiguration context, TimingMode mode, AudioClient client) {
        if (outputMixer == null || mode == null || context == null || client == null) {
            throw new NullPointerException();
        }
        if (inputMixer == null && context.getInputChannelCount() > 0) {
            throw new NullPointerException();
        }
        if (context.getOutputChannelCount() == 0) {
            throw new IllegalArgumentException();
        }
        if (!context.isFixedBufferSize()) {
            context = new AudioConfiguration(context.getSampleRate(), context.getInputChannelCount(), context.getOutputChannelCount(), context.getMaxBufferSize(), true);
        }
        return new JavasoundAudioServer(inputMixer, outputMixer, mode, context, client);
    }

    public static JavasoundAudioServer create(String device, AudioConfiguration context, TimingMode mode, AudioClient client) throws Exception {
        if (mode == null || client == null) {
            throw new NullPointerException();
        }
        if (device == null) {
            device = "";
        }
        if (context.getOutputChannelCount() == 0) {
            throw new IllegalArgumentException();
        }
        Mixer in = null;
        if (context.getInputChannelCount() > 0) {
            in = JavasoundAudioServer.findInputMixer(device);
        }
        Mixer out = JavasoundAudioServer.findOutputMixer(device);
        if (!context.isFixedBufferSize()) {
            context = new AudioConfiguration(context.getSampleRate(), context.getInputChannelCount(), context.getOutputChannelCount(), context.getMaxBufferSize(), true);
        }
        return new JavasoundAudioServer(in, out, mode, context, client);
    }

    private static Mixer findInputMixer(String device) throws Exception {
        Mixer.Info[] infos = AudioSystem.getMixerInfo();
        for (int i = 0; i < infos.length; ++i) {
            Mixer mixer = AudioSystem.getMixer(infos[i]);
            Line.Info[] lines = mixer.getTargetLineInfo();
            if (lines.length <= 0 || infos[i].getName().indexOf(device) < 0) continue;
            LOG.finest("Found input mixer :\n" + infos[i]);
            return mixer;
        }
        throw new Exception();
    }

    private static Mixer findOutputMixer(String device) throws LineUnavailableException {
        Mixer.Info[] infos = AudioSystem.getMixerInfo();
        for (int i = 0; i < infos.length; ++i) {
            Mixer mixer = AudioSystem.getMixer(infos[i]);
            Line.Info[] lines = mixer.getSourceLineInfo();
            if (lines.length <= 0 || infos[i].getName().indexOf(device) < 0) continue;
            LOG.finest("Found output mixer :\n" + infos[i]);
            return mixer;
        }
        throw new LineUnavailableException();
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    private static enum State {
        New,
        Initialising,
        Active,
        Closing,
        Terminated;

    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    public static enum TimingMode {
        Blocking,
        FramePosition,
        Estimated;

    }
}

