# bitmap editor. Produces bitmap files in the weird format required
# by the Tk image create command.

option add *Button.font {helvetica 10 bold}
option add *Label.font {helvetica 9 bold}
option add *Entry.font {courier 10 bold}

set directory {}

proc main {} {

# main sets up the default parameters, defines the gui & bindings
   global par

   set par(top) 8
   set par(left) 8
   set par(size) 12
   set par(default_size) 12
   set par(min_size) 7
   set par(nx) 16
   set par(ny) 16
   set par(lastx) 0
   set par(lasty) 0
   set par(box_bg) grey75
   set par(sel_bg) pink
   set par(sel_fg) #700000
   set par(selected) 0
   set par(script) [info script]
   

   source [file join [file dirname $par(script)] bme_bmp.tcl]
# bme_bmp.tcl defines the bitmaps used on the tool buttons.
# above should find it if it's in the same dir as this
   
   canvas .c -width 384 -height 256 -bg white
   frame .f
   frame .f.fw
   label .f.fw.lw -text width -width 7 -anchor center
   entry .f.fw.ew  -width 3
   
   frame .f.fh
   label .f.fh.lh -text height -width 7 -anchor center
   entry .f.fh.eh -width 3
   button .f.bc -text Reset -command reset -pady 0 -highlightthickness 0
   button .f.o  -text Output -command {output_bitmap}  -pady 0 -highlightthickness 0
   button .f.i  -text Input -command {input_bitmap}  -pady 0 -highlightthickness 0
   button .f.h  -text Help -command {help}  -pady 0 -highlightthickness 0
   
   frame .f.tb
   button .f.tb.bu -image $bm_uarrow -command {vshift up}
   button .f.tb.bd -image $bm_darrow -command {vshift down}
   button .f.tb.bl -image $bm_larrow -command {hshift left}
   button .f.tb.br -image $bm_rarrow -command {hshift right}
   pack .f.tb.bu .f.tb.bd .f.tb.bl .f.tb.br -side left -padx 2
   
   frame .f.tb2
   button .f.tb2.bd -image $bm_osquare -command {fill 0}
   button .f.tb2.bf -image $bm_fsquare -command {fill 1}
   button .f.tb2.bc -image $bm_cboard -command {cboard}
   button .f.tb2.bi -image $bm_hsquare -command {invert}
   pack .f.tb2.bd .f.tb2.bf .f.tb2.bc .f.tb2.bi -side left -padx 2
   
   frame .f.tb3
   button .f.tb3.mu -image $bm_vmup -command {vmirror up}
   button .f.tb3.md -image $bm_vmdn -command {vmirror down}
   button .f.tb3.mr -image $bm_hmr -command {hmirror right}
   button .f.tb3.ml -image $bm_hml -command {hmirror left}
   pack .f.tb3.mu .f.tb3.md .f.tb3.ml .f.tb3.mr  -side left -padx 2
   
   pack .f.fw.lw .f.fw.ew -side left -padx 2 -pady 3
   pack .f.fw  -side top -padx 2 -pady 3 -fill x
   
   pack .f.fh.lh .f.fh.eh -side left -padx 2 -pady 3
   pack .f.fh  -side top -padx 2 -pady 3 -fill x
   
   pack .f.bc .f.o .f.i .f.h -side top -padx 2 -pady 4 -fill x
   pack .f.tb -side top -pady 2 -fill x
   pack .f.tb2 -side top -pady 2 -fill x
   pack .f.tb3 -side top -pady 2 -fill x
   pack .f -padx 6 -pady 2 -side right -anchor n
   pack .c  -side left -padx 2 -pady 2 -fill both -expand 1
   
   bind .c <Button-1> {unselect; do_dot %x %y}
   bind .c <Shift-Button-1> {unselect; do_rect %x %y}
   bind .c <Control-Button-1> {unselect; do_line %x %y}
   bind .c <ButtonPress-3> {unselect; begin_select %x %y}
   bind .c <B3-Motion> {continue_select %x %y}
   bind .c <ButtonRelease-3> {end_select %x %y}
   bind . <Return> {unselect}
   bind . <Escape> {unselect}
   bind . <F1> {help}
   
   after 100 reset
# windoze 3 needs the delay to not go berserk.
}

##################################################################
# The following procs draw things in the canvas                  # 
##################################################################

proc draw_grid {} {
   global par

   .c delete all
   draw_grid_lines
}


proc draw_grid_lines {} {
   global par

   set par(bot) [expr {$par(top) + $par(size) * $par(ny)}]
   set par(right) [expr {$par(left) + $par(size) * $par(nx)}]

   for {set i 0} {$i <= $par(nx)} {incr i} {
      set x [expr {$par(left) + $i * $par(size)}]
      .c create line $x $par(top) $x $par(bot)
   }

   for {set i 0} {$i <= $par(ny)} {incr i} {
      set y [expr {$par(top) + $i * $par(size)}]
      .c create line $par(left) $y $par(right) $y
   }

}


proc do_rect {xx yy} {
# toggle  a rectangle
   
   global par bits
   
   set c [get_grid_coordinates $xx $yy]
   if {$c == {}} return
   set gx [lindex $c 0]
   set gy [lindex $c 1]

   set startx [set lastx $par(lastx)]
   set starty [set lasty $par(lasty)]

   set par(lastx) $gx
   set par(lasty) $gy

   if {$lastx > $gx} {
      set t $lastx
      set lastx $gx
      set gx $t
   }
   if {$lasty > $gy} {
      set t $lasty
      set lasty $gy
      set gy $t
   }

   for {set x $lastx} {$x <= $gx} {incr x} {
      for {set y $lasty} {$y <= $gy} {incr y} {
         if {$x == $startx && $y == $starty} continue
         toggle_dot $x $y
      }
   }
}



proc do_line {xx yy} {
# toggle  a line
   
   global par bits
   
   set c [get_grid_coordinates $xx $yy]
   if {$c == {}} return
   set gx [lindex $c 0]
   set gy [lindex $c 1]

   set startx [set lastx $par(lastx)]
   set starty [set lasty $par(lasty)]
   set par(lastx) $gx
   set par(lasty) $gy

   if {$lastx > $gx} {
      set t $lastx
      set lastx $gx
      set gx $t
      set t $lasty
      set lasty $gy
      set gy $t
   }

   draw_line $lastx $lasty $gx $gy
   toggle_dot $startx $starty
}


proc draw_line {x1 y1 x2 y2} {
# bresenham algorithm. It's assumed that x2 >= x1

   set dx [expr {abs($x2 - $x1)}]
   set dy [expr {abs($y2 - $y1)}]
   set my1 [expr {-$y1}]
   set my2 [expr {-$y2}]
   set st [expr {$dy > $dx}]
   
   if {$st} {
      set d $x1;  set x1 $my1;   set my1 $d
      set d $x2;  set x2 $my2;   set my2 $d
      set d $dx;  set dx $dy;    set dy $d
   
   }
   
   set i1 [expr {2 * $dy}]
   set d [expr {$i1 - $dx}]
   set i2 [expr {$d - $dx}]
   set yi 1
   if {$x1 > $x2} {
      set x $x2
      set y $my2
      set xe $x1
      if {$my2 > $my1} {set yi -1}
   } else {
      set x $x1
      set y $my1
      set xe $x2
      if {$my1 > $my2} {set yi -1}
   }
   while {$x <= $xe} {
      if {$st} {
         set xp $y
         set yp [expr {-$x}]
      } else {
         set xp $x
         set yp [expr {-$y}]
      }
      toggle_dot $xp $yp
      incr x
      if {$d > 0} {
         incr y $yi
         incr d $i2
      } else {
         incr d $i1
      }
   }
}


proc do_dot {x y} {
# toggle pel at canvas position x, y

   global par bits
   
   set c [get_grid_coordinates $x $y]
   if {$c == {}} return
   set gx [lindex $c 0]
   set gy [lindex $c 1]

   toggle_dot $gx $gy

   set par(lastx) $gx
   set par(lasty) $gy
}

proc toggle_dot {gx gy} {
   global par bits

   if {$bits($gx,$gy) == 0} {
      set bits($gx,$gy) 1
      set colour black
   } else {
      set bits($gx,$gy) 0
      set colour white
   }
      
   draw_dot $gx $gy $colour
}


proc draw_dot {gx gy colour} {
#draw dot specified in grid coordinates
   global par bits


   set x1 [expr {$par(left) + $gx * $par(size)}]
   set x2 [expr {$x1 + $par(size)}]
   set y1 [expr {$par(top) + $gy * $par(size)}]
   set y2 [expr {$y1 +$par(size)}]
   .c create rectangle $x1 $y1 $x2 $y2 -fill $colour -outline black
   
   set co $colour
   if {$colour == "white"} {set co $par(box_bg)}
   if {$colour == $par(sel_bg)} {set co $par(box_bg)}
   if {$colour == $par(sel_fg)} {set co "black"}
#   draw_box_dot $gx $gy $co
   
   set q [expr {$par(box_left) + $gx}]
   set r [expr {$par(box_top) + $gy}]
   .c create line $q $r [expr {$q + 1}] $r -fill $co
}


proc draw_box {} {
# real-size display
   
   global par
   
   set l [expr {$par(right) + 32}]
   set t [expr {$par(top) + 32}]
   set r [expr {$par(nx) + $l + 8}]
   set b [expr {$par(ny) + $t + 8}]
   set par(box_left) [expr {$l + 4}]
   set par(box_top) [expr {$t + 4}]
   
   .c create rectangle $l $t $r $b -fill $par(box_bg) -outline {}
}


proc draw_box_dot {x y colour} {
   
   global par
   
   set q [expr {$par(box_left) + $x}]
   set r [expr {$par(box_top) + $y}]
#   .c create rectangle $q $r $q $r -fill $colour -outline {}
   .c create line $q $r [expr {$q + 1}] $r -fill $colour
}


proc redraw {} {
   global par bits
#set t1 [clock clicks -milliseconds]
   draw_grid;     # clears the display
   draw_box
#set t2 [clock clicks -milliseconds]

   paint_selection
   for {set r 0} {$r < $par(ny)} {incr r} {
      for {set c 0} {$c < $par(nx)} {incr c} {
         if {$bits($c,$r)} {
            draw_dot $c $r [expr {[is_selected $c $r] ?\
               $par(sel_fg) : "black"}]
         }
      }
   }
#set t3 [clock clicks -milliseconds]
#puts "[expr $t2 - $t1] [expr $t3-$t2]"
}



proc paint_selection {} {
   global par sel
   
   if {! $par(selected)} {return}
   
   set l [expr {$par(left) + $par(size) * $sel(gx0)}]
   set t [expr {$par(top) + $par(size) * $sel(gy0)}]
   set r [expr {$par(left) + $par(size) * (1 + $sel(gx1))}]
   set b [expr {$par(top) + $par(size) * (1 + $sel(gy1))}]
   .c create rectangle $l $t $r $b -fill $par(sel_bg) -outline {}
   draw_grid_lines
}


################################################################
# End of the canvas drawing routines                           #
################################################################


#################################
# General utility procs         #
#################################


proc help {} {
# display the help file
   global par
   
   if {[winfo exists .bmehelp]} {
      wm deiconify .bmehelp
      raise .bmehelp
   } else {
      source [file join [file dirname $par(script)] bme_help.tcl]
   }
}

proc reset {} {
   global par
   
   wm title . "   BME"
   
   set nx [.f.fw.ew get]
   set ny [.f.fh.eh get]

   if {[regexp "^\ *\[0-9]+\ *\$" $nx]} {set par(nx) $nx}
   if {[regexp "^\ *\[0-9]+\ *\$" $ny]} {set par(ny) $ny}
# avoid string is integer, for 8.0 compatibility
   
   set par(xmax) [expr {$par(nx) - 1}]
   set par(ymax) [expr {$par(ny) - 1}]
   .c delete all
   
   set ww [winfo screenwidth .]
   set wh [winfo screenheight .]
   set wmax [expr {0.80 * $ww}]
   set hmax [expr {0.75 * $wh}]
   
   set size $par(default_size)
   set w [expr {$par(nx) * ( 1 + $size) + 200}]
   set h [expr {$par(ny) * $size + 160}]
   if {$w > $wmax} {
      set size [expr {int((($wmax - 200) / $par(nx)) - 1) } ]
      set w [expr {$par(nx) * ( 1 + $size) + 200}]
      set h [expr {$par(ny) * $size + 20}]
   }
   if {$h > $hmax} {
      set size [expr {int((($hmax - 20) / $par(ny))) } ]
      set w [expr {$par(nx) * ( 1 + $size) + 200}]
      set h [expr {$par(ny) * $size + 160}]
   }
   
   if {$size < $par(min_size) } {
      tk_messageBox -type ok -message "Too big!"
      return
   } 
   
   set par(size) $size
   
   set xoff [expr {($ww - $w) / 2}]
   if {$h < 312} {set h 312}
   wm geometry . "${w}x${h}-32-32"
#   wm geometry . "-32-32"
   
   clear_bits
   draw_grid
   draw_box
}


proc clear_bits {} {
   global par bits

   for {set x 0} {$x < $par(nx)} {incr x} {
      for {set y 0} {$y < $par(ny)} {incr y} {
         set bits($x,$y) 0
      }
   }
   set par(lastx) 0
   set par(lasty) 0
}


proc get_grid_coordinates {x y} {
# canvas coordinates to grid coordinates
   
   global par 
   
   if {$x < $par(left)} {return {}}
   if {$y < $par(top)} {return {}}
   if {$x >= $par(right)} {return {}}
   if {$y >= $par(bot)} {return {}}

   set gx [expr {($x - $par(left)) / $par(size)}]
   set gy [expr {($y - $par(top)) / $par(size)}]
   return "$gx $gy"
}

proc cutout {s i j} {
# return the string produced by cutting out
# characters 1 to j from string s
# This avoids string range, for 8.0 compatibility

   if {$i < 0 || $j < $i} {return $s}
   set z [expr {$i - 1}]
   set w [expr {$j + 1}]
   set a [string range $s 0 $z]
   append a [string range $s $w end]
   return $a
}

#################################
# End of utility procs          #
#################################


#################################
#  I/O procs                    #
#################################

proc input_bitmap {} {
# read in a xbm file
   
   global par bits directory

   set f [tk_getOpenFile -initialdir $directory\
      -filetypes {{{X bitmaps} *.xbm} {{all files} *.*}}]
   if {$f == {}} {return}
   set directory [file dirname $f]
   set ch [open $f r]
   set data [read $ch]
   close $ch

# strip any comments
   while {1} {
      set i1 [string first /* $data]
      if {$i1 == -1} break
      set i2 [string first */ $data]
      if {$i2 == -1 || $i2 < $i2} {
         tk_messageBox -type ok -message "Invalid comment"
         return
      }
      incr i2
      set data [cutout $data $i1 $i2]
# avoid string replace for 8.0 compatibility
   }

# get width and height
   if {0 == [regexp "width\ +(\[0-9]+)" $data x w]} {
         tk_messageBox -type ok -message "Can't find width"
         return
   }
   if {0 == [regexp "height\ +(\[0-9]+)" $data y h]} {
         tk_messageBox -type ok -message "Can't find height"
         return
   }

# update display width/height
   .f.fw.ew delete 0 end
   .f.fw.ew insert end $w
   .f.fh.eh delete 0 end
   .f.fh.eh insert end $h

   reset
   wm title . "     $f"


# toss everything before the bit data
   set i1 [string first \{ $data]
   set data [cutout $data 0 $i1]
   set data [split $data ",\}\ \n\t"]

# read the bit data
   set bits_avail 0
   set ix 0
   for {set i 0} {$i < $h} {incr i} {
      for {set j 0} {$j < $w} {incr j} {
         if {!$bits_avail} {
            while {1} {
               set b [lindex $data $ix]
               incr ix
               if {$b != {}} break
            }
            set bits_avail 8
         }
         set bits($j,$i) [expr {$b & 1}]
         set b [expr {$b / 2}]
         incr bits_avail -1
      }
      set bits_avail 0
   }
   redraw
}

proc output_bitmap {} {
# bitmap out to file in weird xbm format used by tk
   
   global par bits directory
   
   set f [tk_getSaveFile -initialdir $directory]
   if {$f == {}} {return}
   
   set directory [file dirname $f]
   set ch [open $f w]
   
   puts $ch [format "#define x_width %d" $par(nx)]
   puts $ch [format "#define x_height %d" $par(ny)]
   puts $ch "static char x_bits\[] = {"
   
   set n 0
   for {set i 0} {$i < $par(ny)} {incr i} {
      set j 0
      set b 0
      set p 1
      while {$j < $par(nx)} {
         set b [expr {$b + $p * $bits($j,$i)}]
         set p [expr {$p * 2}]
         incr j
         if {0 == [expr {$j % 8}] } {
            puts -nonewline $ch [format "0x%02x, " $b]
# wtf doesn't tcl's format put 0x on 0 if simply %#x ??
            set b 0
            set p 1
            incr n
            if {0 == [expr {$n % 16}]} {puts $ch {}}
         }
      }
      if {0 != [expr {$j % 8}] } {
         puts -nonewline $ch [format "%#x, " $b]
         incr n
         if {0 == [expr {$n % 16}]} {puts $ch {}}
      }
   }
   puts $ch "};"
   close $ch
   
   wm title . $f
}

##############################
# End of I/O procs           #
##############################



####################################
# Selection handling procs         #
####################################

proc unselect {} {
   global par
   
   if {$par(selected)} {
      set par(selected) 0
      redraw
   }
}


proc is_selected {x y} {
   global par sel
# is x, y in the selected region ?
   
   if {!$par(selected)} {return 0}
   
   if {$x < $sel(gx0)} {return 0}
   if {$x > $sel(gx1)} {return 0}
   if {$y < $sel(gy0)} {return 0}
   if {$y > $sel(gy1)} {return 0}
   return 1
}


proc begin_select {x y} {
   global par sel

   set sel(started) 0
   set par(selected) 0
   set c [get_grid_coordinates $x $y]
   
   if {$c == {}} return
   
   set sel(gx0) [lindex $c 0]
   set sel(gy0) [lindex $c 1]
   set x0 [expr {$par(left) + $sel(gx0) * $par(size)}]
   set y0 [expr {$par(top) + $sel(gy0) * $par(size)}]
   
   set sel(rect) [.c create rectangle $x0 $y0 $x0 $y0 -outline red]
   
   set sel(started) 1
}


proc continue_select {x y} {
   global par sel

   if {! $sel(started)} return
   set c [get_grid_coordinates $x $y]
   
   if {$c == {}} {set c "$par(xmax) $par(ymax)"}
   
   set sel(gx1) [lindex $c 0]
   set sel(gy1) [lindex $c 1]
   if {$sel(gx1) > $sel(gx0)} {
      set x0 [expr {$par(left) + $sel(gx0) * $par(size)}]
      set x1 [expr {$par(left) + ($sel(gx1) + 1) * $par(size)}]
   } else {
      set x0 [expr {$par(left) + ($sel(gx0) + 1) * $par(size)}]
      set x1 [expr {$par(left) + $sel(gx1) * $par(size)}]
   }
   if {$sel(gy1) > $sel(gy0)} {
      set y0 [expr {$par(top) + $sel(gy0) * $par(size)}]
      set y1 [expr {$par(top) + ($sel(gy1) + 1) * $par(size)}]
   } else {
      set y0 [expr {$par(top) + ($sel(gy0) + 1) * $par(size)}]
      set y1 [expr {$par(top) + $sel(gy1) * $par(size)}]
   }
   
   .c delete $sel(rect)
   set sel(rect) [.c create rectangle $x0 $y0 $x1 $y1\
          -outline #c00040 -width 2]
}


proc end_select {x y} {
   global par sel

   if {! $sel(started)} return
   set c [get_grid_coordinates $x $y]
   
   if {$c == {}} {set c "$par(xmax) $par(ymax)"}
   
   set sel(gx1) [lindex $c 0]
   set sel(gy1) [lindex $c 1]
   if {$sel(gx1) < $sel(gx0)} {
      set t $sel(gx0)
      set sel(gx0) $sel(gx1)
      set sel(gx1) $t
   }
   if {$sel(gy1) < $sel(gy0)} {
      set t $sel(gy0)
      set sel(gy0) $sel(gy1)
      set sel(gy1) $t
   }
   .c delete $sel(rect)
   
   set sel(started) 0
   set par(selected) 1
#puts "$sel(gx0),$sel(gy0)  $sel(gx1),$sel(gy1)"
   redraw
}


proc get_region {} {
   global par sel
   
# if there is a selection, return it as list left , top, right, bot.
# else return whole area
   
   if {$par(selected)} {
      return [list $sel(gx0) $sel(gy0) $sel(gx1) $sel(gy1)]
   } else {
      return [list 0 0 $par(xmax) $par(ymax)]
   }
}


###################################
# End of selection handling procs #
###################################



##############################
# Bitmap manipulation procs  #
##############################

proc vshift {dir} {
# move image or selection up or down 1 pixel
   global par bits sel

   set s $par(selected)

   if {$s} {
      set left $sel(gx0)
      set right $sel(gx1)
   } else {
      set left 0
      set right $par(xmax)
   }
   
   if {$dir == "up"} {
      set start [expr {$s ? $sel(gy0) : 0}]
      set first [expr {$start + 1}]
      set last [expr {$s ? $sel(gy1) + 1 : $par(ny)}]
      set lm1 [expr {$last - 1}]
      set inc 1
   } else {
      set start [expr {$s ? $sel(gy1) : $par(ymax)}]
      set first [expr {$start - 1}]
      set last [expr {$s ? $sel(gy0) - 1 : -1}]
      set lm1 [expr {$last + 1}]
      set inc -1
   }
   
   for {set c $left} {$c <= $right} {incr c} {
      set j $start
      for {set r $first} {$r != $last} {incr r $inc} {
         set bits($c,$j) $bits($c,$r)
         incr j $inc
      }
      set bits($c,$lm1) 0
   }

   redraw
}
   
   

proc hshift {dir} {
# move image or selection left or right 1 pixel
   
   global par bits sel

   set s $par(selected)
   
   if {$s} {
      set top $sel(gy0)
      set bot $sel(gy1)
   } else {
      set top 0
      set bot $par(ymax)
   }
   
   if {$dir == "left"} {
      set start [expr {$s ? $sel(gx0) : 0}]
      set first [expr {$start + 1}]
      set last [expr {$s ? $sel(gx1) + 1 : $par(nx)}]
      set lm1 [expr {$last - 1}]
      set inc 1
   } else {
      set start [expr {$s ? $sel(gx1) : $par(xmax)}]
      set first [expr {$start - 1}]
      set last [expr {$s ? $sel(gx0) - 1 : -1}]
      set lm1 [expr {$last + 1}]
      set inc -1
   }
   
   for {set r $top} {$r <= $bot} {incr r} {
      set j $start
      for {set c $first} {$c != $last} {incr c $inc} {
         set bits($j,$r) $bits($c,$r)
         incr j $inc
      }
      set bits($lm1,$r) 0
   }

   redraw
}



proc fill {data} {
   global par bits sel
   
   set s [get_region]

   for {set r [lindex $s 1]} {$r <= [lindex $s 3]} {incr r} {
      for {set c [lindex $s 0]} {$c <= [lindex $s 2]} {incr c} {
         set bits($c,$r) $data
      }
   }
   if {! $par(selected)} redraw
   unselect
# unselect does a redraw. It's a no-op if no selection
}

proc cboard {} {
   global par bits sel
# checkerboard fill
   set s [get_region]

   for {set r [lindex $s 1]} {$r <= [lindex $s 3]} {incr r} {
      for {set c [lindex $s 0]} {$c <= [lindex $s 2]} {incr c} {
         if {0 == [expr {($r + $c) % 2}]} {
            set bits($c,$r) 1
         } else {
            set bits($c,$r) 0
         }
      }
   }
   if {! $par(selected)} redraw
   unselect
}



proc invert {} {
   global par bits sel
# Invert all bits in selection or whole
   
   set s [get_region]
   
   for {set r [lindex $s 1]} {$r <= [lindex $s 3]} {incr r} {
      for {set c [lindex $s 0]} {$c <= [lindex $s 2]} {incr c} {
         set bits($c,$r) [expr { ! $bits($c,$r)}]
      }
   }
   if {! $par(selected)} redraw
   unselect
}


proc vmirror {dir} {
   global par bits sel
   
   set s [get_region]

   set b [lindex $s 3]
   set t [lindex $s 1]
   set nr [expr { ($b - $t + 1) / 2 }]
   
   for {set i 0} {$i < $nr} {incr i} {
      for {set c [lindex $s 0]} {$c <= [lindex $s 2]} {incr c} {
         if {$dir == "up"} {
            set bits($c,$t) $bits($c,$b)
         } else {
            set bits($c,$b) $bits($c,$t)
         }
      }
      incr t
      incr b -1
   }
         
   if {! $par(selected)} redraw
   unselect
}


proc hmirror {dir} {
   global par bits sel
   
   set s [get_region]

   set l [lindex $s 0]
   set r [lindex $s 2]
   set nc [expr { ($r - $l + 1) / 2 }]
   
   for {set c 0} {$c < $nc} {incr c} {
      for {set i [lindex $s 1]} {$i <= [lindex $s 3]} {incr i} {
         if {$dir == "right"} {
            set bits($r,$i) $bits($l,$i)
         } else {
            set bits($l,$i) $bits($r,$i)
         }
      }
      incr l
      incr r -1
   }
         
   if {! $par(selected)} redraw
   unselect
}

#####################################
# End of bitmap manipulation procs  #
#####################################


main

