/*---------------------------------------------------------------------------*/
// Baseline Wavelet Transform Coder Construction Kit
//
// Geoff Davis
// gdavis@cs.dartmouth.edu
// http://www.cs.dartmouth.edu/~gdavis
//
// Copyright 1996 Geoff Davis 9/11/96
//
// Permission is granted to use this software for research purposes as
// long as this notice stays attached to this software.
//
/*---------------------------------------------------------------------------*/
// coeffset.hh
//
// Each CoeffSet object corresponds to a subset of transform
// coefficients.  the object contains rate/distortion curves for its
// associated set of coefficients which it obtains from the quantizer
// passed in during construction.  Also included are functions for
// encoding and decoding the set given a quantizer precision level and
// functions for reading and writing the quantizer parameters.
//
// Functions:
// ----------
// getRateDist      Get rate/distortion curves for the set.
// encode           Encode coefficients with the specified precision
//                  and write them using the given encoder.
// decode           Read quantized coefficients from the given decoder
//                  and dequantize them.
// writeHeader      Write quantizer parameters using the given encoder
// readHeader       Read quantizer parameters using the given decoder
//
/*---------------------------------------------------------------------------*/
#ifndef _COEFFSET_
#define _COEFFSET_
#include "coder.h"
#include "quantizer.h"
#include "packet.h"
/*---------------------------------------------------------------------------*/

class CoeffSet {
public:
  CoeffSet (Real *data, int ndata, Quantizer *quant);
  ~CoeffSet ();

  void getRateDist (int nQuant, Real minStepSize, int subband);

  void writeHeader (Encoder *encoder, int precision, int subband)
        { quant->writeHeader (encoder, precision, subband); };
  void readHeader  (Decoder *decoder, int &precision, int subband)
        { quant->readHeader  (decoder, precision, subband); };

  void encode (Encoder **encoder, int precision, int subband, PacketEnc **pkt,
               int modulus)
        { 
          //printf("coeff->encode...\n");
          quant->quantize (encoder, precision, subband, pkt, modulus); 
        };
  void decode (Decoder **decoder, int precision, int subband, int *missing, 
               PacketDec **pkt, int modulus)
        { 
          if (modulus==0)
            quant->setDataDecode (data, nData, subband);
            
	  quant->dequantize (decoder,precision,subband,missing,pkt,modulus); 
	};
  
  // ----------------------- Added ------------------------------------
  int packet(int subband, int position)
        { return quant->packet(subband, position); };
  int getCnt(int layer, int context, int cnt)
        { return quant->getCnt(layer, context, cnt); };
  int getStp(int layer, int context, int cnt)
        { return quant->getStp(layer, context, cnt); };
  int getPts(int layer, int context, int cnt)
        { return quant->getPts(layer, context, cnt); };
  void setCnt(int context, int pos, int value)
        { quant->setCnt(context, pos, value); };
  void setStp(int context, int pos, int value)
        { quant->setStp(context, pos, value); };
  void setPts(int context, int pos, int value)
        { quant->setPts(context, pos, value); };            
  // ----------------------End of Added -------------------------------
  
  int  nData;
  Real *data;
  Quantizer *quant;

  // The data in each CoeffSet is quantized using quantizers with
  // different levels of precision.  From this we generate a
  // rate/distortion curve which is used to optimize bit allocation.
  int  nQuant;    // # of quantizers used
  Real *rate;     // cost (in bits) to perform each quantization
  Real *dist;     // distortion incurred using each quantizer 
};

/*---------------------------------------------------------------------------*/
#endif
/*---------------------------------------------------------------------------*/
