/*****************************************************************************
 **                                                                         **
 **  drawmv.c                                                               **
 **                                                                         **
 **  author : Seung-Jong Choi                                               **
 **                                                                         **
 *****************************************************************************/
#include <stdio.h>
#include <rasterfile.h>
#include <math.h>
#include <pixrect/pixrect.h>     /* refer /usr/include/pixrect */
#include <pixrect/pr_io.h>       /* refer /usr/include/pixrect */
#include <pixrect/memvar.h>
#include "struct.h"

#define MAX     80
#define WHITE   255
#define BLACK   1

char  inname[MAX], mvname[MAX], infoname[MAX], rasname[MAX];
int   arrow, index, backimage, rectangle, printout;

main(argc, argv)
     int argc;
     char *argv[];
{
  FILE  *fp;
  Pixrect *pr;
  char  ttname[MAX];
  float mvx, mvy;
  mvnode     mvtop;
  vector_ptr fmv;
  videoinfo info;
  YUVimage frame;
  colormap_t *colormap=0;
  int   i, j, x, y, X, Y;
  int   hor, ver, chor, cver, length, xblk, yblk, xnum, ynum, xblock, yblock;
  int   pr_depth=8, pr_type=RT_STANDARD, background;

  /* command-line processing */
  read_command(argc, argv);  
  
  /* read the information file */
  info.ywidth=720;  info.yheight=480;
  info.cwidth=720;  info.cheight=240;
  info.xblk=64;     info.yblk=64;
  info.xnum=12;     info.ynum=8;
  /*
  info.ywidth=352;  info.yheight=288;
  info.cwidth=176;  info.cheight=144;
  info.xblk=64;     info.yblk=64;
  info.xnum=6;     info.ynum=5;
  */

  hor = info.ywidth;  ver = info.yheight;
  chor = info.cwidth; cver = info.cheight;
  xblk = info.xblk; yblk = info.yblk;
  xnum = info.xnum; ynum = info.ynum;

  /* read the current frame */
  frame_alloc(&frame, info);
  read_frame(&frame, info, inname, index);
  
  /* allocate the memory and read the MV file*/
  fmv = (vector_ptr) calloc(xnum*ynum, sizeof(vector));
  read_mv(&mvtop, fmv, info, mvname, index);
  
  /* allocate the memory to the pixrect */
  pr = (Pixrect *) mem_create(hor, ver, pr_depth=8); 
  for(y=0 ; y<ver ; y++) {
    for(x=0 ; x<hor ; x++){
      background = (backimage)? frame.Y[y*hor+x] : WHITE;
      pr_put(pr, x, y, background);  
    }
  }

  /* plot the motion vector */
  for(y=0, Y=0 ; Y<ynum ; y+=yblk, Y++){
    for(x=0, X=0 ; X<xnum ; x+=xblk, X++){
      if(printout){
	print_mv(&fmv[Y*xnum+X], x, y, xblk, yblk, hor, ver);
      }
      else if(rectangle){
	draw_line(pr,&fmv[Y*xnum+X], x, y, xblk, yblk, hor, ver);
      }
      else{
	draw_vector(pr,&fmv[Y*xnum+X], x, y, xblk, yblk, hor, ver);
      }
      if(printout) {getchar(); printf("\n");}
    }
    if(printout) {getchar(); printf("\n");}
  }
    
  /* write the output to the file */
  sprintf(ttname, "%s%03d", rasname, index);  
  if( !(fp = fopen(ttname, "wb")) ){
    printf("error: cannot open %s file for writing\n", rasname);
    exit(1);
  }
  colormap = (colormap_t *) calloc(1, sizeof(colormap_t));
  colormap->map[0] = (unsigned char *) calloc(256, sizeof(unsigned char));
  colormap->map[1] = (unsigned char *) calloc(256, sizeof(unsigned char));
  colormap->map[2] = (unsigned char *) calloc(256, sizeof(unsigned char));
  colormap->type=1;
  colormap->length=256;
  for(i=0 ; i<3 ; i++){
    for(j=0 ; j<256 ; j++) colormap->map[i][j] = (unsigned char)j;
  }
  pr_dump(pr, fp,  colormap, pr_type, 1);
  pr_destroy(pr);
  fclose(fp);
}

/*****************************************************************************/
/*                            draw_line()                                    */
/*****************************************************************************/
draw_line(pr, fmv, cx, cy, xblk, yblk, hor, ver)
     Pixrect *pr;
     vector_ptr fmv;
     int cx, cy, xblk, yblk, hor, ver;
{
  int xblock, yblock, pr_value, pr_op=PIX_SRC;

  /* this routine draws the variable block motion vectors */
  
  pr_value = (backimage)? WHITE : BLACK;
  if(fmv->child){
    draw_line(pr, fmv->child0, cx,        cy,        xblk/2, yblk/2,hor,ver);
    draw_line(pr, fmv->child1, cx+xblk/2, cy,        xblk/2, yblk/2,hor,ver);
    draw_line(pr, fmv->child2, cx,        cy+yblk/2, xblk/2, yblk/2,hor,ver);
    draw_line(pr, fmv->child3, cx+xblk/2, cy+yblk/2, xblk/2, yblk/2,hor,ver);
  }
  else{
    /* consider the small block around the boundaries */
    xblock = (cx+xblk<=hor)?  xblk : hor-cx;
    yblock = (cy+yblk<=ver)?  yblk : ver-cy;
    if(cy+yblk<ver-1) pr_vector(pr,cx,cy+yblk,cx+xblk,cy+yblk,pr_op,pr_value);
    if(cx+xblk<hor-1) pr_vector(pr,cx+xblk,cy+yblk,cx+xblk,cy,pr_op, pr_value);
    return;
  }
}

/*****************************************************************************/
/*                            draw_vector()                                  */
/*****************************************************************************/
draw_vector(pr, fmv, cx, cy, xblk, yblk, hor, ver)
     Pixrect *pr;
     vector_ptr fmv;
     int cx, cy, xblk, yblk, hor, ver;
{
  int xblock, yblock;

  /* this routine draws the variable block motion vectors */

  if(fmv->child){
    draw_vector(pr, fmv->child0, cx,        cy,        xblk/2, yblk/2,hor,ver);
    draw_vector(pr, fmv->child1, cx+xblk/2, cy,        xblk/2, yblk/2,hor,ver);
    draw_vector(pr, fmv->child2, cx,        cy+yblk/2, xblk/2, yblk/2,hor,ver);
    draw_vector(pr, fmv->child3, cx+xblk/2, cy+yblk/2, xblk/2, yblk/2,hor,ver);
  }
  else{
    /* consider the small block around the boundaries */
    xblock = (cx+xblk<=hor)?  xblk : hor-cx;
    yblock = (cy+yblk<=ver)?  yblk : ver-cy;
    if(xblk && yblk) draw_one(pr, fmv, cx, cy, xblock, yblock);
    return;
  }
}

/*****************************************************************************/
/*                            draw_one()                                     */
/*****************************************************************************/
draw_one(pr, fmv, posx, posy, xblk, yblk)
     Pixrect *pr;
     vector_ptr fmv;
     int posx, posy;
     int xblk, yblk;
{
  int   x0, y0, x1, y1;
  float x2, y2, x3, y3, x4, y4;
  int   pr_value, pr_op=PIX_SRC;

     /****************************************************************
      *                        (x4,y4)                               *
      *                           \                                  *
      *                            \                                 *
      *     (x0,y0)---------------0-> (x1,y1)                        *
      *                  0:(x2,y2) /                                 *
      *                           /                                  *
      *                        (x3,y3)                               *
      ***************************************************************/

  pr_value = (backimage)? WHITE : BLACK;

  x0 = posx + xblk/2;   /* starting point */
  y0 = posy + yblk/2;
  x1 = x0 + nint(fmv->mvx);       /* ending pint */
  y1 = y0 + nint(fmv->mvy);

  if(!arrow){   /* write the center point */
    pr_vector(pr, x0, y0-1, x0, y0-1, pr_op, pr_value);
    pr_vector(pr, x0, y0+1, x0, y0+1, pr_op, pr_value);
    pr_vector(pr, x0-1, y0, x0-1, y0, pr_op, pr_value);
    pr_vector(pr, x0+1, y0, x0+1, y0, pr_op, pr_value);
  }

  /* draw the main displacement */
  pr_vector(pr, x0, y0, x1, y1, pr_op, pr_value);

  if(x0!=x1 || y0!=y1){
    x2 = x1 + sqrt(4.5 * (x0 - x1) * (x0 - x1) /
		   ((x0 - x1) * (x0 - x1) + (y0 - y1) * (y0 - y1)));
    x3 = x1 - sqrt(4.5 * (x0 - x1) * (x0 - x1) /
		   ((x0 - x1) * (x0 - x1) + (y0 - y1) * (y0 - y1)));
    x2 = fabs(x2 - x0) < fabs(x3 - x0) ? x2 : x3;
    y2 = y1 + sqrt(4.5 * (y0 - y1) * (y0 - y1) /
		   ((x0 - x1) * (x0 - x1) + (y0 - y1) * (y0 - y1)));
    y3 = y1 - sqrt(4.5 * (y0 - y1) * (y0 - y1) /
		   ((x0 - x1) * (x0 - x1) + (y0 - y1) * (y0 - y1)));
    y2 = fabs(y2 - y0) < fabs(y3 - y0) ? y2 : y3;
    /* now (x2, y2) is the point on the arrow, which is the point of 
     * intersection of the line joining the tip of the wings, and the 
     * arrow itself. 
     */

    /* 3.0 was 2.25*/
    x3 = x2 + sqrt(3.0 * (y0 - y1) * (y0 - y1) /
		   ((x0 - x1) * (x0 - x1) + (y0 - y1) * (y0 - y1)));
    x4 = x2 - sqrt(3.0 * (y0 - y1) * (y0 - y1) /
		   ((x0 - x1) * (x0 - x1) + (y0 - y1) * (y0 - y1)));
    if(y0 != y1){
      y3 = y2 + (x3  - x2) * (x1 - x0) / (y0 - y1);
      y4 = y2 + (x4  - x2) * (x1 - x0) / (y0 - y1);
    }
    else {
      y3 = y2 + 2;
      y4 = y2 - 2;
    }
    if(arrow){       /* draw the arrow wings */
      pr_vector(pr, nint(x3), nint(y3), x1, y1, pr_op, pr_value);
      pr_vector(pr, nint(x4), nint(y4), x1, y1, pr_op, pr_value);
    }
  }
}
			   
/****************************************************************************/
/*                          read_command()                                  */
/****************************************************************************/
read_command(argc, argv)
     int argc;
     char **argv;
{
  int argnum=1, i;

/* set the default value */
  arrow=1;  backimage=1;
  rectangle=0; printout=0;

/* read the argument and set the value */
  for(i=1; i<argc ; i++){
    if(*(argv[i])=='-'){
      switch(*(++argv[i])){
      default:
	printf("-%c such an option is not available\n", *(argv[i]));
	usage(); exit(1);
      case 'a':
	arrow=0; break;
      case 'b':
	backimage=0; break;
      case 'p':
	printout=1; break;
      case 'r':
	rectangle=1; break;
      }
    }
    else{
      switch(argnum){
      default:
	printf("more parameters are specified\n");
	usage(); exit(1);
      case 1:
	strcpy(inname, argv[i]);
	argnum++; break;
      case 2:
	strcpy(mvname, argv[i]);
	argnum++; break;
      case 3:
	strcpy(rasname, argv[i]);
	argnum++; break;
      case 4:
        index=atoi(argv[i]);
	argnum++; break;
      }
    }
  }

  if(argnum==1){
    usage();
    exit(1);
  }
     
}

/****************************************************************************/
/*                            usage()                                       */
/****************************************************************************/
usage()
{ 
  printf("drawmv: inname, mvname, rasname, index\n");
  printf("-a arrow Off(def. On)\n");
  printf("-b background=black\n");
  printf("-r rectangular rather than arrow\n");
  printf("-p prinout\n");
  printf("ex> drawmv sif. mv zzz 1    (image+arrow)\n");
  printf("ex> drawmv sif. mv zzz 1 -r (image+block)\n");
  printf("ex> drawmv sif. mv zzz 1 -b (     +arrow)\n");
  printf("ex> drawmv sif. mv zzz 1 -p (print)\n");
}
