#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "structN.h"
#include "rasterfile.h"
#include "basic.h"
#include "ras_util.h"
#include "display.h"
#include "dpx.h"
#include "unix_pc.h"
#include "chrom.h"
#include "miscN.h"

#define RGBYUV //do RGB to YUV conversion
//#define OUTPUT_RAS


/*****************************************************************************/
/*                                 clipping()                                */
/*****************************************************************************/
int clipping(float ipix)
{
  int opix;

  if     (ipix>255.0) opix = 255;
  else if(ipix<0.0)   opix = 0;
  else              opix = nint(ipix);

  return (opix);
}

/*****************************************************************************/
/*                                 clipping2()                               */
/*****************************************************************************/
float clipping2(float ipix)
{
  float opix;

  if     (ipix>255.0) opix = 255.;
  else if(ipix<0.0)   opix = 0.;
  else              opix = ipix;

  return (opix);
}
/*****************************************************************************/
/*                                read_header()                              */
/*****************************************************************************/
void read_header(char *name, videoinfo *info)
{
  FILE *fpio;
  videoheader header;

  if(!(fpio = fopen(name, "rb"))){
    printf("read_header: %s\n", name); exit(1);
  }
  fread(&header, sizeof(videoheader), 1, fpio); 
  fclose(fpio);

  header2info(info, header);  

}

/*****************************************************************************/
/*                               write_header()                              */
/*****************************************************************************/
void write_header(char *name, videoinfo info)
{
  FILE *fpio;
  videoheader header;

  info2header(&header, info);

  if(!(fpio = fopen(name, "wb"))){
    printf("write_header: %s\n", name); exit(1);
  }
  fwrite(&header, sizeof(videoheader), 1, fpio); 
  fclose(fpio);
}

/*****************************************************************************/
/*                                 read_child()                              */
/*****************************************************************************/
void read_child(vector_ptr fmv, FILE *fpio)
{
  /* if child is NULL, then there is nothing to do */
  /* else, allocate the memory */
  /*       and read the 4 child motion vectors */
  /*       and call again */

  if(fmv->child){
    fmv->child0 = (vector_ptr) calloc(1, sizeof(vector));
    fmv->child1 = (vector_ptr) calloc(1, sizeof(vector));
    fmv->child2 = (vector_ptr) calloc(1, sizeof(vector));
    fmv->child3 = (vector_ptr) calloc(1, sizeof(vector));

    fread(fmv->child0, sizeof(vector), 1, fpio);
    fread(fmv->child1, sizeof(vector), 1, fpio);
    fread(fmv->child2, sizeof(vector), 1, fpio);
    fread(fmv->child3, sizeof(vector), 1, fpio);

    fmv->child0->parent = fmv;
    fmv->child1->parent = fmv;
    fmv->child2->parent = fmv;
    fmv->child3->parent = fmv;

    read_child(fmv->child0, fpio);
    read_child(fmv->child1, fpio);
    read_child(fmv->child2, fpio);
    read_child(fmv->child3, fpio);
  }
  else{
  }
}


/*****************************************************************************/
/*                                read_mv()                                  */
/*****************************************************************************/
void read_mv(mvnode_ptr mvtop, vector_ptr fmv, videoinfo info, char *mvname, int index)
{
  FILE *fpio;
  char ttname[80];
  int  i, size;

  /* this routine reads the quadtree structured motion vectors */
  /* first, read the 1st layer vectors */
  /* and then call the read_child routine which recursively operates */

  sprintf(ttname, "%s%03d", mvname, index);
  if(!(fpio = fopen(ttname, "rb"))){
    printf("read_mv: %s\n", ttname); exit(1);
  }

  /* read the top value */
  fread(mvtop, sizeof(mvnode), 1, fpio);        /* read every 1st layers */

  size = (info.xnum) * (info.ynum);    
  fread(fmv, sizeof(vector), size, fpio);        /* read every 1st layers */
  for(i=0 ; i<size ; i++) read_child(fmv+i, fpio); /* read children */

  fclose(fpio);

}

/*****************************************************************************/
/*                               write_child()                               */
/*****************************************************************************/
void write_child(vector_ptr fmv, FILE *fpio)
{
  /* if child is NULL, then there is nothing to do */
  /* else, write the 4 child motion vectors and call again */

  if(fmv->child){
    fwrite(fmv->child0, sizeof(vector), 1, fpio);
    fwrite(fmv->child1, sizeof(vector), 1, fpio);
    fwrite(fmv->child2, sizeof(vector), 1, fpio);
    fwrite(fmv->child3, sizeof(vector), 1, fpio);

    write_child(fmv->child0, fpio);
    write_child(fmv->child1, fpio);
    write_child(fmv->child2, fpio);
    write_child(fmv->child3, fpio);
  }
  else{
  }
}


/*****************************************************************************/
/*                                   write_mv()                              */
/*****************************************************************************/
void write_mv(mvnode_ptr mvtop, vector_ptr fmv, videoinfo info, char *mvname, int index)
{
  FILE *fpio;
  char ttname[80];
  int  i, size;

  /* this routine writes the quadtree structured motion vectors */
  /* first, write the 1st layer vectors */
  /* and then call the write_child routine which recursively operates */

  sprintf(ttname, "%s%03d", mvname, index);
  if(!(fpio = fopen(ttname, "wb"))){
    printf("write_mv: %s\n", ttname); exit(1);
  }

  fwrite(mvtop, sizeof(mvnode), 1, fpio);        /* read every 1st layers */

  size = (info.xnum) * (info.ynum);
  fwrite(fmv, sizeof(vector), size, fpio);        /* write every 1st layers */
  for(i=0 ; i<size ; i++) write_child(fmv+i, fpio); /* write children */

  fclose(fpio);
}

/*****************************************************************************/
/*                                   print_mv()                              */
/*****************************************************************************/
void print_mv(vector_ptr fmv, int cx, int cy, int xblk, int yblk, int hor, int ver)
{
  float mvx, mvy, mad;

  
  /* this routine draws the variable block motion vectors */

  if(fmv->child){

	  if(xblk >=32){
        mvx = fmv->mvx;
        mvy = fmv->mvy;
        mad = fmv->mad;
        printf("x %d y %d xblk %d yblk %d mvx %.1f mvy %.1f mad %.1f\n", cx, cy, xblk, yblk, mvx, mvy, mad);
	  }
    print_mv(fmv->child0, cx,        cy,        xblk/2, yblk/2, hor, ver);
    print_mv(fmv->child1, cx+xblk/2, cy,        xblk/2, yblk/2, hor, ver);
    print_mv(fmv->child2, cx,        cy+yblk/2, xblk/2, yblk/2, hor, ver);
    print_mv(fmv->child3, cx+xblk/2, cy+yblk/2, xblk/2, yblk/2, hor, ver);
  }
  else{
    if(cx<hor && cy<ver && xblk>=32){
      mvx = fmv->mvx;
      mvy = fmv->mvy;
      mad = fmv->mad;
      printf("x %d y %d xblk %d yblk mvx %.1f mvy %.1f mad %.1f\n", cx, cy, xblk, yblk, mvx, mvy, mad);
	}

    return;
  }
}



/*****************************************************************************/
/*                              print_vector()                               */
/*****************************************************************************/
void print_vector(vector_ptr fmv, videoinfo info)
{
  int x, y, X, Y, xnum, ynum, xblk, yblk, hor, ver;

  xnum = info.xnum; ynum = info.ynum; 
  xblk = info.xblk; yblk = info.yblk; 
  hor  = info.ywidth; ver = info.yheight; 


  for(y=0, Y=0 ; Y<ynum ; y+=yblk, Y++){
    for(x=0, X=0 ; X<xnum ; x+=xblk, X++){
      print_mv(&fmv[Y*xnum+X], x, y, xblk, yblk, hor, ver);
      getchar(); printf("\n");
    }
    getchar(); printf("\n");
  }
}


/*****************************************************************************
 **                                                                         ** 
 **  ras2yuv                                                                **
 **                                                                         ** 
 **  -. convert the sun rasterfile to YUV file (YUV with 4:1:1 or 4:2:2)    **
 **                                                                         ** 
 **  author : Seung-Jong Choi                                               **
 **                                                                         ** 
 *****************************************************************************/

/* sub422=0; */ /* 4:1:1 subsampling */
/*sub422=1;  */ /* 4:2:2 subsampling */
void ras2yuv(char *rasname, int yhor, int yver, float *Yfr, int chor, int cver, float *Ufr, float *Vfr)
{
  unsigned char *RGBfr, *temp, map[768];
  FILE              *fpin;
  int               i, j, hor, ver, mapsize, sub422;
  float             Ydata, Udata, Vdata, Rdata, Gdata, Bdata;
  struct rasterfile *header;
  U32 *p;

  /*
header.ras_magic = 100;
  fpin = fopen("test", "wt");
  fwrite(&header, 32, 1, fpin);
  fclose(fpin);

header.ras_magic = 0;
  fpin=fopen("test", "rt");
  fread(&header, 32, 1, fpin);
  fclose(fpin);
 */
  

  if(!(fpin = fopen(rasname, "rb"))){
    printf("ras2yuv: can't open rasterfile %s.\n", rasname);
    exit(1);
  }

  header = (struct rasterfile *)getarray(1, sizeof(struct rasterfile), "header");
    /* read the header of rasterfile */
  if(fread(header, sizeof(struct rasterfile), 1, fpin)!=1){
	  printf("\n");
    printf("ras2yuv: can't read header from %s.\n", rasname);
    exit(1);
  }

  p =(U32 *)header;
  for(i=0; i<8; i++){
	  p[i] = exchange_4byte_order(p[i]);
  }

  header = (struct rasterfile *)p;



  hor = header->ras_width; ver = header->ras_height;
  if((yhor != hor) || (yver != ver) ){
	  printf("size of image error info.width %d, info.height %d, header->ras_width %d, header->ras_height %d (ioN.c)!\n",
		yhor, yver, hor, ver);
	  header->ras_depth = 24; hor = yhor; ver = yver;
    /*exit(0);*/
  }


  /************ allocate memory and read the data of rasterfile **********/
/*
  if(sub422){ 
    RGBfr = (unsigned char*) calloc(ver*hor*3,   sizeof(unsigned char));
    Yfr   = (float*) calloc(hor*ver,     sizeof(float));
    Ufr   = (float*) calloc((hor/2)*ver, sizeof(float));
    Vfr   = (float*) calloc((hor/2)*ver, sizeof(float));
  }
  else {   
    RGBfr = (unsigned char*) calloc(ver*hor*3,  sizeof(unsigned char));
    Yfr   = (float*) calloc(hor*ver,    sizeof(float));
    Ufr   = (float*) calloc(hor*ver/4,  sizeof(float));
    Vfr   = (float*) calloc(hor*ver/4,  sizeof(float));
  }
*/

    /* read the map */
  mapsize = header->ras_maplength; 
  if((mapsize != 768) && (mapsize != 0)){
    printf("the size of maplength %d is not 768(ioN.c)\n", mapsize);
    exit(0);
  }

  if(mapsize){
      if(fread(map, sizeof(unsigned char), mapsize, fpin) != (unsigned)mapsize){
	printf("ras2yuv: error in reading of map from %s.\n", rasname);
	exit(1);
      }
  }

    /********************* read the data *****************/
  if(header->ras_depth==GRAYDATA){
    if((chor != 0) || (cver != 0)){
        printf(" the size of chrom component error(ras2yuv.c)!\n");
        exit(0);
	}
    temp = (unsigned char *)calloc(hor*ver, sizeof(unsigned char));
    if(fread(temp, sizeof(unsigned char), hor*ver, fpin) != (unsigned)(hor*ver)){
	printf("ras2yuv: can't read the Y data from %s.\n", rasname);
	exit(1);
      }
    for( i = 0; i < hor*ver; i++)
      Yfr[i] = (float)temp[i];
    free(temp);
  }
  else	if(header->ras_depth==RGBDATA){

      if((yhor == 2*chor) && (yver == 2*cver))
        sub422 = 0;
      else
        if((yhor == 2*chor) && (yver == cver))
          sub422 = 1;
		else 
			if((yhor == chor) && (yver == cver))
			   sub422 = 2;
            else{
               printf("size of image error(ras2yuv.c)\n");
               exit(0);
			}

      RGBfr = (unsigned char*) calloc(ver*hor*3,  sizeof(unsigned char));
      if(fread(RGBfr, sizeof(unsigned char), 3*hor*ver, fpin) != (unsigned)(3*hor*ver)){
    	printf("ras2yuv: can't read the RGB data from %s.\n", rasname);
	    exit(1);
      }
      for(i=0 ; i<ver ; i++){
    	for(j=0 ; j<hor ; j++){
	  /* read the data and change to the float */
	    Bdata = (float) RGBfr[i*(3*hor)+3*j];
	    Gdata = (float) RGBfr[i*(3*hor)+3*j+1];
	    Rdata = (float) RGBfr[i*(3*hor)+3*j+2];

		RGB2YUV(Rdata, Gdata, Bdata, &Ydata, &Udata, &Vdata);
  	    Udata += 128.;
	    Vdata += 128.;



	  /* clipping the data */
/*	  if(Ydata>=255.) Ydata=255.; if(Ydata<=0.) Ydata=0.;
	  if(Udata>=255.) Udata=255.; if(Udata<=0.) Udata=0.;
	  if(Vdata>=255.) Vdata=255.; if(Vdata<=0.) Vdata=0.;
*/

	  if(sub422 == 2){
  	    Yfr[i*hor+j] = Ydata;
	    Ufr[i*hor+j] = Udata;
	    Vfr[i*hor+j] = Vdata;
	  }
	  else
		  if(sub422 == 1){
	        Yfr[i*hor+j] = Ydata;
	        if(!(j%2)){
	          Ufr[i*(hor/2)+(j/2)] = Udata;
	          Vfr[i*(hor/2)+(j/2)] = Vdata;
			}
		  }
	      else {
	        Yfr[i*hor+j] = Ydata;
	        if(!(j%2) && !(i%2)){
	          Ufr[(i/2)*(hor/2)+j/2] = Udata;
	          Vfr[(i/2)*(hor/2)+j/2] = Vdata;
			}
		  }
	} /* for j*/
	  } /* for i */
      free(RGBfr);

  }
  else {
      printf("ras2yuv: %s is not supported.\n", rasname);
      exit(1);
  }


  fclose(fpin);

free(header);

}



/*****************************************************************************
 **                                                                         ** 
 **  yuv2ras                                                                **
 **                                                                         ** 
 **  -. convert the YUV file to sun rasterfile (YUV with 4:1:1 or 4:2:2)    **
 **                                                                         ** 
 **  author : Seung-Jong Choi                                               **
 **                                                                         ** 
 *****************************************************************************/

void yuv2ras(char *rasname, int yhor, int yver, float *Yframe, int chor, int cver, float *Uframe, float *Vframe)
{
  int i, j, ci, cj;
  float Ydata, Udata, Vdata, Rdata, Gdata, Bdata;
  struct rasterfile header;
  unsigned char *RGBframe, *temp;

  ci = (cver)? yver/cver : 0;
  cj = (chor)? yhor/chor : 0;

  /* allocate memory */

 
/*
  Yframe    = (unsigned char*) calloc(yhor*yver,   sizeof(unsigned char));
  Uframe    = (unsigned char*) calloc(chor*cver,   sizeof(unsigned char));
  Vframe    = (unsigned char*) calloc(chor*cver,   sizeof(unsigned char));
*/


    /* read the data from SIF or CCIR */
/*
    if(!(fpin  = fopen(yuvname, "rb"))){
      printf("yuv2ras: can't open %s for reading.\n", yuvname);
      exit(1);
    }

    if(fread(Yframe, sizeof(unsigned char), yhor*yver, fpin) != yhor*yver){
      printf("yuv2ras: can't read Y from %s\n", yuvname);
      exit(1);
    }
    if(fread(Uframe, sizeof(unsigned char), chor*cver, fpin) != chor*cver){
      printf("yuv2ras: can't read U from %s\n", yuvname);
      exit(1);
    }
    if(fread(Vframe, sizeof(unsigned char), chor*cver, fpin) != chor*cver){
      printf("yuv2ras: can't read V from %s\n", yuvname);
      exit(1);
    }
    fclose(fpin);
*/



    /* convert YUV to BGR */
    if(chor*cver){
      RGBframe  = (unsigned char*) calloc(yhor*yver*3, sizeof(unsigned char));
      for(i=0 ; i<yver ; i++){
        for(j=0 ; j<yhor ; j++){
          Ydata =  clipping2(Yframe[i*yhor+j]);
          Udata =  clipping2(Uframe[(i/ci)*chor+(j/cj)]);
          Vdata =  clipping2(Vframe[(i/ci)*chor+(j/cj)]);

          /* YUV -> RGB */
		  Udata -= 128.;
		  Vdata -= 128.;
          YUV2RGB(Ydata, Udata, Vdata, &Rdata, &Gdata, &Bdata);

          /* clipping */
          if(Rdata>=255.) Rdata=255.; if(Rdata<=0.) Rdata=0.;
          if(Gdata>=255.) Gdata=255.; if(Gdata<=0.) Gdata=0.;
          if(Bdata>=255.) Bdata=255.; if(Bdata<=0.) Bdata=0.;

          /* formatting */
          RGBframe[i*3*yhor+3*j]   = (unsigned char)nint(Bdata);
          RGBframe[i*3*yhor+3*j+1] = (unsigned char)nint(Gdata);
          RGBframe[i*3*yhor+3*j+2] = (unsigned char)nint(Rdata);
        }
      }
      header = make_header(yhor, yver, RGBDATA);

      write_ras(rasname, header, RGBframe);
      free(RGBframe);
    }
    else {   /* gray file */
      temp = (unsigned char*) calloc(yhor*yver, sizeof(unsigned char));
      header = make_header(yhor, yver, GRAYDATA);
      for(i = 0; i < yhor*yver; i++)
        {
          if(Yframe[i]>=255.) temp[i] = 255; 
          else
            if(Yframe[i]<=0.) temp[i] = 0;
            else
              temp[i] = (unsigned char)Yframe[i];
        }
      write_ras(rasname, header, temp);
      free(temp);
    }



}




/*****************************************************************************
 **                                                                         ** 
 **  yuv2RGB                                                                **
 **                                                                         ** 
 **  -. convert the YUV file to RGB  (YUV with 4:1:1 or 4:2:2)              **
 **                                                                         ** 
 **                                                                         **
 **                                                                         ** 
 *****************************************************************************/
 
void yuv2RGB(unsigned char *RGBframe, int yhor, int yver, float *Yframe, int chor, int cver, float *Uframe, float *Vframe)
{
  int i, j, ci, cj;
  float Ydata, Udata, Vdata, Rdata, Gdata, Bdata;



  ci = (cver)? yver/cver : 0;
  cj = (chor)? yhor/chor : 0;

  /* allocate memory */



    /* convert YUV to BGR */
    if(chor*cver){
/*      RGBframe  = (unsigned char*) calloc(yhor*yver*3, sizeof(unsigned char));*/
      for(i=0 ; i<yver ; i++){
        for(j=0 ; j<yhor ; j++){
          Ydata =  clipping2(Yframe[i*yhor+j]);  /* QUEST */
          Udata =  clipping2(Uframe[(i/ci)*chor+(j/cj)]);
          Vdata =  clipping2(Vframe[(i/ci)*chor+(j/cj)]);

          /* YUV -> RGB */
		  Udata -= 128.;
		  Vdata -= 128.;
          YUV2RGB(Ydata, Udata, Vdata, &Rdata, &Gdata, &Bdata);

          /* clipping */
          if(Rdata>=255.) Rdata=255.; if(Rdata<=0.) Rdata=0.;
          if(Gdata>=255.) Gdata=255.; if(Gdata<=0.) Gdata=0.;
          if(Bdata>=255.) Bdata=255.; if(Bdata<=0.) Bdata=0.;

          /* formatting */
          RGBframe[i*3*yhor+3*j]   = (unsigned char)nint(Bdata);
          RGBframe[i*3*yhor+3*j+1] = (unsigned char)nint(Gdata);
          RGBframe[i*3*yhor+3*j+2] = (unsigned char)nint(Rdata);
        }
      }


    }
    else {   /* gray file */

      for(i = 0; i < yhor*yver; i++)
        {
          if(Yframe[i]>=255.) RGBframe[i] = 255; 
          else
            if(Yframe[i]<=0.) RGBframe[i] = 0;
            else
              RGBframe[i] = (unsigned char)Yframe[i];
        }
    }



}


/*****************************************************************************/
/*                              read_frame()                                 */
/*****************************************************************************/
void read_frame(YUVimage_ptr frame, videoinfo info,  char *inname, int index, enum FORMAT format)
{
  int  i, j, size, ret, row, col, pos;
  float Rdata, Gdata, Bdata;
  unsigned char *temp;
  float *tempf;
  char frame_name[80];
  U16 *RGBframe;
  //float *LRGB;
  struct dpx *dpxheader;
  FILE *fpio;

  switch(format){
  case RAS:
	  sprintf(frame_name, inname, index);
	  ras2yuv(frame_name, info.ywidth, info.yheight, frame->Y, info.cwidth, info.cheight, frame->U, frame->V);
	  break;
  case YUV:
    /* assume that the order of data file is Y[total], U[total], V[total] */
    sprintf(frame_name, inname, index);
    if(!(fpio = fopen(frame_name, "rb"))){
      printf("read_frame: %s\n", frame_name); exit(1);
    }

    size = (info.ywidth) * (info.yheight);
    temp = (unsigned char *) getarray(size, sizeof(unsigned char), "temp");
    if((ret=fread(temp, sizeof(unsigned char), size, fpio))!=size){

      printf("error for reading file %s.Y in read_frame(): return size %d, target size %d\n", frame_name, ret, size);
      exit(1);
    }
    for(i=0 ; i<size ; i++) frame->Y[i] = (float) temp[i];

    size = (info.cwidth) * (info.cheight);
    if(fread(temp, sizeof(unsigned char), size, fpio)!=(unsigned)(size)){
      printf("error for reading file %s.U in read_frame()\n", frame_name);
      exit(1);
    }
    for(i=0 ; i<size ; i++) frame->U[i] = (float) temp[i];

    if(fread(temp, sizeof(unsigned char), size, fpio)!=(unsigned)(size)){
      printf("error for reading file %s.V in read_frame()\n", frame_name);
      exit(1);
    }
    for(i=0 ; i<size ; i++) frame->V[i] = (float) temp[i];

    free(temp);
    fclose(fpio);
	break;

  case DPX:
	  //printf("index %d %d\n", index, 0);
	  sprintf(frame_name, inname, index);
	  RGBframe = read_dpx(frame_name, &dpxheader, &row, &col);

	  if( (row != info.yheight) || (col != info.ywidth) ){
		  printf(" the size of image does not match with the info\n");
		  exit(1);
	  }
	  

      for(i=0 ; i<info.yheight; i++){
        for(j=0 ; j<info.ywidth; j++){
			Bdata = RGBframe[i*(3*info.ywidth)+3*j];
			Gdata = RGBframe[i*(3*info.ywidth)+3*j+1];
			Rdata = RGBframe[i*(3*info.ywidth)+3*j+2];

			if(info.coding_domain != LOG){ 
			  Bdata = convertLOG(PEAK, Bdata, info.coding_domain);
			  Gdata = convertLOG(PEAK, Gdata, info.coding_domain);
			  Rdata = convertLOG(PEAK, Rdata, info.coding_domain);
			}
			/* convert RGB to YUV */
			pos = i* info.ywidth + j; 

#ifdef RGBYUV
            RGB2YUV(Rdata, Gdata, Bdata, &(frame->Y[pos]), &(frame->U[pos]), &(frame->V[pos]));

			if(info.coding_domain != LOG){ 
			  frame->U[pos] += (PEAK+1)/2;
			  frame->V[pos] += (PEAK+1)/2;
			}
			else{
			  frame->U[pos] += 1024/2;
			  frame->V[pos] += 1024/2;
			}
#else
			frame->Y[pos] = Rdata;
			frame->U[pos] = Gdata;
			frame->V[pos] = Bdata;
#endif
		}
	  }
     

//      sprintf(dpxname, "%s%04d%s", "temp", index, ".dpx");
//      write_dpx(dpxname, *frame, info);

	  free(RGBframe);
	  free(dpxheader);

	  break;
  /* IVB 2004/4/7 ---------------------------------------------- */
  case FLT:
	  /* assume that the order of data file is Y[total], U[total], V[total] */
    sprintf(frame_name, inname, index);
    if(!(fpio = fopen(frame_name, "rb"))){
      printf("read_frame: %s\n", frame_name); exit(1);
    }

    size = (info.ywidth) * (info.yheight);
    tempf = (float *) getarray(size, sizeof(float), "tempf");
    if((ret=fread(tempf, sizeof(float), size, fpio))!=size){

      printf("error for reading file %s.Y in read_frame(): return size %d, target size %d\n", frame_name, ret, size);
      exit(1);
    }
    for(i=0 ; i<size ; i++) frame->Y[i] = (float) tempf[i];

    size = (info.cwidth) * (info.cheight);
    if(fread(tempf, sizeof(float), size, fpio)!=(unsigned)(size)){
      printf("error for reading file %s.U in read_frame()\n", frame_name);
      exit(1);
    }
    for(i=0 ; i<size ; i++) frame->U[i] = (float) tempf[i];

    if(fread(tempf, sizeof(float), size, fpio)!=(unsigned)(size)){
      printf("error for reading file %s.V in read_frame()\n", frame_name);
      exit(1);
    }
    for(i=0 ; i<size ; i++) frame->V[i] = (float) tempf[i];

    free(tempf);
    fclose(fpio);

	break;
  /* IVB 2004/4/7 -------------- end --------------------------- */
  default:
	  printf("image format error    format = %d(ioN.c)\n", format);
	  exit(0);

  }


}


/*****************************************************************************/
/*                               write_frame()                               */
/*****************************************************************************/
void write_frame(YUVimage frame, videoinfo info, char *inname, int index, enum FORMAT format)
{
  FILE *fpio;
  int  i, size;
  char frame_name[80];
  unsigned char *temp;
  float *tempf;

  switch(format){
  case RAS:
    sprintf(frame_name, inname, index);
    yuv2ras(frame_name, info.ywidth, info.yheight, frame.Y, info.cwidth, info.cheight, frame.U, frame.V);
  break;

  case YUV:
    /* assume that the order of data file is Y[total], U[total], V[total] */
    sprintf(frame_name, inname, index);
    if(!(fpio = fopen(frame_name, "wb"))){
      printf("write_frame: %s\n", frame_name); exit(1);
    }
    size = (info.ywidth) * (info.yheight);
    temp = (unsigned char *) calloc(size, sizeof(unsigned char));
    for(i=0 ; i<size ; i++)  temp[i] = (unsigned char)clipping(frame.Y[i]);
    if(fwrite(temp, sizeof(unsigned char), size, fpio)!=(unsigned)(size)){
      printf("error in write_frame()\n");
      exit(1);
    }

    size = (info.cwidth) * (info.cheight);  /* QUEST */
    for(i=0 ; i<size ; i++)  temp[i] = (unsigned char)clipping(frame.U[i]);
    if(fwrite(temp, sizeof(unsigned char), size, fpio)!=(unsigned)(size)){
      printf("error in write_frame()\n");
      exit(1);
    }

    for(i=0 ; i<size ; i++) temp[i] = (unsigned char)clipping(frame.V[i]);
    if(fwrite(temp, sizeof(unsigned char), size, fpio)!=(unsigned)(size)){
      printf("error in write_frame()\n");
      exit(1);
    }
    free(temp);
    fclose(fpio);
  
	break;

  case DPX:
	int j, pos;
    float Ydata, Udata, Vdata;
    float  *LRGB;

    LRGB = (float *)getarray(info.ywidth*info.yheight*3, sizeof(float), "LRGB");
    
	for(i=0 ; i<info.yheight; i++){
	  for(j=0 ; j<info.ywidth; j++){
		pos=i*info.ywidth+j;
		Ydata =  frame.Y[pos];
		Udata =  frame.U[pos];
		Vdata =  frame.V[pos];
		/* YUV -> RGB */
		if(info.coding_domain != LOG){
		  Udata -= (PEAK+1)/2.;
		  Vdata -= (PEAK+1)/2.;
		}
		else{
		  Udata -= 1024/2.;
		  Vdata -= 1024/2.;
		}
		YUV2RGB(Ydata, Udata, Vdata, &(LRGB[pos*3+2]), &(LRGB[pos*3+1]), &(LRGB[pos*3]));
	  }
	}
	  //printf("Check the DPX header (ioN.c) \n"); getchar();
	  
	  sprintf(frame_name, inname, index);

	  save_dpx_header(info, index); // save the header from the original dpx file for the reconstructed frame

	  write_dpx(frame_name, LRGB, info);


#ifdef OUTPUT_RAS
	  unsigned char *RGBframe;
	  int data;
	  struct rasterfile header;
	  char rasname[80];

	  RGBframe = (unsigned char *)getarray(info.ywidth*info.yheight*3, sizeof(unsigned char), "LRGB");

	  strcpy(rasname, frame_name);
	  strcat(rasname, ".ras");

      header = make_header(info.ywidth, info.yheight, RGBDATA);

	  switch (info.coding_domain){
	  case VIDEO:
		for(i=0; i<info.ywidth*info.yheight*3; i++){

		  data = nint(LRGB[i]*255./PEAK);
		  if(data<0)
			RGBframe[i] = 0;
		  else if(data > 255)
			RGBframe[i] = 255;
		  else
			RGBframe[i] = data;
		}
		break;

	  case LINEAR: // so we need to do Gamma correction
		for(i=0; i<info.ywidth*info.yheight*3; i++){  
		  
		  LRGB[i] = LRGB[i]/PEAK;
		  if(LRGB[i] > 1) LRGB[i] = 1;
		  else if(LRGB[i] < 0) LRGB[i] = 0;

		  data = nint(255 * pow(LRGB[i], 1./Gamma));
		  if(data<0)
			RGBframe[i] = 0;
		  else if(data > 255)
			RGBframe[i] = 255;
		  else
			RGBframe[i] = data;
		}
		break;

	  case LOG: 
		for(i=0; i<info.ywidth*info.yheight*3; i++){

		  //LRGB[i] = LRGB[i]*1023/PEAK;

		  if(LRGB[i] < 0) LRGB[i] = 0;
		  else if(LRGB[i]>1023) LRGB[i] = 1023;

          RGBframe[i] = nint(convertLOG(255, LRGB[i], VIDEO));
		}
		break;

	  default:
		printf("coding_domain error (ioN.c)\n");
		exit(1);
	  }			
	
	  write_ras(rasname, header, RGBframe);

	  free(RGBframe);
#endif

	  free(LRGB); 
 	  break;

  /* IVB 2004/4/7 ---------------------------------------------- */
  case FLT:
	/* assume that the order of data file is Y[total], U[total], V[total] */
    sprintf(frame_name, inname, index);
    if(!(fpio = fopen(frame_name, "wb"))){
      printf("write_frame FLT: %s\n", frame_name); exit(1);
    }
    size = (info.ywidth) * (info.yheight);
    tempf = (float *) calloc(size, sizeof(float));
    for(i=0 ; i<size ; i++)  tempf[i] = frame.Y[i];
    if(fwrite(tempf, sizeof(float), size, fpio)!=(unsigned)(size)){
      printf("error in write_frame()\n");
      exit(1);
    }

    size = (info.cwidth) * (info.cheight);  /* QUEST */
    for(i=0 ; i<size ; i++)  tempf[i] = frame.U[i];
    if(fwrite(tempf, sizeof(float), size, fpio)!=(unsigned)(size)){
      printf("error in write_frame()\n");
      exit(1);
    }

    for(i=0 ; i<size ; i++) tempf[i] = frame.V[i];
    if(fwrite(tempf, sizeof(float), size, fpio)!=(unsigned)(size)){
      printf("error in write_frame()\n");
      exit(1);
    }
    free(tempf);
    fclose(fpio);

	break;
  /* IVB 2004/4/7 -------------- end --------------------------- */

  default:
	  printf("image format error format = %d(ioN.c)\n", format);
	  exit(0);

  }

}







 


















