function GME_complexity_noise();

clear all
close all


fprintf(1,'****************************************************\n');
fprintf(1,'*      GME performance analysis on synthetic MVF   *\n');
fprintf(1,'****************************************************\n');
fprintf(1,'       Designed by Yue-Meng Chen, Nov,  2009      \n\n');

%% parameters of a synthetic motion field
%
blkSiz = 16; % block size
MVRES = 1;  % integer-pixel: 1, half-pixel: 2
height = 288; nH = height/blkSiz; % CIF resolution
width =352; nW = width/blkSiz;

% the image coordinates
[coorBlkY,coorBlkX]=ndgrid(1:nH,1:nW);
coorX = coorBlkX.*blkSiz-blkSiz/2;
coorY = coorBlkY.*blkSiz-blkSiz/2;

% four global motion models to be tested!
m(1, :) =   [0.95     0        10.4238    0      0.95      5.7927    0           0];
m(2, :) =   [0.9964   -0.0249  1.0981     0.0856 0.9457    -7.2      0           0];
m(3, :) =   [0.9964   -0.0249  6.0981     0.0249 0.9964    2.5109    -2.7e-5     1.9e-5];
m(4, :) =   [1        0        4.4154     0      1         0         -1.1263e-4  0];

% standard deviation of Gaussian noise
stdDev = [0.7, 1.5, 2.2, 3.0];

% the parameters for the cascade
MAXITER = 10;
rOUTLIERS = 0.7; % target percentage of outliers in a motion field.
% the threshold of gradient descent for GME with the cascade
REJRATE_CAS = 0.9; 
% the threshold of gradient descent for GME without the cascade
REJRATE_ORI = rOUTLIERS*REJRATE_CAS; 

% the global motion model
GMMODE = 4; % 1: Translational, 2: Isotropic, 3: Affine, 4: Perspective

for GMID = 1:4
    % generate motion field in floating point
    [px, py] = mvGen_f(m(GMID, :), coorX, coorY);

    for nSigma = 1:4
        
        if nSigma == 2 || nSigma == 3
            continue;
        else
            [nSigma]
        end
        for runs = 1:1:50

            % Gaussian noise on X/Y channel, and rounding it to integer
            % pixel
            nX = stdDev(nSigma).*randn(nH, nW);
            npx = round(px+nX);
            nY = stdDev(nSigma).*randn(nH, nW);
            npy = round(py+nY);

            %% GME with cascade
            casGDsnr(runs, :) = snrGMECascade(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_CAS, nH, nW);
            
            %% GME with plain gradient descent
            GDsnr(runs, :) = snrGMEPureGD(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW);
            
            %% GME with filter from Ref. [6]
            fltSNR(runs, :) = snrGMEFilter(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW);
            
            %% GME with LS based RANSAC
            ranSNR(runs, :) = snrGMERansac(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW);
            
            %% GME with LS using M-estimator
            LSMESNR(runs, :) = snrGMELSME(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW);
        end

        % average over 50 runs
        mGDsnr(nSigma, :) = mean(GDsnr);
        mCasGDsnr(nSigma, :) = mean(casGDsnr);
        mRANsnr(nSigma, :) = mean(ranSNR);
        mFLTsnr(nSigma, :) = mean(fltSNR);
        mLSMEsnr(nSigma, :) = mean(LSMESNR);
        
        % plot the results
        ploting(nSigma, GMID, stdDev, mGDsnr, mCasGDsnr, mFLTsnr, mRANsnr, mLSMEsnr);
    end

    if GMID == 1
        break;
    end
end
disp('done!')

function ploting(nSigma, GMID, stdDev, mGDsnr, mCasGDsnr, mFLTsnr, mRANsnr, mLSMEsnr)
figure, % Sigma: 3.0      
plot(0:length(mCasGDsnr)-1, mCasGDsnr(nSigma,:), 'r-o', 'LineWidth',2)
hold on  
plot(0:length(mGDsnr)-1, mGDsnr(nSigma,:), 'b:p', 'LineWidth',2)
hold on
plot(0:length(mFLTsnr)-1, mFLTsnr(nSigma,:), 'k:^', 'LineWidth',2)
hold on
plot(0:length(mRANsnr)-1, mRANsnr(nSigma,:), 'g--+', 'LineWidth',2)
hold on
plot(0:length(mLSMEsnr)-1, mLSMEsnr(nSigma,:), 'c-d', 'LineWidth',2)
hold off
legend('CAS\_GD','GD', 'FLT\_GD', 'RAN\_LS', 'LSS-MS','FontSize',12)
box on
grid off
%         ylim([0 40])
xlabel('Number of iterations','FontSize',18)
ylabel('Average SNR (dB)','FontSize',18)
title(sprintf('SNR vs. Iterations - GM %d, sd=%1.1f', GMID,stdDev(nSigma)) ,'FontSize',18)
set(gca,'FontSize',18)

function [wSNR] = snrGMECascade(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_CAS, nH, nW)

% outlier rejection using the cascade
iMap = MVRemCas(npx, npy, rOUTLIERS, blkSiz);

% GME using Newton-Raphson method
mm0 = mvGME_NR_test(GMMODE, npx(:), npy(:), iMap(:), ...
    coorX(:), coorY(:), MAXITER, REJRATE_CAS, []);
for ii = 1:MAXITER+1
    % synthesize MV field using estimated GM parameters
    [mmpx, mmpy] = mvGen_f(mm0(ii,:), coorX, coorY);
    % SNR check-up
    wSNR(ii) = mvSNR(px, py, mmpx, mmpy, ones(nH, nW));
end

function [woSNR] = snrGMEPureGD(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW)

% No outlier rejection, all MVs are taken as inliers for plain GD-GME
iMap = ones(size(px));
mm1 = mvGME_NR_test(GMMODE, npx(:), npy(:), iMap(:), ...
    coorX(:), coorY(:), MAXITER, REJRATE_ORI, []);
for ii = 1:MAXITER+1
    [mmpx, mmpy] = mvGen_f(mm1(ii,:), coorX, coorY);% SNR check-up
    woSNR(ii) = mvSNR(px, py, mmpx, mmpy, ones(nH, nW));
end

function [fltSNR] = snrGMEFilter(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW)

% outlier rejection using filter from [6]
iMap = MVRemFlt(npx, npy);
rejRate = nH*nW*REJRATE_ORI/sum(iMap(:));
if rejRate > 1
    rejRate = 1;
end

% check if at least 4 MVs obtained after filtering
if sum(iMap(:)) > 4

    % GME based on all inliers using Newton-Raphson method
    mm0 = mvGME_NR_test(GMMODE, npx(:), npy(:), iMap(:), ...
        coorX(:), coorY(:), MAXITER, rejRate, []);
    for ii = 1:MAXITER+1
        [mmpx, mmpy] = mvGen_f(mm0(ii,:), coorX, coorY);% SNR check-up
        fltSNR(ii) = mvSNR(px, py, mmpx, mmpy, ones(nH, nW));
    end
else
    fltSNR(1:MAXITER+1) = 0;
end
            
function [ranSNR] = snrGMERansac(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW)

% get the reference coordinates
rcoorX = coorX + npx;
rcoorY = coorY + npy;
for ii = 1:MAXITER+1
    [ranM, iter] = ranGME(GMMODE, rcoorX(:), rcoorY(:), coorX(:), coorY(:), ii);
    if isempty(ranM)
        if ii == 1
            ranSNR(ii) = 0;
        else
            ranSNR(ii) = ranSNR(ii-1);
        end
    else
        [mmpx, mmpy] = mvGen_f(ranM, coorX, coorY);% SNR check-up
        rsnr = mvSNR(px, py, mmpx, mmpy, ones(nH, nW));
        if ii == 1
            ranSNR(ii) = max(0, rsnr);
        else
            ranSNR(ii) = max(ranSNR(ii-1), rsnr);
        end
    end
end

function [LSMESNR] = snrGMELSME(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW)


for ii = 1:MAXITER+1
    if ii == 1
        % initializing the parameter
        MM = zeros(1,8);
        MM(1) = 1;
        MM(5) = 1;
        MM(3) = sum(npx(:))/(nH*nW);
        MM(6) = sum(npy(:))/(nH*nW);
    else
        % GME using weighted least square solution
        MM = gmeLsme(coorX, coorY, npx, npy, MM);
    end
    
    % generate MV field using GM parameter estimates
    [mmpx, mmpy] = mvGen_f(MM, coorX, coorY); 
    % SNR check-up
    LSMESNR(ii) = mvSNR(px, py, mmpx, mmpy, ones(nH, nW));
end
