function GME_comp_outlier_noise();

clear all
close all


fprintf(1,'****************************************************\n');
fprintf(1,'*      GME performance analysis on synthetic MVF   *\n');
fprintf(1,'****************************************************\n');
fprintf(1,'       Designed by Yue-Meng Chen, Jul,  2009      \n\n');

%% parameters of a synthetic motion field
%
blkSiz = 16; % block size
MVRES = 1;  % integer-pixel: 1, half-pixel: 2
height = 288; nH = height/blkSiz; % CIF resolution
width =352; nW = width/blkSiz;

% the image coordinates
[coorBlkY,coorBlkX]=ndgrid(1:nH,1:nW);
coorX = coorBlkX.*blkSiz-blkSiz/2;
coorY = coorBlkY.*blkSiz-blkSiz/2;

% four global motion models to be tested!
m(1, :) =   [0.95     0        10.4238    0      0.95      5.7927    0           0];
m(2, :) =   [0.9964   -0.0249  1.0981     0.0856 0.9457    -7.2      0           0];
m(3, :) =   [0.9964   -0.0249  6.0981     0.0249 0.9964    2.5109    -2.7e-5     1.9e-5];
m(4, :) =   [1        0        4.4154     0      1         0         -1.1263e-4  0];

% standard deviation of Gaussian noise
stdDev = [0.7, 1.5, 2.2, 3.0];
noiID = 2; % set standard deviation of Gaussian noise to 2.2.
fgSiz = [0, 0.02, 0.10, 0.20];
fgMV = 5; % simulated size of MVs associated with a moving object

% the parameters for the cascade
MAXITER = 10;
rOUTLIERS = 0.7; % target percentage of outliers in a motion field.
% the threshold of gradient descent for GME with the cascade
REJRATE_CAS = 0.9; 
% the threshold of gradient descent for GME without the cascade
REJRATE_ORI = rOUTLIERS*REJRATE_CAS; 

% the global motion model
GMMODE = 4; % 1: Translational, 2: Isotropic, 3: Affine, 4: Perspective

for GMID = 1:4
    % generate motion field in floating point
    [px, py] = mvGen_f(m(GMID, :), coorX, coorY);

    if GMID == 1 || GMID == 2 || GMID == 4
        continue;
    end
    
    for fgType = 1:4
        [fgType]
        if fgType > 1
            % simulating MV field corrupted by both Gaussian noise and
            % moving objects
            fgN=round((fgSiz(fgType)*nH*nW).^0.5)
            fgMVF = zeros(size(px));
            strH=round((nH-fgN)/2);
            strW=round((nW-fgN)/2);
            % Moving objects always located at center of the image
            fgMVF(strH:strH+fgN-1,strW:strW+fgN-1) = fgMV;
            px = px+fgMVF;
            py = py+fgMVF;
        end
        
        for runs = 1:1:50

            % Gaussian noise on X/Y channel, and rounding it to integer
            % pixel
            nX = stdDev(noiID).*randn(nH, nW);
            npx = round(px+nX);
            nY = stdDev(noiID).*randn(nH, nW);
            npy = round(py+nY);

            %% GME with cascade
            casGDsnr(runs, :) = snrGMECascade(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_CAS, nH, nW);
            
            %% GME with pure gradient descent
            GDsnr(runs, :) = snrGMEPureGD(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW);
            
            %% GME with filter from Ref. [6]
            fltSNR(runs, :) = snrGMEFilter(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW);
            
            %% GME with LS based RANSAC
            ranSNR(runs, :) = snrGMERansac(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW);
            
            %% GME with LS using M-estimator
            LSMESNR(runs, :) = snrGMELSME(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW);

        end

        % average over 50 runs
        mGDsnr(fgType, :) = mean(GDsnr);
        mCasGDsnr(fgType, :) = mean(casGDsnr);
        mRANsnr(fgType, :) = mean(ranSNR);
        mFLTsnr(fgType, :) = mean(fltSNR);
        mLSMEsnr(fgType, :) = mean(LSMESNR);
        
        % plot the results
        ploting(fgType, GMID, fgSiz, mGDsnr, mCasGDsnr, mFLTsnr, mRANsnr, mLSMEsnr);
    end
end
disp('done!')

function ploting(fgType, GMID, fgSiz, mGDsnr, mCasGDsnr, mFLTsnr, mRANsnr, mLSMEsnr)

figure, 
plot(0:length(mCasGDsnr)-1, mCasGDsnr(fgType,:), 'r-o', 'LineWidth',2)
hold on  
plot(0:length(mGDsnr)-1, mGDsnr(fgType,:), 'b:p', 'LineWidth',2)
hold on
plot(0:length(mFLTsnr)-1, mFLTsnr(fgType,:), 'k:^', 'LineWidth',2)
hold on
plot(0:length(mRANsnr)-1, mRANsnr(fgType,:), 'g--+', 'LineWidth',2)
hold on
plot(0:length(mLSMEsnr)-1, mLSMEsnr(fgType,:), 'c-d', 'LineWidth',2)
hold off
legend('CAS\_GD','GD', 'FLT\_GD', 'RAN\_LS', 'LSS-MS','FontSize',12)
box on
grid off
xlabel('Number of iterations','FontSize',18)
ylabel('Average SNR (dB)','FontSize',18)
title(sprintf('GM %d, sd=1.5, outliers: %d%%', GMID,fgSiz(fgType)*100) ,'FontSize',18)
set(gca,'FontSize',18)

function [wSNR] = snrGMECascade(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_CAS, nH, nW)

% outlier rejection using the cascade
iMap = MVRemCas(npx, npy, rOUTLIERS, blkSiz);

% GME based on all inliers using Newton-Raphson method
mm0 = mvGME_NR_test(GMMODE, npx(:), npy(:), iMap(:), ...
    coorX(:), coorY(:), MAXITER, REJRATE_CAS, []);
for ii = 1:MAXITER+1
    % synthesize MV field using estimated GM parameters
    [mmpx, mmpy] = mvGen_f(mm0(ii,:), coorX, coorY);
    % SNR check-up
    wSNR(ii) = mvSNR(px, py, mmpx, mmpy, ones(nH, nW));
end

function [woSNR] = snrGMEPureGD(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW)

% No outlier rejection, all MVs are taken as inliers for plain GD-GME
iMap = ones(size(px));
mm1 = mvGME_NR_test(GMMODE, npx(:), npy(:), iMap(:), ...
    coorX(:), coorY(:), MAXITER, REJRATE_ORI, []);
for ii = 1:MAXITER+1
    % synthesize MV field using estimated GM parameters
    [mmpx, mmpy] = mvGen_f(mm1(ii,:), coorX, coorY);
    % SNR check-up
    woSNR(ii) = mvSNR(px, py, mmpx, mmpy, ones(nH, nW));
end

function [fltSNR] = snrGMEFilter(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW)

% outlier rejection using filter from [6]
iMap = MVRemFlt(npx, npy);
rejRate = nH*nW*REJRATE_ORI/sum(iMap(:));
if rejRate > 1
    rejRate = 1;
end

% check if at least 4 MVs obtained after filtering
if sum(iMap(:)) > 4

    % GME based on all inliers using Newton-Raphson method
    mm0 = mvGME_NR_test(GMMODE, npx(:), npy(:), iMap(:), ...
        coorX(:), coorY(:), MAXITER, rejRate, []);
    for ii = 1:MAXITER+1
        % synthesize MV field using estimated GM parameters
        [mmpx, mmpy] = mvGen_f(mm0(ii,:), coorX, coorY);
        % SNR check-up
        fltSNR(ii) = mvSNR(px, py, mmpx, mmpy, ones(nH, nW));
    end
else
    fltSNR(1:MAXITER+1) = 0;
end
            
function [ranSNR] = snrGMERansac(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW)

% get the reference coordinates
rcoorX = coorX + npx;
rcoorY = coorY + npy;
for ii = 1:MAXITER+1
    [ranM, iter] = ranGME(GMMODE, rcoorX(:), rcoorY(:), coorX(:), coorY(:), ii);
    if isempty(ranM)
        if ii == 1
            ranSNR(ii) = 0;
        else
            ranSNR(ii) = ranSNR(ii-1);
        end
    else
        [mmpx, mmpy] = mvGen_f(ranM, coorX, coorY);% SNR check-up
        rsnr = mvSNR(px, py, mmpx, mmpy, ones(nH, nW));
        if ii == 1
            ranSNR(ii) = max(0, rsnr);
        else
            ranSNR(ii) = max(ranSNR(ii-1), rsnr);
        end
    end
end

function [LSMESNR] = snrGMELSME(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW)


for ii = 1:MAXITER+1
    if ii == 1
        % initializing the parameter
        MM = zeros(1,8);
        MM(1) = 1;
        MM(5) = 1;
        MM(3) = sum(npx(:))/(nH*nW);
        MM(6) = sum(npy(:))/(nH*nW);
    else
        % GME using weighted least square solution
        MM = gmeLsme(coorX, coorY, npx, npy, MM);
    end
    
    % generate MV field using GM parameter estimates
    [mmpx, mmpy] = mvGen_f(MM, coorX, coorY); 
    % SNR check-up
    LSMESNR(ii) = mvSNR(px, py, mmpx, mmpy, ones(nH, nW));
end
