function GME_synthetic_mvf_noise()
clear all
close all


fprintf(1,'****************************************************\n');
fprintf(1,'*      GME performance analysis on synthetic MVF   *\n');
fprintf(1,'****************************************************\n');
fprintf(1,'       Designed by Yue-Meng Chen, Jul,  2009      \n\n');

%% parameters of a synthetic motion field
%
blkSiz = 16; % block size
MVRES = 1;  % integer-pixel: 1, half-pixel: 2
height = 288; nH = height/blkSiz;% CIF resolution
width =352; nW = width/blkSiz;

% get the image coordinates
[coorBlkY,coorBlkX]=ndgrid(1:nH,1:nW);
coorX = coorBlkX.*blkSiz-blkSiz/2;
coorY = coorBlkY.*blkSiz-blkSiz/2;

% four global motion models to be tested!
m(1, :) =   [0.95     0        10.4238    0      0.95      5.7927    0           0];
m(2, :) =   [0.9964   -0.0249  1.0981     0.0856 0.9457    -7.2      0           0];
m(3, :) =   [0.9964   -0.0249  6.0981     0.0249 0.9964    2.5109    -2.7e-5     1.9e-5];
m(4, :) =   [1        0        4.4154     0      1         0         -1.1263e-4  0];

% standard deviation of Gaussian noise
stdDev = [0.7, 1.5, 2.2, 3.0];

% the parameters for the cascade
ITER_CASCADE = 2; % iterations for GME with cascade
ITER_FILTER6 = 2; % iterations for GME with filter from Ref. [6]
ITER_LSS_MS = 4; % iterations for LSS-MS
ITER_GD = 6; % iterations for plain GD-GME 
ITER_RANSAC = 500;
rOUTLIERS = 0.7; % target percentage of outliers in a motion field.
% the threshold of gradient descent for GME with the cascade
REJRATE_CAS = 0.9; 
% the threshold of gradient descent for GME without the cascade
REJRATE_ORI = rOUTLIERS*REJRATE_CAS; 

% the global motion model
GMMODE = 4; % 1: Translational, 2: Isotropic, 3: Affine, 4: Perspective

for GMID = 1:4
    disp(sprintf('GM ID: %d', GMID));
    % generate motion field in floating point
    [px, py] = mvGen_f(m(GMID, :), coorX, coorY);

    for nSigma = 1:4
        disp(sprintf('noise sd: %1.1f', stdDev(nSigma)));
        for runs = 1:1:50

            % Gaussian noise on X/Y channel, and rounding it to integer
            % pixel
            nX = stdDev(nSigma).*randn(nH, nW);
            npx = round(px+nX);
            nY = stdDev(nSigma).*randn(nH, nW);
            npy = round(py+nY);


            %% GME with cascade
            % inlier map initialization
            casGDsnr(runs) = snrGMECascade(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, ITER_CASCADE, REJRATE_CAS, nH, nW); 

            %% GME without cascade
            GDsnr(runs) = snrGMEPureGD(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, ITER_GD, REJRATE_ORI, nH, nW);
            
            %% GME with filter from Ref. [6]
            fltSNR(runs) = snrGMEFilter(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, ITER_FILTER6, REJRATE_ORI, nH, nW);
            
            %% GME with RANSAC
            [ranSNR(runs), ranIter] = snrGMERansac(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, ITER_RANSAC, REJRATE_ORI, nH, nW);
            
            %% GME with LS using M-estimator
            LSMESNR(runs) = snrGMELSME(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, ITER_LSS_MS, REJRATE_ORI, nH, nW);
        end

        % average on 50 runs
        mcasGDsnr(GMID, nSigma) = mean(casGDsnr);
        mGDsnr(GMID, nSigma) = mean(GDsnr);
        mranSNR(GMID, nSigma) = mean(ranSNR);
        mfltSNR(GMID, nSigma) = mean(fltSNR);
        mLSMESNR(GMID, nSigma) = mean(LSMESNR);
    end

end

% print out the results
[mcasGDsnr]
[mGDsnr]
[mfltSNR]
[mranSNR]
[mLSMESNR]
disp('done!')

return;


function [wSNR] = snrGMECascade(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_CAS, nH, nW)

% outlier rejection using the cascade
iMap = MVRemCas(npx, npy, rOUTLIERS, blkSiz);

% GME based on all inliers using Newton-Raphson method
mm0 = mvGME_NR_test(GMMODE, npx(:), npy(:), iMap(:), ...
    coorX(:), coorY(:), MAXITER, REJRATE_CAS,[]);

% synthesize MV field using estimated GM parameters
[mmpx, mmpy] = mvGen_f(mm0(MAXITER+1,:), coorX, coorY);
% SNR check-up
wSNR = mvSNR(px, py, mmpx, mmpy, ones(nH, nW));


function [woSNR] = snrGMEPureGD(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW)

% No outlier rejection, all MVs are taken as inliers for plain GD-GME
iMap = ones(size(px));
mm1 = mvGME_NR_test(GMMODE, npx(:), npy(:), iMap(:), ...
    coorX(:), coorY(:), MAXITER, REJRATE_ORI,[]);

% synthesize MV field using estimated GM parameters
[mmpx, mmpy] = mvGen_f(mm1(MAXITER+1,:), coorX, coorY);
% SNR check-up
woSNR = mvSNR(px, py, mmpx, mmpy, ones(nH, nW));



function [fltSNR] = snrGMEFilter(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW)

% outlier rejection using filter from Ref. [6]
iMap = MVRemFlt(npx, npy);
rejRate = nH*nW*REJRATE_ORI/sum(iMap(:));
if rejRate > 1
    rejRate = 1;
end

% check if at least 4 MVs obtained after filtering
if sum(iMap(:)) > 4

    % GME based on all inliers using Newton-Raphson method
    mm0 = mvGME_NR_test(GMMODE, npx(:), npy(:), iMap(:), ...
        coorX(:), coorY(:), MAXITER, rejRate, []);
    % synthesize MV field using estimated GM parameters
    [mmpx, mmpy] = mvGen_f(mm0(MAXITER+1,:), coorX, coorY);
    % SNR check-up
    fltSNR = mvSNR(px, py, mmpx, mmpy, ones(nH, nW));
else
    fltSNR = 0;
end
      
function [ranSNR, ranIter] = snrGMERansac(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW)


% get the reference coordinates
rcoorX = coorX + npx;
rcoorY = coorY + npy;
[ranM, nIter] = ranGME(GMMODE, rcoorX(:), rcoorY(:), coorX(:), coorY(:), MAXITER);
if isempty(ranM)
    ranSNR = 0;
    ranIter = nIter;
else
    % synthesize MV field using estimated GM parameters
    [mmpx, mmpy] = mvGen_f(ranM, coorX, coorY);
    % SNR check-up
    rsnr = mvSNR(px, py, mmpx, mmpy, ones(nH, nW));
    ranSNR = max(0, rsnr);
    ranIter = nIter;
end

function [LSMESNR] = snrGMELSME(npx, npy, px, py, rOUTLIERS, blkSiz, ...
                GMMODE, coorX, coorY, MAXITER, REJRATE_ORI, nH, nW)


for ii = 1:MAXITER+1
    if ii == 1
        % initializing the parameter
        MM = zeros(1,8);
        MM(1) = 1;
        MM(5) = 1;
        MM(3) = sum(npx(:))/(nH*nW);
        MM(6) = sum(npy(:))/(nH*nW);
    else
        % GME using weighted least square solution
        MM = gmeLsme(coorX, coorY, npx, npy, MM);
    end
end

% synthesize MV field using estimated GM parameters
[mmpx, mmpy] = mvGen_f(MM, coorX, coorY); 
% SNR check-up
LSMESNR = mvSNR(px, py, mmpx, mmpy, ones(nH, nW));
