#include <iostream>
#include <vector>
#include <tuple>
#include <map>
#include <set>

#include "../../midifile/include/MidiFile.h"
#include "lz4.h"
#include "midi.pb.h"

// need this for json out
#include <google/protobuf/util/json_util.h>

#include "encoder.h"
#include "encoder_types.h"

#include "skipgram_bloomfilter.h"

using namespace std;

//cd src/dataset_builder_2; protoc --cpp_out . midi.proto; cd ../..; python3 setup.py install

// installing protobuf
// brew install protobuf ...
// export PKG_CONFIG_PATH=/usr/local/Cellar/protobuf/3.12.2/lib/pkgconfig
// pkg-config --cflags --libs protobuf
// protoc --cpp_out . midi.proto
// protoc --cpp_out src/dataset_builder_2 src/dataset_builder_2/midi.proto

//g++ token_bold.cpp ../../midifile/src/Binasc.cpp ../../midifile/src/MidiEvent.cpp ../../midifile/src/MidiEventList.cpp ../../midifile/src/MidiFile.cpp ../../midifile/src/MidiMessage.cpp midi.pb.cc lz4.c -I../../midifile/include -pthread -I/usr/local/Cellar/protobuf/3.12.2/include -L/usr/local/Cellar/protobuf/3.12.2/lib -lprotobuf --std=c++11; ./a.out

using Event = tuple<int,int,int,int>; // (TIME,VELOCITY,PITCH,INSTRUMENT) 
using EventList = google::protobuf::RepeatedPtrField<midi::Event>;

// how to handle saving large datasets ...

class Jagged {
public:
  Jagged(string filepath_) {
    filepath = filepath_;
    header_filepath = filepath_ + ".header";
    can_write = false;
    can_read = false;
    flush_count = 0;
    num_bars = 4;
    max_tracks = 12;
    max_seq_len = 2048;

    encoder = NULL;
  }

  void set_seed(int seed) {
    srand(seed); // set the seed
  }

  void set_num_bars(int x) {
    num_bars = x;
  }

  void set_max_tracks(int x) {
    max_tracks = x;
  }

  void set_max_seq_len(int x) {
    max_seq_len = x;
  }

  void enable_write() {
    assert(can_read == false);
    if (can_write) { return; }
    // check that the current file is empty unless force flag is present ?
    fs.open(filepath, ios::out | ios::binary);
    can_write = true;
  }
  
  void enable_read() {
    assert(can_write == false);
    if (can_read) { return; }
    fs.open(filepath, ios::in | ios::binary);
    header_fs.open(header_filepath, ios::in | ios::binary);
    header.ParseFromIstream(&header_fs);
    can_read = true;
  }

  void append(string &s, size_t split_id) {
    enable_write();

    size_t start = fs.tellp();
    // begin compress ===============================
    size_t src_size = sizeof(char)*s.size();
    size_t dst_capacity = LZ4_compressBound(src_size);
    char* dst = new char[dst_capacity];
    size_t dst_size = LZ4_compress_default(
      (char*)s.c_str(), dst, src_size, dst_capacity);
    fs.write(dst, dst_size);
    delete[] dst;
    // end compress =================================
    size_t end = fs.tellp();
    midi::Dataset::Item *item;
    switch (split_id) {
      case 0: item = header.add_train(); break;
      case 1: item = header.add_valid(); break;
      case 2: item = header.add_test(); break;
    }
    item->set_start(start);
    item->set_end(end);
    item->set_src_size(src_size);
    flush_count++;

    if (flush_count >= 1000) {
      flush();
      flush_count = 0;
    }
  }

  string read(size_t index, size_t split_id) {
    enable_read();

    midi::Dataset::Item item;
    switch (split_id) {
      case 0: item = header.train(index); break;
      case 1: item = header.valid(index); break;
      case 2: item = header.test(index); break;
    }
    size_t csize = item.end() - item.start();
    char* src = new char[csize/sizeof(char)];
    fs.seekg(item.start());
    fs.read(src, csize);
    string x(item.src_size(), ' ');
    LZ4_decompress_safe(src,(char*)x.c_str(),csize,item.src_size());
    delete[] src;
    return x;
  }

  py::bytes read_bytes(size_t index, size_t split_id) {
    return py::bytes(read(index, split_id));
  }

  string read_json(size_t index, size_t split_id) {
    midi::Piece p;
    string serialized_data = read(index, split_id);
    p.ParseFromString(serialized_data);
    string json_string;
    google::protobuf::util::MessageToJsonString(p, &json_string);
    return json_string;
  }

  // a new function will create batches of sequences really long
  vector<int> encode_for_continue(int split_id, int seq_len, bool LEFT_PAD) {
    
    // wrap this in a while block
    bool no_success = true;
    vector<int> tokens;
    while (no_success) {
      try {

        int index = rand() % get_split_size(split_id);
        midi::Piece p;
        string serialized_data = read(index, split_id);
        p.ParseFromString(serialized_data);

        // pick a random transpose amount
        vector<int> ts;
        for (int t=-6; t<=6; t++) {
          if ((p.min_pitch() + t >= 0) && (p.max_pitch() + t < 128)) {
            ts.push_back(t);
          }
        }

        // encode into a token sequence
        EncoderConfig e;
        e.transpose = ts[rand() % ts.size()];
        e.seed = -1;
        e.do_track_shuffle = true;
        tokens = encoder->encode(&p,&e);

        // pad the tokens (on the LEFT) so they are multiple of seq_len
        // if its equal to zero then we don't need to pad
        if (LEFT_PAD && (tokens.size() % seq_len != 0)) {
          int pad_amount = seq_len - (tokens.size() % seq_len);
          tokens.insert(tokens.begin(), pad_amount, 0);
        }
        no_success = false; // parsing complete
      }
      catch(...) {

      }
    }
    return tokens;
  }

  // fixed length batches with continuation
  tuple<vector<vector<int>>,vector<int>> read_batch_w_continue(int batch_size, int split_id, ENCODER_TYPE et, int seq_len, bool FULL_BATCH) {
    enable_read();

    if (!encoder) {
      encoder = getEncoder(et);
    }

    while (bstore.size() < batch_size) {
      bstore.push_back( vector<int>() );
    }
    
    // create a batch
    vector<int> is_continued(batch_size,0);
    vector<vector<int>> batch(batch_size,vector<int>(seq_len,0));
    for (int i=0; i<batch_size; i++) {

      // add new sequence if necesary
      if (bstore[i].size() > 0) {
        is_continued[i] = 1;
      }
      while (bstore[i].size() == 0) {
        try {
          vector<int> tokens;
          if (!FULL_BATCH) {
            tokens = encode_for_continue(split_id, seq_len, true);
          }
          else {
            while (tokens.size() < seq_len) {
              tokens = encode_for_continue(split_id, seq_len, false);
            }
            assert(tokens.size() >= seq_len);
          }
          copy(tokens.begin(), tokens.end(), back_inserter(bstore[i]));
          is_continued[i] = 0;
        }
        catch(...) {
          cout << "encoding failed ..." << endl;
        }
      }

      // copy first seq_len tokens
      for (int j=0; j<seq_len; j++) {
        batch[i][j] = bstore[i][j];
      }

      // always erase first seq_len tokens
      bstore[i].erase(bstore[i].begin(), bstore[i].begin() + seq_len);

    }
    return make_pair(batch, is_continued);
  }

  tuple<vector<vector<int>>,vector<vector<int>>> read_batch(int batch_size, size_t split_id, ENCODER_TYPE et, int minlen) {
    //srand(time(NULL));
    enable_read();
    int nitems = get_split_size(split_id);
    int index;
    midi::Piece p;
    int maxlen = minlen; // every batch will have atleast this shape
    vector<vector<int>> batch;
    vector<vector<int>> mask;
    ENCODER* enc = getEncoder(et);
    EncoderConfig e;
    while (batch.size() < batch_size) {
      try {
        index = rand() % nitems;
        //cout << "INDEX = " << index << endl;
        string serialized_data = read(index, split_id);
        p.ParseFromString(serialized_data);
        // determine a valid transpose on the range [-6,6]
        vector<int> ts;
        for (int t=-6; t<=6; t++) {
          if ((p.min_pitch() + t >= 0) && (p.max_pitch() + t < 128)) {
            ts.push_back(t);
          }
        }
        int random_transpose = ts[rand() % ts.size()];
        e.transpose = random_transpose;
        e.seed = -1;
        
        // these can be set by the user with functions
        e.num_bars = num_bars;
        e.max_tracks = max_tracks;
        //e.num_bars = p.segment_length(); // get segment length
        // set this to restrict number of tracks
        // make sure the model is trained on shorter combos too
        if (et == TRACK_INST_HEADER_ENCODER) {
          e.max_tracks = rand() % 10 + 2; // random on [2,12]
        }
        if ((et == TRACK_ONE_TWO_THREE_BAR_FILL_ENCODER) || (et == TRACK_BAR_FILL_ENCODER)) {
          e.multi_fill.clear(); // so that it will be randomly init each time
        }
        vector<int> raw_tokens = enc->encode(&p,&e);
        vector<int> tokens;
        if (raw_tokens.size() > max_seq_len) {
          // pick a random section
          int offset = rand() % (raw_tokens.size() - max_seq_len + 1);
          copy(
            raw_tokens.begin() + offset, 
            raw_tokens.begin() + offset + max_seq_len,
            back_inserter(tokens));
        }
        else {
          copy(raw_tokens.begin(), raw_tokens.end(), back_inserter(tokens));
        }
        batch.push_back( tokens );
        mask.push_back( vector<int>(tokens.size(),1) );
        maxlen = max((int)tokens.size(), maxlen);
      }
      catch(...) {
        cout << "encoding failed ..." << endl;
      }
    }
    // right pad the sequences
    for (int i=0; i<batch_size; i++) {
      batch[i].insert(batch[i].end(), maxlen-batch[i].size(), 0);
      mask[i].insert(mask[i].end(), maxlen-mask[i].size(), 0);
      assert(mask[i].size() == maxlen);
      assert(batch[i].size() == maxlen);
    }

    return make_pair(batch,mask);
  }

  int get_size() {
    enable_read();
    return header.train_size() + header.valid_size() + header.test_size();
  }

  int get_split_size(int split_id) {
    enable_read();
    switch (split_id) {
      case 0 : return header.train_size();
      case 1 : return header.valid_size();
      case 2 : return header.test_size();
    }
    return 0; // invalid split id
  }

  void flush() {
    fs.flush();
    header_fs.open(header_filepath, ios::out | ios::binary);
    if (!header.SerializeToOstream(&header_fs)) {
      cerr << "ERROR : Failed to write header file" << endl;
    }
    header_fs.close();
  }

  void close() {
    flush();
    fs.close();
    header_fs.close();
    can_read = false;
    can_write = false;
  }
  
private:
  string filepath;
  string header_filepath;
  fstream fs;
  fstream header_fs;
  bool can_write;
  bool can_read;
  midi::Dataset header;
  int flush_count;

  int num_bars;
  int max_tracks;
  int max_seq_len;

  vector<vector<int>> bstore;
  ENCODER *encoder;
};


// encode midi for dataset
py::bytes encode(string &filepath, EncoderConfig *ec, map<string,vector<string>> &genre_data) {
  string x;
  midi::Piece p;
  parse_new(filepath, &p, ec, &genre_data);
  // filtering out for track dataset
  if (!p.valid_segments_size()) {
    return py::bytes(x); // empty bytes
  }
  p.SerializeToString(&x);
  return py::bytes(x);
}



// HELPER FOR MANIPULATING MIDI::PIECE
// maybe it should just skip if track_num is invalid
void partial_copy(midi::Piece *src, midi::Piece *dst, vector<int> &tracks, int start_bar, int n_bars) {
  midi::Track src_track;
  midi::Track *dst_track;
  midi::Bar src_bar;
  midi::Bar *dst_bar;
  midi::Event src_event;
  midi::Event *dst_event;

  int end_bar = start_bar + n_bars;
  if (n_bars == -1) {
    end_bar = src->tracks(0).bars_size();
  }

  // we can get bar_start by taking the time of the first event
  // and then subtract the qtime
  int event_id = src->tracks(tracks[0]).bars(start_bar).events(0);
  int start_time = src->events(event_id).time() - src->events(event_id).qtime();

  // copy piece metadata
  dst->set_resolution( src->resolution() );
  dst->set_tempo( src->tempo() );
  dst->set_min_pitch( src->min_pitch() ); // this may be wrong now ...
  dst->set_max_pitch( src->max_pitch() ); // this may be wrong now ...
  // valid tracks and valid_segments would need to be recalculated
  dst->add_valid_segments(0);
  dst->add_valid_tracks( (1<<(tracks.size())) - 1 );

  int track_count = 0;
  for (auto const track_num : tracks) {
    if ((track_num < 0) || (track_num >= src->tracks_size())) {
      cout << "ERROR : TRACK " << track_num << " IS OUT OF RANGE!" << endl;
      throw(1);
    }
    src_track = src->tracks(track_num);
    dst_track = dst->add_tracks();
    
    // set the other metadata for each track
    dst_track->set_is_drum( src_track.is_drum() );
    dst_track->set_min_pitch( src_track.min_pitch() );
    dst_track->set_max_pitch( src_track.max_pitch() );
    dst_track->set_instrument( src_track.instrument() );

    for (int bar_num=start_bar; bar_num<end_bar; bar_num++) {
      if ((bar_num < 0) || (bar_num >= src_track.bars_size())) {
        cout << "ERROR : BAR " << bar_num << " IS OUT OF RANGE!" << endl;
        throw(1);
      }
      src_bar = src_track.bars(bar_num);
      dst_bar = dst_track->add_bars();
      
      // set the other metadata for each bar
      dst_bar->set_is_four_four( src_bar.is_four_four() );
      dst_bar->set_has_notes( src_bar.has_notes() );

      for (int i=0; i<src_bar.events_size(); i++) {
        int current_event_count = dst->events_size();
        src_event = src->events(src_bar.events(i));
        dst_event = dst->add_events();
        
        // copy all members of the event
        // make time relative to time of start_bar
        dst_event->set_time( src_event.time() - start_time );
        dst_event->set_velocity( src_event.velocity() );
        dst_event->set_pitch( src_event.pitch() );
        dst_event->set_instrument( src_event.instrument() );
        dst_event->set_track( track_count );
        dst_event->set_qtime( src_event.qtime() );
        dst_event->set_bar( src_event.bar() );
        dst_event->set_is_drum( src_event.is_drum() );

        // add to the bar / track
        dst_bar->add_events( current_event_count );
      }
    }
    track_count++;
  }
}

string partial_copy_json(string &json_string, vector<int> &tracks, int start_bar, int n_bars) {
  string json_output;
  midi::Piece src, dst;
  google::protobuf::util::JsonStringToMessage(json_string.c_str(), &src);
  partial_copy(&src, &dst, tracks, start_bar, n_bars);
  google::protobuf::util::MessageToJsonString(dst, &json_output);
  return json_output;
}


// get valid segments
void find_valid_segments(midi::Piece *p) {
  p->clear_valid_tracks();
  p->clear_valid_segments();

  int seglen = 4;
  for (int i=0; i<p->tracks(0).bars_size()-seglen+1; i++) {
    // check that it is 4/4
    bool is_four_four = true;
    for (int k=0; k<seglen; k++) {
      is_four_four &= (bool)p->tracks(0).bars(i+k).is_four_four();
    }

    // check which tracks are valid
    uint32_t vtracks = 0;
    for (int j=0; j<p->tracks_size(); j++) {
      for (int k=0; k<seglen; k++) {
        if (p->tracks(j).bars(i+k).has_notes()) {
          vtracks |= ((uint32_t)1 << j);
        }
      }
    }

    if ((__builtin_popcount(vtracks) >= 1) && (is_four_four)) {
      p->add_valid_tracks(vtracks);
      p->add_valid_segments(i);
    }
  }
}

/*
void remove_unused_events(midi::Piece *p) {
  set<int> used_events;
  for (const auto track : p->tracks()) {
    for (const auto bar : track.bars()) {
      for (const auto event : )
      used_events.add()
    }
  }
}
*/

// ========================================================================
// ========================================================================

void partial_copy_v2(midi::Piece *src, midi::Piece *dst, vector<int> tracks, vector<int> bars, set<int> *blanks) {
  //cout << "=======================" << endl;
  int event_count = 0;
  dst->CopyFrom(*src);
  dst->clear_tracks();
  dst->clear_events();
  for (const int track_num : tracks) {
    const midi::Track track = src->tracks(track_num);
    midi::Track *t = dst->add_tracks();
    t->CopyFrom(track);
    t->clear_bars();
    if ((!blanks) || (blanks->find(track_num) == blanks->end())) {
      int new_bar_num = 0;
      int start_time = track.bars(bars[0]).time();
      for (const int bar_num : bars) {
        midi::Bar *b = t->add_bars();
        // only copy over if it exists in original
        if (bar_num < track.bars_size()) {
          const midi::Bar bar = track.bars(bar_num);
          b->CopyFrom(bar);
          b->clear_events();
          b->set_time( b->time() - start_time );
          for (const auto event : bar.events()) {
            b->add_events(event_count);
            midi::Event *e = dst->add_events();
            e->CopyFrom(src->events(event));
            e->set_time(e->time() - start_time); // change the time
            e->set_bar(new_bar_num); // reset the bar number
            event_count++;
          }
        }
        new_bar_num++;
      }
    }
  }
}

template<typename T>
vector<T> arange(T start, T stop, T step = 1) {
  vector<T> values;
  for (T value = start; value < stop; value += step)
      values.push_back(value);
  return values;
}

template<typename T>
vector<T> arange(T stop) {
  return arange(0, stop, 1);
}

int get_bar_count(midi::Piece *p) {
  int bar_count = INT_MAX;
  for (const auto track : p->tracks()) {
    bar_count = min(bar_count, track.bars_size());
  }
  return bar_count;
}

string clear_track(string &json_string, int track_num) {
  midi::Piece p;
  google::protobuf::util::JsonStringToMessage(json_string.c_str(), &p);

  // inner
  midi::Piece o;
  vector<int> tracks = arange(p.tracks_size());
  vector<int> bars = arange(get_bar_count(&p));
  set<int> blanks;
  blanks.insert(track_num);
  partial_copy_v2(&p, &o, tracks, bars, &blanks);
  // inner

  string json_output;
  google::protobuf::util::MessageToJsonString(o, &json_output);
  return json_output;
}

string remove_track(string &json_string, int track_num) {
  midi::Piece p;
  google::protobuf::util::JsonStringToMessage(json_string.c_str(), &p);

  // inner
  midi::Piece o;
  vector<int> tracks = arange(p.tracks_size());
  vector<int> bars = arange(get_bar_count(&p));
  tracks.erase(tracks.begin() + track_num);
  partial_copy_v2(&p, &o, tracks, bars, NULL);
  find_valid_segments(&o);
  // inner

  string json_output;
  google::protobuf::util::MessageToJsonString(o, &json_output);
  return json_output;
}

string remove_tracks(string &json_string, vector<int> rtracks) {
  midi::Piece p;
  google::protobuf::util::JsonStringToMessage(json_string.c_str(), &p);

  // inner
  midi::Piece o;
  set<int> h(rtracks.begin(), rtracks.end());
  vector<int> tracks;
  for (int i=0; i<p.tracks_size(); i++) {
    if (h.find(i) == h.end()) {
      tracks.push_back(i);
    }
  }
  vector<int> bars = arange(get_bar_count(&p));
  partial_copy_v2(&p, &o, tracks, bars, NULL);
  find_valid_segments(&o);
  // inner

  string json_output;
  google::protobuf::util::MessageToJsonString(o, &json_output);
  return json_output;
}

string prune_tracks(string &json_string, vector<int> tracks) {
  midi::Piece p;
  google::protobuf::util::JsonStringToMessage(json_string.c_str(), &p);

  // inner
  midi::Piece o;
  vector<int> bars = arange(get_bar_count(&p));
  // remove tracks that don't exist
  vector<int> valid_tracks;
  for (const auto track : tracks) {
    if ((track < p.tracks_size()) && (track >= 0)) {
      valid_tracks.push_back( track );
    }
  }
  partial_copy_v2(&p, &o, valid_tracks, bars, NULL);
  EncoderConfig ec;
  // manually set valid segments
  o.clear_valid_segments();
  o.clear_valid_tracks();
  o.add_valid_segments(0);
  o.add_valid_tracks((1<<o.tracks_size())-1);
  // inner

  string json_output;
  google::protobuf::util::MessageToJsonString(o, &json_output);
  return json_output;
}

string prune_tracks_and_bars(string &json_string, vector<int> tracks, int nbars) {
  midi::Piece p;
  google::protobuf::util::JsonStringToMessage(json_string.c_str(), &p);

  // inner
  midi::Piece o;
  // remove tracks that don't exist
  if (tracks.size() == 0) {
    tracks = arange(p.tracks_size());
  }
  vector<int> valid_bars = arange(nbars);
  vector<int> valid_tracks;
  for (const auto track : tracks) {
    if ((track < p.tracks_size()) && (track >= 0)) {
      if (p.tracks(track).bars_size() >= nbars) {
        valid_tracks.push_back( track );
      }
    }
  }
  
  partial_copy_v2(&p, &o, valid_tracks, valid_bars, NULL);
  EncoderConfig ec;
  // manually set valid segments
  o.clear_valid_segments();
  o.clear_valid_tracks();
  o.add_valid_segments(0);
  o.add_valid_tracks((1<<o.tracks_size())-1);
  // inner

  string json_output;
  google::protobuf::util::MessageToJsonString(o, &json_output);
  return json_output;
}

string update_segments(string &json_string, EncoderConfig *ec) {
  midi::Piece p;
  google::protobuf::util::JsonStringToMessage(json_string.c_str(), &p);

  // inner
  update_valid_segments(&p, ec);
  // inner

  string json_output;
  google::protobuf::util::MessageToJsonString(p, &json_output);
  return json_output;
}

string add_track(string &json_string) {
  midi::Piece p;
  google::protobuf::util::JsonStringToMessage(json_string.c_str(), &p);

  // inner
  midi::Piece o;
  o.CopyFrom(p);
  midi::Track *t = o.add_tracks();
  // inner

  string json_output;
  google::protobuf::util::MessageToJsonString(o, &json_output);
  return json_output;
}

string select_segment(string &json_string, int max_tracks, int index) {
  midi::Piece p;
  google::protobuf::util::JsonStringToMessage(json_string.c_str(), &p);
  string json_output;

  // inner
  if (p.valid_segments_size() == 0) { 
    return json_output; 
  }
  if (index == -1) {
    srand(time(NULL));
    index = rand() % p.valid_segments_size();
  }
  int start_bar = p.valid_segments(index);

  vector<int> vtracks;
  for (int i=0; i<32; i++) {
    if ((p.valid_tracks(index) & (1<<i)) && (vtracks.size()<max_tracks)) {
      vtracks.push_back(i);
    }
  }

  //cout << index << " " << vtracks.size() << endl;

  midi::Piece o; 
  vector<int> bars = arange(start_bar,start_bar+p.segment_length());
  partial_copy_v2(&p, &o, vtracks, bars, NULL);
  //find_valid_segments(&o);
  // manually set valid segments
  o.clear_valid_segments();
  o.clear_valid_tracks();
  o.add_valid_segments(0);
  o.add_valid_tracks((1<<o.tracks_size())-1);
  // inner
  
  google::protobuf::util::MessageToJsonString(o, &json_output);
  return json_output;
}


/*
vector<int> midi_to_tokens(string &filepath, int resolution, ENCODER_TYPE et, int transpose, int seed) {
  midi::Piece p;
  parse_new(filepath, resolution, &p);
  ENCODER *encoder = getEncoder(et);
  return encoder->encode(&p,transpose,seed);
}

// rename to midi to json
string midi_to_json(string &filepath, int resolution, int max_tracks) {

  string json_string;
  midi::Piece p;
  parse_new(filepath, resolution, &p);

  // pick a random segment
  if (max_tracks > 0) {
    if (p.valid_segments_size() == 0) {
      return json_string; // empty string
    }
    midi::Piece segment;
    int segment_idx = rand() % p.valid_segments_size();
    int start_bar = p.valid_segments(segment_idx);
    vector<int> vtracks;
    for (int i=0; i<32; i++) {
      if (p.valid_tracks(segment_idx) & (1<<i)) {
        vtracks.push_back(i);
      }
    }
    vtracks.resize(max_tracks); // only keep the first max tracks
    if (vtracks.size() < max_tracks) {
      return json_string; // empty string
    }
    partial_copy(&p, &segment, vtracks, start_bar, 4);
    google::protobuf::util::MessageToJsonString(segment, &json_string);
    return json_string;
  }

  google::protobuf::util::MessageToJsonString(p, &json_string);
  return json_string;
}

vector<string> encode_to_pretty(string &filepath, int resolution, ENCODER_TYPE et, int transpose, int seed) {
  ENCODER *encoder = getEncoder(et);
  return encoder->rep->pretty(
    midi_to_tokens(filepath, resolution, et, transpose, seed));
}

// rename to tokens_to_midi
void tokens_to_midi(string &filepath, vector<int> &tokens, ENCODER_TYPE et) {
  ENCODER *encoder = getEncoder(et);
  midi::Piece *p = encoder->decode(tokens);
  write_midi(p, filepath);
}

string tokens_to_json(vector<int> &tokens, ENCODER_TYPE et) {
  ENCODER *encoder = getEncoder(et);
  midi::Piece *p = encoder->decode(tokens);
  string json_string;
  google::protobuf::util::MessageToJsonString(*p, &json_string);
  delete p; // dellocate
  return json_string;
}

void json_to_midi(string &filepath, string &json_string) {
  midi::Piece p;
  google::protobuf::util::JsonStringToMessage(json_string.c_str(), &p);
  write_midi(&p, filepath);
}

vector<int> json_to_tokens(string &json_string, ENCODER_TYPE et) {
  ENCODER *encoder = getEncoder(et);
  midi::Piece p;
  google::protobuf::util::JsonStringToMessage(json_string.c_str(), &p);
  return encoder->encode(&p,0,82340);
}


void back_and_forth(string &src_path, string &dest_path, int resolution, ENCODER_TYPE et, int transpose) {
  midi::Piece *p = new midi::Piece;
  cout << "parsing..." << endl;
  parse_new(src_path, resolution, p);
  if (et != NO_ENCODER) {
    ENCODER *encoder = getEncoder(et);
    cout << "encoding ..." << endl;
    vector<int> tokens = encoder->encode(p,transpose,-1);
    cout << "decoding ... (" << tokens.size() << ")" << endl;
    p = encoder->decode(tokens);
  }
  write_midi(p, dest_path);
  delete p;
}


////////////////////////////////////////////////////////////////
// JSON HELPERS
string parse_to_json(string &path) {
  string json_string;
  midi::Piece p;
  parse_new(path, 12, &p);
  google::protobuf::util::MessageToJsonString(p, &json_string);
  return json_string;
}
*/
