/************************************************************************ 
 * 
 *  coder.c, main coding engine of tmn (TMN encoder) 
 * 
 *  Copyright (C) 1997  University of BC, Canada 
 * 
 *  Contacts:  
 *  Michael Gallant                   <mikeg@ee.ubc.ca> 
 *  Guy Cote                          <guyc@ee.ubc.ca> 
 *  Berna Erol                        <bernae@ee.ubc.ca> 
 * 
 *  UBC Image Processing Laboratory   http://www.ee.ubc.ca/image 
 *  2356 Main Mall                    tel.: +1 604 822 4051 
 *  Vancouver BC Canada V6T1Z4        fax.: +1 604 822 5949 
 * 
 ************************************************************************/ 
 
/* 
 * Disclaimer of Warranty 
 * 
 * These software programs are available to the user without any 
 * license fee or royalty on an "as is" basis. The University of 
 * British Columbia disclaims any and all warranties, whether 
 * express, implied, or statuary, including any implied warranties 
 * or merchantability or of fitness for a particular purpose.  In no 
 * event shall the copyright-holder be liable for any incidental, 
 * punitive, or consequential damages of any kind whatsoever arising 
 * from the use of these programs. 
 * 
 * This disclaimer of warranty extends to the user of these programs 
 * and user's customers, employees, agents, transferees, successors, 
 * and assigns. 
 * 
 * The University of British Columbia does not represent or warrant 
 * that the programs furnished hereunder are free of infringement of 
 * any third-party patents. 
 * 
 * Commercial implementations of H.263, including shareware, are 
 * subject to royalty fees to patent holders.  Many of these patents 
 * are general enough such that they are unavoidable regardless of 
 * implementation design. 
 * 
*/ 
 
#include"sim.h" 

 extern int which_dct;
extern int lossless_codec;


/********************************************************************** 
 * 
 *	Name:         CodeOneTrueB 
 *	Description:  code one true B image  
 *	 
 *	Input:        pointer to image, prev_image, prev_recon, Q 
 *         
 *	Returns:      pointer to reconstructed image 
 *	Side effects: memory is allocated to recon image 
 * 
 *	Date: 950930  Author: Michael Gallant <mikeg@ee.ubc.ca> 
 * 
 ***********************************************************************/ 
 
void CodeOneTrueB(PictImage *curr, PictImage *B_image, PictImage *prev,  
          PictImage *pr, int QP, int frameskip, Bits *bits, 
          Pict *pic, PictImage *B_recon, PictImage *recon) 
{ 
  unsigned char *prev_ipol, *next_ipol, *pi, *ni; 
  PictImage *prev_recon=NULL, *next_recon=NULL, *pr_edge = NULL, *nr_edge = NULL; 
  MotionVector *MV[7][MBR+1][MBC+2]; 
  MotionVector ZERO = {0,0,0,0,0}; 
  MB_Structure *recon_data_true_B = NULL; 
  MB_Structure *diff;  
  int *rcoeff, *coeff; 
  int *qcoeff_true_B = NULL; 
  int Mode; 
  int CBP; 
  int newgob, advanced_temporarily_off = DEF_AIC_MODE; 
  int i,j,k; 
  int true_B_prediction_type; 
 
  /* buffer control vars */ 
  float QP_cumulative = (float)0.0; 
  int abs_mb_num = 0, QuantChangePostponed = 0; 
  int QP_new, QP_prev, QP_xmitted=QP; 
 
  MB_Structure *pred = (MB_Structure *)malloc(sizeof(MB_Structure)); 
 
  /* advanced intra coding variables */ 
  int *store_rcoeff, *store_coeff, *pcoeff;  
 
  ZeroBits(bits); 
 
  /* Currently, the MV info is stored in 7 MV structs per MB. MV[0] 
   * stores the 16x16 MV. MV[1] through MV[4] store the 8x8 MVs. MV[6] 
   * stores the PB delta MV or the forward predicted B MV. MV[5] 
   * stores the backward predicted MV for true-B pictures. */  
 
  InitializeMV(MV); 
 
  /* Mark PMV's outside the frame. */ 
  for (i = 1; i < (pels>>4)+1; i++)  
  { 
    for (j = 0; j < 7; j++) 
    { 
      MarkVec(MV[j][0][i]);   
    } 
    MV[0][0][i]->Mode = MODE_INTRA; 
  } 
 
  /* Zero out PMV's outside the frame */ 
  for (i = 0; i < (lines>>4)+1; i++)  
  { 
    for (j = 0; j < 7; j++)  
    { 
      ZeroVec(MV[j][i][0]); 
      ZeroVec(MV[j][i][(pels>>4)+1]); 
    } 
 
    MV[0][i][0]->Mode = MODE_INTRA; 
    MV[0][i][(pels>>4)+1]->Mode = MODE_INTRA; 
  } 
 
  GenerateFrameAndInterpolatedImages(pr, pic, &prev_ipol, &prev_recon, &pi, &pr_edge); 
  GenerateFrameAndInterpolatedImages(recon, pic, &next_ipol, &next_recon, &ni, &nr_edge); 
 
  if (adv_pred) 
  { 
    /* Turn off advanced coding since there can be only 1  
     * motion vector for B frames of IPB. */ 
    advanced_temporarily_off = YES; 
    overlapping_MC = OFF; 
    adv_pred = OFF; 
    use_4mv = OFF; 
  } 
 
  /* If coding a B-picture, need to pass pictures for forward AND 
   * backward prediction, as well as interpolated forward and backward 
   * pictures for half-pel prediction. */ 
  MotionEstimatePicture( B_image->lum,prev_recon->lum, next_recon->lum,          
                         prev_ipol, next_ipol, pic->seek_dist, MV, 
                         pic->use_gobsync, B_PICTURE_ESTIMATION); 
 
  QP_new = QP_xmitted = QP_prev = QP; /* Copy the passed value of QP */ 
 
  for ( j = 0; j < lines/MB_SIZE; j++)  
  { 
    newgob = 0; 
 
    if (j == 0)  
    { 
      if (advanced_intra_coding) 
      { 
         /* store the coeff for the frame */ 
         if ((store_coeff=(int *)calloc(384*(pels/MB_SIZE)*(lines/MB_SIZE), sizeof(int))) == 0)  
         { 
            fprintf(stderr,"coder(): Couldn't allocate store_coeff.\n"); 
            exit(-1); 
         } 
         if ((store_rcoeff=(int *)calloc(384*(pels/MB_SIZE)*(lines/MB_SIZE), sizeof(int))) == 0)  
         { 
            fprintf(stderr,"coder(): Couldn't allocate store_rcoeff.\n"); 
            exit(-1); 
         } 
      } 
 
      pic->QUANT = QP_new; 
 
      bits->header += CountBitsPicture(pic); 
 
      QP_xmitted = QP_prev = QP_new; 
    } 
    else if (pic->use_gobsync && j%pic->use_gobsync == 0)  
    { 
      bits->header += CountBitsGOB(j,QP_new,pic); /* insert gob sync */ 
      QP_xmitted = QP_prev = QP_new; 
      newgob = 1; 
    } 
  
    for ( i = 0; i < pels/MB_SIZE; i++)  
    { 
      pic->MB = i + j * (pels/MB_SIZE); 
       
      /* store the QP for every macroblock */ 
      quant_map[j+1][i+1] = QP_xmitted; 
 
      if ((rcoeff = (int *)malloc(sizeof(int)*384)) == NULL)  
      { 
        fprintf(stderr,"MB_Coder: Could not allocate space for rcoeff\n"); 
        exit(-1); 
      }   
 
      /* Predict true B-MB */ 
      diff = Predict_True_B( B_image, prev_recon, prev_ipol,next_recon,  
                             next_ipol, pred,i*MB_SIZE, j*MB_SIZE, 
                             MV, True_B_Direct_Mode_MV, pic->TRB,  
                             &true_B_prediction_type, &Mode, pic->RTYPE); 
 
      if (B_INTRA_PREDICTION == true_B_prediction_type) 
      { 
        FillLumBlock(i*MB_SIZE, j*MB_SIZE, B_image, diff); 
        FillChromBlock(i*MB_SIZE, j*MB_SIZE, B_image, diff); 
      } 
 
      if ((qcoeff_true_B=(int *)malloc(sizeof(int)*384)) == 0)  
      { 
        fprintf(stderr,"coder(): Couldn't allocate qcoeff_true_B.\n"); 
        exit(-1); 
      } 
      coeff = MB_Encode(diff); 
 
      if (advanced_intra_coding)  
      { 
        if (!(Mode == MODE_INTRA || Mode == MODE_INTRA_Q))  
       	{ 
          for (k=0;k<6;k++)  
          {     
       	     /* store default coeff if non-intra macroblock */ 
	    if (!which_dct) {
	      store_coeff[(i + j * pels/MB_SIZE) * 384 + k * 64] = 1024; 
	      store_rcoeff[(i + j * pels/MB_SIZE) * 384 + k * 64] = 1024; 
	    } else {
	      store_coeff[(i + j * pels/MB_SIZE) * 384 + k * 64] = DEFAULT_DC_BINDCT;
	      store_rcoeff[(i + j * pels/MB_SIZE) * 384 + k * 64] = DEFAULT_DC_BINDCT;
	    }
          } 
          for (k=0;k<6;k++) 
            Quant_blk(coeff,qcoeff_true_B,QP_xmitted, Mode,k); 
          CBP = FindCBP(qcoeff_true_B, Mode, 64); 
          if (CBP == 0 && (Mode == MODE_INTER || Mode == MODE_INTER_Q))  
            ZeroMBlock(diff); 
          else 
          {   
            for (k=0;k<6;k++) 
            { 
              Quant_blk(coeff,qcoeff_true_B,QP_xmitted, Mode,k); 
              Dequant(qcoeff_true_B, rcoeff, QP_xmitted, Mode,k); 
            } 
            MB_Decode(rcoeff, diff); 
          } 
       	} 
       	else  
        { 
       	  if ((pcoeff=(int *)malloc(sizeof(int)*384)) == 0)  
          { 
	          fprintf(stderr,"coder(): Couldn't allocate pcoeff.\n"); 
	          exit(-1); 
          } 
 
          /* store the quantized DCT coefficients */ 
          memcpy( (void *) (store_coeff + (i + j*pels/MB_SIZE)*384), (void *) coeff, sizeof(int) * 384); 
 
          /* Do Intra mode prediction */ 
          pic->Intra_Mode = Choose_Intra_Mode(pcoeff, store_coeff, i, j, newgob); 
 
          for (k=0;k<6;k++)  
          {  
            Intra_AC_DC_Encode(coeff, store_rcoeff, pic->Intra_Mode, i, j, newgob,k); 
            Quant_blk(coeff,pcoeff,QP_xmitted,Mode,k); 
            Dequant(pcoeff, rcoeff, QP_xmitted, Mode,k); 
            Intra_AC_DC_Decode(rcoeff, store_rcoeff, pic->Intra_Mode, i, j, newgob,k); 
          } 
          MB_Decode(rcoeff, diff); 
          CBP = FindCBP(pcoeff,Mode,64); 
	}     
      } 
      else 
      {   
        for (k=0;k<6;k++) 
          Quant_blk(coeff,qcoeff_true_B,QP_xmitted, Mode,k); 
        CBP = FindCBP(qcoeff_true_B, Mode, 64); 
        if (CBP == 0 && (Mode == MODE_INTER || Mode == MODE_INTER_Q))  
          ZeroMBlock(diff); 
        else 
        { 
          for (k=0;k<6;k++)   
            Dequant(qcoeff_true_B, rcoeff, QP_xmitted, Mode,k);  
          MB_Decode(rcoeff, diff); 
        } 
      } 
 
      recon_data_true_B = MB_Recon_True_B( prev_recon, prev_ipol, diff,  
                                           next_recon, next_ipol,  
                                           i*MB_SIZE,j*MB_SIZE,MV, 
                                           True_B_Direct_Mode_MV, pic->TRB, 
                                           true_B_prediction_type, pic->RTYPE); 
      Clip(recon_data_true_B); 
      free(diff); 
      free(coeff); 
 
      if ((CBP==0) && (B_DIRECT_PREDICTION == true_B_prediction_type) && 
          (Mode == MODE_INTER) && (0 == pic->DQUANT) )  
      { 
        coded_map[j+1][i+1] = 0; 
        quant_map[j+1][i+1] = 0; 
 
        CountBitsScalMB(Mode, 1, CBP, 0, pic, bits, true_B_prediction_type, 0); 
      } 
      else  
      { 
        CountBitsScalMB(Mode,0,CBP,0,pic,bits,true_B_prediction_type, 0); 
        if (!CBP) 
            intra_refresh[j+1][i+1] += 1; 
 
        if (MODE_INTER == Mode || MODE_INTER_Q == Mode) 
        { 
	        coded_map[j+1][i+1] = 1; 
	        quant_map[j+1][i+1] = QP_xmitted; 
 
          bits->no_inter++; 
          CountBitsVectors( MV, bits, i, j, Mode, newgob, pic,  
                            true_B_prediction_type); 
        } 
        else  
        { 
          /* MODE_INTRA or MODE_INTRA_Q */ 
          coded_map[j+1][i+1] = 2; 
          quant_map[j+1][i+1] = QP_xmitted; 
          intra_refresh[j+1][i+1] = 0; 
 
          bits->no_intra++; 
        } 
          
        if ( (Mode == MODE_INTRA || Mode == MODE_INTRA_Q) && advanced_intra_coding ) 
        { 
          Scan(pcoeff,pic->Intra_Mode); 
          CountBitsCoeff(pcoeff, Mode, CBP, bits, 64); 
        } 
        else if (CBP || Mode == MODE_INTRA || Mode == MODE_INTRA_Q) 
        { 
          Scan(qcoeff_true_B,0); 
          CountBitsCoeff(qcoeff_true_B, Mode, CBP, bits, 64); 
        } 
      } 
      QP_prev = QP_xmitted; 
     
      abs_mb_num++; 
      QP_cumulative += QP_xmitted;      
#ifdef PRINTQ  
      /* most useful when quantizer changes within a picture */ 
      if (QuantChangePostponed) 
        fprintf(stdout,"@%2d",QP_xmitted); 
      else 
        fprintf(stdout," %2d",QP_xmitted); 
#endif 
 
      ReconImage(i,j,recon_data_true_B,B_recon); 
  
      free(qcoeff_true_B); 
      free(recon_data_true_B); 
 
      if (advanced_intra_coding && (Mode == MODE_INTRA || Mode == MODE_INTRA_Q))   
        free(pcoeff); 
    } 
#ifdef PRINTQ 
    fprintf(stdout,"\n"); 
#endif 
  } 
 
  pic->QP_mean = QP_cumulative/(float)abs_mb_num; 
 
  /* Free memory */ 
  free(pred); 
  free(prev_recon); 
  free(next_recon); 
  FreeImage(pr_edge); 
  FreeImage(nr_edge); 
  free(pi); 
  free(ni); 
   
  for (j = 0; j < (lines>>4)+1; j++) 
    for (i = 0; i < (pels>>4)+2; i++)  
      for (k = 0; k < 7; k++) 
        free(MV[k][j][i]); 
 
  if (advanced_intra_coding) 
  { 
    free(store_coeff); 
    free(store_rcoeff); 
  } 
 
  if (advanced_temporarily_off) 
  { 
    overlapping_MC = ON; 
    adv_pred = ON; 
    use_4mv = ON; 
    advanced_temporarily_off = NO; 
  } 
 
  return; 
} 
 
/*********************************************************************** 
 * 
 *	Name:           Predict_True_B 
 *	Description:    Predicts B macroblock in true B frame prediction 
 * 
 *	Input:	        pointers to current frame, previous recon. previous 
 *                      interpolated, next recon, next interpolated, 
 *                      pos. in image, MV data, Direct Mode MV data, 
 *                      TRB for Direct Mode prediction, and MB prediction type. 
 *	Returns:        pointer to differential MB data after prediction 
 *	Side effects: 
 * 
 *	Date: 970831    Author: Michael Gallant ---mikeg@ee.ubc.ca 
 * 
 ***********************************************************************/ 
 
MB_Structure *Predict_True_B (PictImage * curr_image, PictImage * prev_image, 
                              unsigned char *prev_ipol, PictImage * next_image,  
                              unsigned char *next_ipol, MB_Structure * pred_macroblock,  
                              int x, int y, MotionVector * MV[7][MBR + 1][MBC + 2],  
                              MotionVector * Direct_MV[5][MBR][MBC], 
                              int TRB, int *prediction_type, int *mode, int RTYPE) 
{ 
  int curr[16][16]; 
  MB_Structure *forward_pred = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MB_Structure *backward_pred = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MB_Structure *direct_f_pred = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MB_Structure *direct_b_pred = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MB_Structure *direct_pred = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MB_Structure *bidir_pred = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MB_Structure *pred_error = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MotionVector *forward_MV, *backward_MV, *direct_MV[5]; 
  int SADbackw, SADbidir, SADforw, SADdirect, SADmin; 
  int dx, dy, xmb, ymb, i, j, k; 
  int xvec, yvec, mvx, mvy; 
 
  xmb = x / MB_SIZE + 1; 
  ymb = y / MB_SIZE + 1; 
 
  forward_MV = MV[0][ymb][xmb]; 
  backward_MV = MV[5][ymb][xmb]; 
  direct_MV[0] = Direct_MV[0][ymb - 1][xmb - 1]; 
  direct_MV[1] = Direct_MV[1][ymb - 1][xmb - 1]; 
  direct_MV[2] = Direct_MV[2][ymb - 1][xmb - 1]; 
  direct_MV[3] = Direct_MV[3][ymb - 1][xmb - 1]; 
  direct_MV[4] = Direct_MV[4][ymb - 1][xmb - 1]; 
 
  /* Find MB in current image */ 
  FindMB (x, y, curr_image->lum, curr); 
 
  /* Forward prediction. */ 
  FindPred (x, y, forward_MV, prev_ipol, &forward_pred->lum[0][0], 16, 0); 
 
  /* Backward prediction. */ 
  FindPred (x, y, backward_MV, next_ipol, &backward_pred->lum[0][0], 16, 0); 
 
  /* Direct prediction, forward and backward components. */ 
  if (direct_MV[0]->Mode == MODE_INTER4V || direct_MV[0]->Mode == MODE_INTER4V_Q) 
  { 
    /* Mode INTER4V */ 
    FindForwLumPredDirectTrueB (prev_ipol, x, y, direct_MV[1], &direct_f_pred->lum[0][0], TRB, 8, 0); 
    FindForwLumPredDirectTrueB (prev_ipol, x, y, direct_MV[2], &direct_f_pred->lum[0][8], TRB, 8, 1); 
    FindForwLumPredDirectTrueB (prev_ipol, x, y, direct_MV[3], &direct_f_pred->lum[8][0], TRB, 8, 2); 
    FindForwLumPredDirectTrueB (prev_ipol, x, y, direct_MV[4], &direct_f_pred->lum[8][8], TRB, 8, 3); 
 
    FindBackwLumPredDirectTrueB (next_ipol, x, y, direct_MV[1], &direct_b_pred->lum[0][0], TRB, 8, 0); 
    FindBackwLumPredDirectTrueB (next_ipol, x, y, direct_MV[2], &direct_b_pred->lum[0][8], TRB, 8, 1); 
    FindBackwLumPredDirectTrueB (next_ipol, x, y, direct_MV[3], &direct_b_pred->lum[8][0], TRB, 8, 2); 
    FindBackwLumPredDirectTrueB (next_ipol, x, y, direct_MV[4], &direct_b_pred->lum[8][8], TRB, 8, 3); 
  } 
  else 
  { 
    FindForwLumPredDirectTrueB (prev_ipol, x, y, direct_MV[0], &direct_f_pred->lum[0][0], TRB, 16, 0); 
    FindBackwLumPredDirectTrueB (next_ipol, x, y, direct_MV[0], &direct_b_pred->lum[0][0], TRB, 16, 0); 
  } 
 
  for (j = 0; j < MB_SIZE; j++) 
  { 
    for (i = 0; i < MB_SIZE; i++) 
    { 
      /* Bidirectional prediction. */ 
      bidir_pred->lum[j][i] = (forward_pred->lum[j][i] + 
                               backward_pred->lum[j][i]) / 2; 
 
      /* Direct prediction. */ 
      direct_pred->lum[j][i] = (direct_f_pred->lum[j][i] + 
                                direct_b_pred->lum[j][i]) / 2; 
    } 
  } 
 
  SADforw = SAD_MB_integer (&curr[0][0], &forward_pred->lum[0][0], 16, INT_MAX); 
  SADforw -= 50; 
 
  SADbackw = SAD_MB_integer (&curr[0][0], &backward_pred->lum[0][0], 16, INT_MAX); 
 
  SADdirect = SAD_MB_integer (&curr[0][0], &direct_pred->lum[0][0], 16, INT_MAX); 
  SADdirect -= 100; 
 
  SADbidir = SAD_MB_integer (&curr[0][0], &bidir_pred->lum[0][0], 16, INT_MAX); 
  SADbidir += 75; 
 
  /* Prediction direction decision. */ 
  if ((SADdirect <= SADforw) && (SADdirect <= SADbackw) && 
      (SADdirect <= SADbidir)) 
  { 
    *prediction_type = B_DIRECT_PREDICTION; 
    SADmin = SADdirect; 
  }  
  else if ((SADforw < SADdirect) && (SADforw <= SADbackw) && 
             (SADforw <= SADbidir)) 
  { 
    *prediction_type = B_FORWARD_PREDICTION; 
    SADmin = SADforw; 
  }  
  else if ((SADbackw < SADdirect) && (SADbackw < SADforw) && 
             (SADbackw <= SADbidir)) 
  { 
    *prediction_type = B_BACKWARD_PREDICTION; 
    SADmin = SADbackw; 
  }  
  else 
  { 
    *prediction_type = B_BIDIRECTIONAL_PREDICTION; 
    SADmin = SADbidir; 
  } 
 
  *(mode) = ChooseMode( curr_image->lum, x, y, SADmin); 
                                 
  if (MODE_INTRA == *(mode) || MODE_INTRA_Q == *(mode))  
  { 
    *prediction_type = B_INTRA_PREDICTION; 
  } 
 
  switch (*prediction_type) 
  { 
    case B_FORWARD_PREDICTION: 
 
      /* Translate MV to chrominance-resolution (downsampled). */ 
      dx = 2 * forward_MV->x + forward_MV->x_half; 
      dy = 2 * forward_MV->y + forward_MV->y_half; 
      dx = (dx % 4 == 0 ? dx >> 1 : (dx >> 1) | 1); 
      dy = (dy % 4 == 0 ? dy >> 1 : (dy >> 1) | 1); 
 
      /* Predict B as if P. */ 
      DoPredChrom_P (x, y, dx, dy, curr_image, prev_image, forward_pred, pred_error, RTYPE); 
 
      for (j = 0; j < MB_SIZE; j++) 
      { 
        for (i = 0; i < MB_SIZE; i++) 
        {  
          pred_error->lum[j][i] = *(curr_image->lum + x + i + (y + j) * pels) - forward_pred->lum[j][i]; 
          pred_macroblock->lum[j][i] = forward_pred->lum[j][i]; 
        } 
      } 
      for (j = 0; j < MB_SIZE >> 1; j++) 
      { 
        for (i = 0; i < MB_SIZE >> 1; i++) 
        { 
          pred_macroblock->Cr[j][i] = forward_pred->Cr[j][i]; 
          pred_macroblock->Cb[j][i] = forward_pred->Cb[j][i]; 
        } 
      } 
 
      forward_MV->Mode = MODE_INTER; 
      backward_MV->Mode = MODE_INTRA; 
 
      /* Set backward MV data to 0 for future MV prediction. */ 
      backward_MV->x = backward_MV->x_half = 0; 
      backward_MV->y = backward_MV->y_half = 0; 
     
      break; 
 
    case B_BACKWARD_PREDICTION:     
 
      /* Translate MV to chrominance-resolution (downsampled). */ 
      dx = 2 * backward_MV->x + backward_MV->x_half; 
      dy = 2 * backward_MV->y + backward_MV->y_half; 
      dx = (dx % 4 == 0 ? dx >> 1 : (dx >> 1) | 1); 
      dy = (dy % 4 == 0 ? dy >> 1 : (dy >> 1) | 1); 
 
      /* Predict B as if P. */ 
      DoPredChrom_P (x, y, dx, dy, curr_image, next_image, backward_pred, pred_error, RTYPE); 
 
      for (j = 0; j < MB_SIZE; j++) 
      { 
        for (i = 0; i < MB_SIZE; i++) 
        { 
          pred_error->lum[j][i] = *(curr_image->lum + x + i + (y + j) * pels) - backward_pred->lum[j][i]; 
          pred_macroblock->lum[j][i] = backward_pred->lum[j][i]; 
        } 
      } 
 
      for (j = 0; j < MB_SIZE >> 1; j++) 
      { 
        for (i = 0; i < MB_SIZE >> 1; i++) 
        { 
          pred_macroblock->Cr[j][i] = backward_pred->Cr[j][i]; 
          pred_macroblock->Cb[j][i] = backward_pred->Cb[j][i]; 
        } 
      } 
 
      forward_MV->Mode = MODE_INTRA; 
      backward_MV->Mode = MODE_INTER; 
 
      /* Set forward MV data to 0 for future MV prediction. */ 
      forward_MV->x = forward_MV->x_half = 0; 
      forward_MV->y = forward_MV->y_half = 0; 
 
      break; 
 
    case B_DIRECT_PREDICTION:   
 
      /* Direct prediction, forward and backward components. */ 
      if (direct_MV[0]->Mode == MODE_INTER4V || direct_MV[0]->Mode == MODE_INTER4V_Q) 
      { 
        /* chroma vectors are sum of B luma vectors divided and rounded */ 
        xvec = yvec = 0; 
        for (k = 1; k <= 4; k++) 
        { 
          xvec += TRB * (2 * direct_MV[k]->x + direct_MV[k]->x_half) / TRP; 
          yvec += TRB * (2 * direct_MV[k]->y + direct_MV[k]->y_half) / TRP; 
        } 
 
        /* round values according to TABLE 16/H.263 */ 
        dx = sign (xvec) * (roundtab[abs (xvec) % 16] + (abs (xvec) / 16) * 2); 
        dy = sign (yvec) * (roundtab[abs (yvec) % 16] + (abs (yvec) / 16) * 2); 
 
        /* Predict B as if P. */ 
        FindChromBlock_P (x, y, dx, dy, prev_image, direct_f_pred, RTYPE); 
  
        /* chroma vectors are sum of B luma vectors divided and rounded */ 
        xvec = yvec = 0; 
        for (k = 1; k <= 4; k++) 
        { 
          mvx = 2 * direct_MV[k]->x + direct_MV[k]->x_half; 
          mvy = 2 * direct_MV[k]->y + direct_MV[k]->y_half; 
          xvec += (TRB - TRP) * mvx / TRP; 
          yvec += (TRB - TRP) * mvy / TRP; 
        } 
 
        /* round values according to TABLE 16/H.263 */ 
        dx = sign (xvec) * (roundtab[abs (xvec) % 16] + (abs (xvec) / 16) * 2); 
        dy = sign (yvec) * (roundtab[abs (yvec) % 16] + (abs (yvec) / 16) * 2); 
 
        /* Predict B as if P. */ 
        FindChromBlock_P (x, y, dx, dy, next_image, direct_b_pred,  RTYPE); 
      } 
      else 
      { 
        mvx = 2 * direct_MV[0]->x + direct_MV[0]->x_half; 
        mvy = 2 * direct_MV[0]->y + direct_MV[0]->y_half; 
 
        /* Translate MV to chrominance-resolution (downsampled). */ 
        dx = TRB * mvx / TRP; 
        dy = TRB * mvy / TRP; 
 
        dx = (dx % 4 == 0 ? dx >> 1 : (dx >> 1) | 1); 
        dy = (dy % 4 == 0 ? dy >> 1 : (dy >> 1) | 1); 
 
        /* Predict B as if P. */ 
        FindChromBlock_P (x, y, dx, dy, prev_image, direct_f_pred, RTYPE); 
 
        /* Translate MV to chrominance-resolution (downsampled). */ 
        dx = (TRB - TRP) * mvx / TRP; 
        dy = (TRB - TRP) * mvy / TRP; 
 
        dx = (dx % 4 == 0 ? dx >> 1 : (dx >> 1) | 1); 
        dy = (dy % 4 == 0 ? dy >> 1 : (dy >> 1) | 1); 
 
        /* Predict B as if P. */ 
        FindChromBlock_P (x, y, dx, dy, next_image, direct_b_pred, RTYPE); 
      } 
 
      for (j = 0; j < MB_SIZE; j++) 
      { 
        for (i = 0; i < MB_SIZE; i++) 
        { 
          pred_error->lum[j][i] = *(curr_image->lum + x + i + (y + j) * pels) - 
                                  direct_pred->lum[j][i]; 
          pred_macroblock->lum[j][i] = direct_pred->lum[j][i]; 
        } 
      } 
      for (j = 0; j < MB_SIZE >> 1; j++) 
      { 
        for (i = 0; i < MB_SIZE >> 1; i++) 
        { 
          pred_macroblock->Cr[j][i] = (direct_f_pred->Cr[j][i] + 
                                       direct_b_pred->Cr[j][i]) / 2; 
          pred_macroblock->Cb[j][i] = (direct_f_pred->Cb[j][i] + 
                                       direct_b_pred->Cb[j][i]) / 2; 
          pred_error->Cr[j][i] = *(curr_image->Cr + (x>>1) + i + ((y>>1) + j) * cpels) - 
                                  pred_macroblock->Cr[j][i]; 
          pred_error->Cb[j][i] = *(curr_image->Cb + (x>>1) + i + ((y>>1) + j) * cpels) - 
                                  pred_macroblock->Cb[j][i]; 
        } 
      } 
 
      forward_MV->Mode = MODE_INTER; 
      backward_MV->Mode = MODE_INTER; 
 
      /* Set forward MV data to 0 for future MV prediction. */ 
      forward_MV->x = forward_MV->x_half = 0; 
      forward_MV->y = forward_MV->y_half = 0; 
 
      /* Set backward MV data to 0 for future MV prediction. */ 
      backward_MV->x = backward_MV->x_half = 0; 
      backward_MV->y = backward_MV->y_half = 0; 
 
      break; 
 
    case B_BIDIRECTIONAL_PREDICTION:   
 
      /* Translate MV to chrominance-resolution (downsampled). */ 
      dx = 2 * forward_MV->x + forward_MV->x_half; 
      dy = 2 * forward_MV->y + forward_MV->y_half; 
      dx = (dx % 4 == 0 ? dx >> 1 : (dx >> 1) | 1); 
      dy = (dy % 4 == 0 ? dy >> 1 : (dy >> 1) | 1); 
 
      /* Predict B as if P. */ 
      DoPredChrom_P (x, y, dx, dy, curr_image, prev_image, forward_pred, pred_error, RTYPE); 
 
      /* Translate MV to chrominance-resolution (downsampled). */ 
      dx = 2 * backward_MV->x + backward_MV->x_half; 
      dy = 2 * backward_MV->y + backward_MV->y_half; 
      dx = (dx % 4 == 0 ? dx >> 1 : (dx >> 1) | 1); 
      dy = (dy % 4 == 0 ? dy >> 1 : (dy >> 1) | 1); 
 
      /* Predict B as if P. */ 
      DoPredChrom_P (x, y, dx, dy, curr_image, next_image, backward_pred, pred_error, RTYPE); 
 
      for (j = 0; j < MB_SIZE; j++) 
      { 
        for (i = 0; i < MB_SIZE; i++) 
        { 
          pred_error->lum[j][i] = *(curr_image->lum + x + i + (y + j) * pels) - 
                                  bidir_pred->lum[j][i]; 
          pred_macroblock->lum[j][i] = bidir_pred->lum[j][i]; 
        } 
      } 
      for (j = 0; j < MB_SIZE >> 1; j++) 
      { 
        for (i = 0; i < MB_SIZE >> 1; i++) 
        { 
          pred_macroblock->Cr[j][i] = (forward_pred->Cr[j][i] + 
                                       backward_pred->Cr[j][i]) / 2; 
          pred_macroblock->Cb[j][i] = (forward_pred->Cb[j][i] + 
                                       backward_pred->Cb[j][i]) / 2; 
          pred_error->Cr[j][i] = *(curr_image->Cr + (x>>1) + i + ((y>>1) + j) * cpels) - 
                                  pred_macroblock->Cr[j][i]; 
          pred_error->Cb[j][i] = *(curr_image->Cb + (x>>1) + i + ((y>>1) + j) * cpels) - 
                                  pred_macroblock->Cb[j][i]; 
        } 
      } 
 
      forward_MV->Mode = MODE_INTER; 
      backward_MV->Mode = MODE_INTER; 
 
      break; 
 
    case B_INTRA_PREDICTION: 
     
      /* Set forward MV data to 0 for future MV prediction. */ 
      forward_MV->x = forward_MV->x_half = 0; 
      forward_MV->y = forward_MV->y_half = 0; 
 
      /* Set backward MV data to 0 for future MV prediction. */ 
      backward_MV->x = backward_MV->x_half = 0; 
      backward_MV->y = backward_MV->y_half = 0; 
 
      forward_MV->Mode = MODE_INTRA; 
      backward_MV->Mode = MODE_INTRA; 
 
      break; 
 
    default: 
     
      fprintf (stderr, "Illegal scalable prediction type in MB_Recon_True_B (pred.c)\n"); 
      exit (-1); 
 
  } 
 
  free (backward_pred); 
  free (forward_pred); 
  free (bidir_pred); 
  free (direct_f_pred); 
  free (direct_b_pred); 
  free (direct_pred); 
 
  return pred_error; 
} 
 
/********************************************************************** 
 * 
 *	Name:           MB_Recon_True_B 
 *	Description:	Reconstructs MB after quantization for true B frames 
 * 
 *	Input:	        pointers to decoded residual, previous recon, previous 
 *                      interpolated, next recon, next interpolated, 
 *                      pos. in image, MV data, Direct Mode MV data, 
 *                      TRB for Direct Mode prediction, and MB prediction type. 
 *	Returns:        pointer to reconstructed MB data after motion compensation 
 *	Side effects:   allocates memory to MB_structure 
 * 
 *	Date: 970831	Author: Michael Gallant --- mikeg@ee.ubc.ca 
 * 
 ***********************************************************************/ 
 
MB_Structure *MB_Recon_True_B (PictImage * prev_image, unsigned char *prev_ipol, 
                               MB_Structure *diff, PictImage * next_image, 
                               unsigned char *next_ipol, int x_curr, int y_curr, 
                               MotionVector * MV[7][MBR + 1][MBC + 2], 
                               MotionVector * Direct_MV[5][MBR][MBC], int TRB, 
                               int prediction_type, int RTYPE) 
{ 
  MB_Structure *bidir_forward = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MB_Structure *bidir_backward = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MB_Structure *direct_f_pred = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MB_Structure *direct_b_pred = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MB_Structure *recon_data = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MotionVector *forward_MV, *backward_MV, *direct_MV[5]; 
  int dxf, dyf, dxb, dyb; 
  int i, j, k, xvec, yvec, mvx, mvy; 
 
  direct_MV[0] = Direct_MV[0][y_curr / MB_SIZE][x_curr / MB_SIZE]; 
  direct_MV[1] = Direct_MV[1][y_curr / MB_SIZE][x_curr / MB_SIZE]; 
  direct_MV[2] = Direct_MV[2][y_curr / MB_SIZE][x_curr / MB_SIZE]; 
  direct_MV[3] = Direct_MV[3][y_curr / MB_SIZE][x_curr / MB_SIZE]; 
  direct_MV[4] = Direct_MV[4][y_curr / MB_SIZE][x_curr / MB_SIZE]; 
 
  switch (prediction_type) 
  { 
 
    case B_DIRECT_PREDICTION: 
 
      /* Direct prediction, forward and backward components. */ 
      if (direct_MV[0]->Mode == MODE_INTER4V || direct_MV[0]->Mode == MODE_INTER4V_Q) 
      { 
        /* Mode INTER4V */ 
        FindForwLumPredDirectTrueB (prev_ipol, x_curr, y_curr, direct_MV[1], &direct_f_pred->lum[0][0], TRB, 8, 0); 
        FindForwLumPredDirectTrueB (prev_ipol, x_curr, y_curr, direct_MV[2], &direct_f_pred->lum[0][8], TRB, 8, 1); 
        FindForwLumPredDirectTrueB (prev_ipol, x_curr, y_curr, direct_MV[3], &direct_f_pred->lum[8][0], TRB, 8, 2); 
        FindForwLumPredDirectTrueB (prev_ipol, x_curr, y_curr, direct_MV[4], &direct_f_pred->lum[8][8], TRB, 8, 3); 
 
        FindBackwLumPredDirectTrueB (next_ipol, x_curr, y_curr, direct_MV[1], &direct_b_pred->lum[0][0], TRB, 8, 0); 
        FindBackwLumPredDirectTrueB (next_ipol, x_curr, y_curr, direct_MV[2], &direct_b_pred->lum[0][8], TRB, 8, 1); 
        FindBackwLumPredDirectTrueB (next_ipol, x_curr, y_curr, direct_MV[3], &direct_b_pred->lum[8][0], TRB, 8, 2); 
        FindBackwLumPredDirectTrueB (next_ipol, x_curr, y_curr, direct_MV[4], &direct_b_pred->lum[8][8], TRB, 8, 3); 
 
        /* chroma vectors are sum of B luma vectors divided and rounded */ 
        xvec = yvec = 0; 
        for (k = 1; k <= 4; k++) 
        { 
          xvec += TRB * (2 * direct_MV[k]->x + direct_MV[k]->x_half) / TRP; 
          yvec += TRB * (2 * direct_MV[k]->y + direct_MV[k]->y_half) / TRP; 
        } 
 
        /* round values according to TABLE 16/H.263 */ 
        dxf = sign (xvec) * (roundtab[abs (xvec) % 16] + (abs (xvec) / 16) * 2); 
        dyf = sign (yvec) * (roundtab[abs (yvec) % 16] + (abs (yvec) / 16) * 2); 
 
        /* Predict B as if P. */ 
        FindChromBlock_P (x_curr, y_curr, dxf, dyf, prev_image, direct_f_pred, RTYPE); 
  
        /* chroma vectors are sum of B luma vectors divided and rounded */ 
        xvec = yvec = 0; 
        for (k = 1; k <= 4; k++) 
        { 
          mvx = 2 * direct_MV[k]->x + direct_MV[k]->x_half; 
          mvy = 2 * direct_MV[k]->y + direct_MV[k]->y_half; 
          xvec += (TRB - TRP) * mvx / TRP; 
          yvec += (TRB - TRP) * mvy / TRP; 
        } 
 
        /* round values according to TABLE 16/H.263 */ 
        dxb = sign (xvec) * (roundtab[abs (xvec) % 16] + (abs (xvec) / 16) * 2); 
        dyb = sign (yvec) * (roundtab[abs (yvec) % 16] + (abs (yvec) / 16) * 2); 
 
        /* Predict B as if P. */ 
        FindChromBlock_P (x_curr, y_curr, dxb, dyb, next_image, direct_b_pred, RTYPE); 
      } 
      else 
      { 
        FindForwLumPredDirectTrueB (prev_ipol, x_curr, y_curr, direct_MV[0], &direct_f_pred->lum[0][0], TRB, 16, 0); 
        FindBackwLumPredDirectTrueB (next_ipol, x_curr, y_curr, direct_MV[0], &direct_b_pred->lum[0][0], TRB, 16, 0); 
 
        mvx = 2 * direct_MV[0]->x + direct_MV[0]->x_half; 
        mvy = 2 * direct_MV[0]->y + direct_MV[0]->y_half; 
 
        /* Translate MV to chrominance-resolution (downsampled). */ 
        dxf = TRB * mvx / TRP; 
        dyf = TRB * mvy / TRP; 
 
        dxf = (dxf % 4 == 0 ? dxf >> 1 : (dxf >> 1) | 1); 
        dyf = (dyf % 4 == 0 ? dyf >> 1 : (dyf >> 1) | 1); 
 
        /* Predict B as if P. */ 
        FindChromBlock_P (x_curr, y_curr, dxf, dyf, prev_image, direct_f_pred, RTYPE); 
 
        /* Translate MV to chrominance-resolution (downsampled). */ 
        dxb = (TRB - TRP) * mvx / TRP; 
        dyb = (TRB - TRP) * mvy / TRP; 
 
        dxb = (dxb % 4 == 0 ? dxb >> 1 : (dxb >> 1) | 1); 
        dyb = (dyb % 4 == 0 ? dyb >> 1 : (dyb >> 1) | 1); 
 
        /* Predict B as if P. */ 
        FindChromBlock_P (x_curr, y_curr, dxb, dyb, next_image, direct_b_pred, RTYPE); 
      } 
 
      for (j = 0; j < MB_SIZE; j++) 
      { 
        for (i = 0; i < MB_SIZE; i++) 
        { 
          diff->lum[j][i] += (direct_f_pred->lum[j][i] + 
                              direct_b_pred->lum[j][i]) / 2; 
        } 
      } 
 
      for (j = 0; j < MB_SIZE >> 1; j++) 
      { 
        for (i = 0; i < MB_SIZE >> 1; i++) 
        { 
          diff->Cr[j][i] += (direct_f_pred->Cr[j][i] + 
                             direct_b_pred->Cr[j][i]) / 2; 
          diff->Cb[j][i] += (direct_f_pred->Cb[j][i] + 
                             direct_b_pred->Cb[j][i]) / 2; 
        } 
      } 
 
      break; 
 
    case B_FORWARD_PREDICTION: 
 
      forward_MV = MV[0][y_curr / MB_SIZE + 1][x_curr / MB_SIZE + 1]; 
 
      if (forward_MV->Mode == MODE_INTER || forward_MV->Mode == MODE_INTER_Q) 
      { 
        /* Inter 16x16 */ 
        ReconLumBlock_P (x_curr, y_curr, forward_MV, prev_ipol, &diff->lum[0][0], 16, 0); 
        dxf = 2 * forward_MV->x + forward_MV->x_half; 
        dyf = 2 * forward_MV->y + forward_MV->y_half; 
        dxf = (dxf % 4 == 0 ? dxf >> 1 : (dxf >> 1) | 1); 
        dyf = (dyf % 4 == 0 ? dyf >> 1 : (dyf >> 1) | 1); 
        ReconChromBlock_P (x_curr, y_curr, dxf, dyf, prev_image, diff, RTYPE); 
      } 
      break; 
 
    case B_BACKWARD_PREDICTION: 
 
      backward_MV = MV[5][y_curr / MB_SIZE + 1][x_curr / MB_SIZE + 1]; 
 
      if (backward_MV->Mode == MODE_INTER || backward_MV->Mode == MODE_INTER_Q) 
      { 
        /* Inter 16x16 */ 
        ReconLumBlock_P (x_curr, y_curr, backward_MV, next_ipol, &diff->lum[0][0], 16, 0); 
        dxb = 2 * backward_MV->x + backward_MV->x_half; 
        dyb = 2 * backward_MV->y + backward_MV->y_half; 
        dxb = (dxb % 4 == 0 ? dxb >> 1 : (dxb >> 1) | 1); 
        dyb = (dyb % 4 == 0 ? dyb >> 1 : (dyb >> 1) | 1); 
        ReconChromBlock_P (x_curr, y_curr, dxb, dyb, next_image, diff, RTYPE); 
      } 
      break; 
 
    case B_BIDIRECTIONAL_PREDICTION: 
 
      forward_MV = MV[0][y_curr / MB_SIZE + 1][x_curr / MB_SIZE + 1]; 
      backward_MV = MV[5][y_curr / MB_SIZE + 1][x_curr / MB_SIZE + 1]; 
 
      if ((forward_MV->Mode == MODE_INTER || forward_MV->Mode == MODE_INTER_Q) || 
          (backward_MV->Mode == MODE_INTER || backward_MV->Mode == MODE_INTER_Q)) 
      { 
        /* Forward prediction. */ 
        FindPred (x_curr, y_curr, forward_MV, prev_ipol, &bidir_forward->lum[0][0], 16, 0); 
 
        /* Backward prediction. */ 
        FindPred (x_curr, y_curr, backward_MV, next_ipol, &bidir_backward->lum[0][0], 16, 0); 
 
        for (j = 0; j < MB_SIZE; j++) 
        { 
          for (i = 0; i < MB_SIZE; i++) 
          { 
            diff->lum[j][i] += (bidir_forward->lum[j][i] + 
                                bidir_backward->lum[j][i]) / 2; 
          } 
        } 
 
        dxf = 2 * forward_MV->x + forward_MV->x_half; 
        dyf = 2 * forward_MV->y + forward_MV->y_half; 
        dxf = (dxf % 4 == 0 ? dxf >> 1 : (dxf >> 1) | 1); 
        dyf = (dyf % 4 == 0 ? dyf >> 1 : (dyf >> 1) | 1); 
 
        dxb = 2 * backward_MV->x + backward_MV->x_half; 
        dyb = 2 * backward_MV->y + backward_MV->y_half; 
        dxb = (dxb % 4 == 0 ? dxb >> 1 : (dxb >> 1) | 1); 
        dyb = (dyb % 4 == 0 ? dyb >> 1 : (dyb >> 1) | 1); 
 
        FindChromBlock_P (x_curr, y_curr, dxf, dyf, prev_image, bidir_forward, RTYPE); 
 
        FindChromBlock_P (x_curr, y_curr, dxb, dyb, next_image, bidir_backward, RTYPE); 
 
        for (j = 0; j < MB_SIZE / 2; j++) 
        { 
          for (i = 0; i < MB_SIZE / 2; i++) 
          { 
            diff->Cr[j][i] += (bidir_forward->Cr[j][i] + 
                               bidir_backward->Cr[j][i]) / 2; 
            diff->Cb[j][i] += (bidir_forward->Cb[j][i] + 
                               bidir_backward->Cb[j][i]) / 2; 
          } 
        } 
      } 
      break; 
 
    case B_INTRA_PREDICTION: 
     
      break; 
 
    default: 
 
      fprintf (stderr, "Illegal scalable prediction type in MB_Recon_True_B (pred.c)\n"); 
      exit (-1); 
      break; 
 
  } 
 
  memcpy (recon_data, diff, sizeof (MB_Structure)); 
 
  free (bidir_forward); 
  free (bidir_backward); 
  free (direct_f_pred); 
  free (direct_b_pred); 
 
  return recon_data; 
} 
 
/********************************************************************** 
 * 
 *	Name:	       FindForwLumPredDirectTrueB 
 *	Description:   Finds the forward luma  prediction in true B frame 
 *                     pred. 
 * 
 *	Input:	       pointer to prev. ipol frame, current positon, 
 *                     MV structure and pred. structure to fill 
 * 
 *	Date: 970831   Author: Michael Gallant --- mikeg@ee.ubc.ca 
 * 
 ***********************************************************************/ 
 
void FindForwLumPredDirectTrueB (unsigned char *prev_ipol, int x_curr, int y_curr, 
                                  MotionVector * fr, int *pred, int TRB, int bs, int comp) 
{ 
  int i, j; 
  int xvec, yvec, lx; 
 
  lx = (mv_outside_frame ? pels + (long_vectors ? 64 : 32) : pels); 
 
  /* Luma */ 
  xvec = (TRB) * (2 * fr->x + fr->x_half) / TRP; 
  yvec = (TRB) * (2 * fr->y + fr->y_half) / TRP; 
 
  x_curr += ((comp & 1) << 3); 
  y_curr += ((comp & 2) << 2); 
 
  for (j = 0; j < bs; j++) 
  { 
    for (i = 0; i < bs; i++) 
    { 
      *(pred + i + j * 16) = *(prev_ipol + (i + x_curr) * 2 + xvec + 
                               ((j + y_curr) * 2 + yvec) * lx * 2); 
    } 
  } 
 
  return; 
} 
 
/********************************************************************** 
 * 
 *	Name:	       FindBackwLumPredDirectTrueB 
 *	Description:   Finds the backward luma  prediction in true B frame 
 *                     pred. 
 * 
 *	Input:	       pointer to next ipol frame, current positon, 
 *                     MV structure and pred. structure to fill 
 * 
 *	Date: 970831   Author: Michael Gallant --- mikeg@ee.ubc.ca 
 * 
 ***********************************************************************/ 
 
void FindBackwLumPredDirectTrueB (unsigned char *next_ipol, int x_curr, int y_curr, 
                                   MotionVector * fr, int *pred, int TRB, int bs, int comp) 
{ 
  int i, j; 
  int xvec, yvec, mvx, mvy, lx; 
 
  lx = (mv_outside_frame ? pels + (long_vectors ? 64 : 32) : pels); 
 
  /* Luma */ 
  mvx = 2 * fr->x + fr->x_half; 
  mvy = 2 * fr->y + fr->y_half; 
 
  xvec = (TRB - TRP) * mvx / TRP; 
  yvec = (TRB - TRP) * mvy / TRP; 
 
   
  x_curr += ((comp & 1) << 3); 
  y_curr += ((comp & 2) << 2); 
 
  for (j = 0; j < bs; j++) 
  { 
    for (i = 0; i < bs; i++) 
    { 
      *(pred + i + j * 16) = *(next_ipol + (i + x_curr) * 2 + xvec + 
                               ((j + y_curr) * 2 + yvec) * lx * 2); 
    } 
  } 
 
  return; 
} 
