/************************************************************************ 
 * 
 *  coder.c, main coding engine of tmn (TMN encoder) 
 * 
 *  Copyright (C) 1997  University of BC, Canada 
 * 
 *  Contacts:  
 *  Michael Gallant                   <mikeg@ee.ubc.ca> 
 *  Guy Cote                          <guyc@ee.ubc.ca> 
 *  Berna Erol                        <bernae@ee.ubc.ca> 
 * 
 *  UBC Image Processing Laboratory   http://www.ee.ubc.ca/image 
 *  2356 Main Mall                    tel.: +1 604 822 4051 
 *  Vancouver BC Canada V6T1Z4        fax.: +1 604 822 5949 
 * 
 *  Copyright (C) 1995, 1996  Telenor R&D, Norway 
 *   
 *  Contacts:  
 *  Robert Danielsen                  <Robert.Danielsen@nta.no> 
 * 
 *  Telenor Research and Development  http://www.nta.no/brukere/DVC/ 
 *  P.O.Box 83                        tel.:   +47 63 84 84 00 
 *  N-2007 Kjeller, Norway            fax.:   +47 63 81 00 76 
 *   
 ************************************************************************/ 
 
/* 
 * Disclaimer of Warranty 
 * 
 * These software programs are available to the user without any 
 * license fee or royalty on an "as is" basis. The University of 
 * British Columbia disclaims any and all warranties, whether 
 * express, implied, or statuary, including any implied warranties 
 * or merchantability or of fitness for a particular purpose.  In no 
 * event shall the copyright-holder be liable for any incidental, 
 * punitive, or consequential damages of any kind whatsoever arising 
 * from the use of these programs. 
 * 
 * This disclaimer of warranty extends to the user of these programs 
 * and user's customers, employees, agents, transferees, successors, 
 * and assigns. 
 * 
 * The University of British Columbia does not represent or warrant 
 * that the programs furnished hereunder are free of infringement of 
 * any third-party patents. 
 * 
 * Commercial implementations of H.263, including shareware, are 
 * subject to royalty fees to patent holders.  Many of these patents 
 * are general enough such that they are unavoidable regardless of 
 * implementation design. 
 * 
*/ 
 
 
/***************************************************************** 
 * 
 * Modified by Pat Mulroy, BT Labs to run syntax based arithmetic 
 * coding.  SAC option, H.263 (Annex E). 
 * 
 *****************************************************************/ 

/**************************************************
 * Modification made by Jie Liang for binDCT:
 * 10/04/2000    Added code to call binDCT.
 *
 *
 **************************************************
 */ 
 
#include"sim.h" 

//declare global variable for bindct type. Defined in main.c
extern int which_dct;
 
/********************************************************************** 
 * 
 *	Name:        MB_Encode 
 *	Description:	DCT  of Macroblocks 
 * 
 *	Input:        MB data struct 
 *	Returns:	Pointer to coefficients  
 *	Side effects:	 
 * 
 *	Date: 930128	Author: Robert.Danielsen@nta.no 
 *      Date: 970915    Author: guyc@ee.ubc.ca 
 *                              Quantization done separatly 
 * 
 **********************************************************************/ 
int *MB_Encode(MB_Structure *mb_orig) 
{ 
  int        i, j, k, l, row, col; 
  int        fblock[64]; 
  int        *coeff; 
  int        *coeff_ind; 
 
  if ((coeff = (int *)malloc(sizeof(int)*384)) == NULL) { 
    fprintf(stderr,"MB_Encode: Could not allocate space for coeff\n"); 
    exit(-1); 
  }  
  coeff_ind = coeff; 
  for (k=0;k<16;k+=8) { 
    for (l=0;l<16;l+=8) { 
      for (i=k,row=0;row<64;i++,row+=8) { 
        for (j=l,col=0;col<8;j++,col++) { 
          *(fblock+row+col) = mb_orig->lum[i][j]; 
        } 
      } 
      
      switch (which_dct) {
        case 0:
		  Dct(fblock,coeff_ind); 
		  break;
	  case 1:  //C1
		fdct_bin_c1(fblock,coeff_ind); 
		break;
	  case 4:  //C4
		fdct_bin_c4(fblock,coeff_ind); 
		break;
	  case 7:  //C7
		fdct_bin_c7(fblock,coeff_ind); 
		break;
	  case 13: //L3
		fdct_bin_l3(fblock,coeff_ind); 
		break;
	  default:
		break;
      }

      coeff_ind += 64; 
    } 
  } 
 
  for (i=0;i<8;i++) { 
    for (j=0;j<8;j++) { 
      *(fblock+i*8+j) = mb_orig->Cb[i][j]; 
    } 
  } 

  switch (which_dct) {
  case 0:
	Dct(fblock,coeff_ind); 
	break;
  case 1:  //C1
	fdct_bin_c1(fblock,coeff_ind); 
	break;
  case 4:  //C4
	fdct_bin_c4(fblock,coeff_ind); 
	break;
  case 7:  //C7
	fdct_bin_c7(fblock,coeff_ind); 
	break;
  case 13: //L3
	fdct_bin_l3(fblock,coeff_ind); 
	break;
  default:
	break;
  }

  coeff_ind += 64; 
  for (i=0;i<8;i++) { 
    for (j=0;j<8;j++) { 
      *(fblock+i*8+j) = mb_orig->Cr[i][j]; 
    } 
  } 

  switch (which_dct) {
  case 0:
	Dct(fblock,coeff_ind); 
	break;
  case 1:  //C1
	fdct_bin_c1(fblock,coeff_ind); 
	break;
  case 4:  //C4
	fdct_bin_c4(fblock,coeff_ind); 
	break;
  case 7:  //C7
	fdct_bin_c7(fblock,coeff_ind); 
	break;
  case 13: //L3
	fdct_bin_l3(fblock,coeff_ind); 
	break;
  default:
	break;
  }
   
  return coeff; 
} 
 
/********************************************************************** 
 * 
 *	Name:        MB_Decode 
 *	Description:	Reconstruction of quantized DCT-coded Macroblocks 
 * 
 *	Input:        Quantized coefficients, MB data 
 *        QP (1-31, 0 = no quant), MB info block 
 *	Returns:	int (just 0) 
 *	Side effects:	 
 * 
 *	Date: 930128	Author: Robert.Danielsen@nta.no 
 *      Date: 970915    Author: guyc@ee.ubc.ca 
 *                              Deqantization done separatly  
 **********************************************************************/ 
int MB_Decode(int *rcoeff, MB_Structure *mb_recon) 
{ 
  int	i, j, k, l, row, col; 
  int	*iblock; 
  int	*rcoeff_ind; 
 
  if ((iblock = (int *)malloc(sizeof(int)*64)) == NULL) { 
    fprintf(stderr,"MB_Coder: Could not allocate space for iblock\n"); 
    exit(-1); 
  } 
 
  /* For control purposes */ 
  /* Zero data */ 
  for (i = 0; i < 16; i++) 
    for (j = 0; j < 16; j++) 
      mb_recon->lum[j][i] = 0; 
  for (i = 0; i < 8; i++)  
    for (j = 0; j < 8; j++) { 
      mb_recon->Cb[j][i] = 0; 
      mb_recon->Cr[j][i] = 0; 
    } 
 
  rcoeff_ind = rcoeff; 
 
  for (k=0;k<16;k+=8) { 
    for (l=0;l<16;l+=8) { 
#ifndef FASTIDCT 
      idctref(rcoeff_ind,iblock);  
#else 
      //idct(rcoeff_ind,iblock);  // original

      switch (which_dct) {
      case 0:
		idct(rcoeff_ind, iblock); 
		break;
      case 1:
		idct_bin_c1(rcoeff_ind, iblock); 
		break;
      case 4:
		idct_bin_c4(rcoeff_ind, iblock); 
		break;
      case 7:
		idct_bin_c7(rcoeff_ind, iblock); 
		break;
      case 13:
		idct_bin_l3(rcoeff_ind, iblock); 
		break;
      default:
		break;
      }

#endif 
      rcoeff_ind += 64; 
      for (i=k,row=0;row<64;i++,row+=8) { 
        for (j=l,col=0;col<8;j++,col++) { 
          mb_recon->lum[i][j] = *(iblock+row+col); 
        } 
      } 
    } 
  } 
 
#ifndef FASTIDCT 
  idctref(rcoeff_ind,iblock);  
#else 
  //idct(rcoeff_ind,iblock);  // original
      switch (which_dct) {
      case 0:
		idct(rcoeff_ind, iblock); 
		break;
      case 1:
		idct_bin_c1(rcoeff_ind, iblock); 
		break;
      case 4:
		idct_bin_c4(rcoeff_ind, iblock); 
		break;
      case 7:
		idct_bin_c7(rcoeff_ind, iblock); 
		break;
      case 13:
		idct_bin_l3(rcoeff_ind, iblock); 
		break;
      default:
		break;
      }
#endif 
  rcoeff_ind += 64; 
  for (i=0;i<8;i++) { 
    for (j=0;j<8;j++) { 
      mb_recon->Cb[i][j] = *(iblock+i*8+j); 
    } 
  } 
 
/*  printf("Cb.\n"); 
  for (i=0;i<8;i++)  
  { 
    printf("%d %d %d %d %d %d %d %d\n", mb_recon->Cb[i][0], 
           mb_recon->Cb[i][1], mb_recon->Cb[i][2], mb_recon->Cb[i][3], 
           mb_recon->Cb[i][4], mb_recon->Cb[i][5], mb_recon->Cb[i][6], 
           mb_recon->Cb[i][7]); 
  } 
  printf("\n"); 
*/ 
#ifndef FASTIDCT 
  idctref(rcoeff_ind,iblock);  
#else 
  //idct(rcoeff_ind,iblock);  // original
       switch (which_dct) {
      case 0:
		idct(rcoeff_ind, iblock); 
		break;
      case 1:
		idct_bin_c1(rcoeff_ind, iblock); 
		break;
      case 4:
		idct_bin_c4(rcoeff_ind, iblock); 
		break;
      case 7:
		idct_bin_c7(rcoeff_ind, iblock); 
		break;
      case 13:
		idct_bin_l3(rcoeff_ind, iblock); 
		break;
      default:
		break;
      }
#endif 
  for (i=0;i<8;i++) { 
    for (j=0;j<8;j++) { 
      mb_recon->Cr[i][j] = *(iblock+i*8+j); 
    } 
  } 
  free(iblock); 
  return 0; 
} 
 
 
/********************************************************************** 
 * 
 *	Name:        FillLumBlock 
 *	Description:        Fills the luminance of one block of PictImage 
 *	 
 *	Input:        Position, pointer to PictImage, array to fill 
 *	Returns:         
 *	Side effects:	fills array 
 * 
 *	Date: 930129	Author: Karl.Lillevold@nta.no 
 * 
 ***********************************************************************/ 
 
void FillLumBlock( int x, int y, PictImage *image, MB_Structure *data) 
{ 
  int n; 
  register int m; 
 
  for (n = 0; n < MB_SIZE; n++) 
    for (m = 0; m < MB_SIZE; m++) 
      data->lum[n][m] =  
        (int)(*(image->lum + x+m + (y+n)*pels)); 
  return; 
} 
 
/********************************************************************** 
 * 
 *	Name:        FillChromBlock 
 *	Description:        Fills the chrominance of one block of PictImage 
 *	 
 *	Input:        Position, pointer to PictImage, array to fill 
 *	Returns:         
 *	Side effects:	fills array 
 *                      128 subtracted from each 
 * 
 *	Date: 930129	Author: Karl.Lillevold@nta.no 
 * 
 ***********************************************************************/ 
 
void FillChromBlock(int x_curr, int y_curr, PictImage *image, 
            MB_Structure *data) 
{ 
  int n; 
  register int m; 
 
  int x, y; 
 
  x = x_curr>>1; 
  y = y_curr>>1; 
 
  for (n = 0; n < (MB_SIZE>>1); n++) 
    for (m = 0; m < (MB_SIZE>>1); m++) { 
      data->Cr[n][m] =  
        (int)(*(image->Cr +x+m + (y+n)*cpels)); 
      data->Cb[n][m] =  
        (int)(*(image->Cb +x+m + (y+n)*cpels)); 
    } 
  return; 
} 
 
 
/********************************************************************** 
 * 
 *	Name:        ZeroMBlock 
 *	Description:        Fills one MB with Zeros 
 *	 
 *	Input:        MB_Structure to zero out 
 *	Returns:         
 *	Side effects:	 
 * 
 *	Date: 940829	Author: Karl.Lillevold@nta.no 
 * 
 ***********************************************************************/ 
 
void ZeroMBlock(MB_Structure *data) 
{ 
  int n; 
  register int m; 
 
  for (n = 0; n < MB_SIZE; n++) 
    for (m = 0; m < MB_SIZE; m++) 
      data->lum[n][m] = 0; 
  for (n = 0; n < (MB_SIZE>>1); n++) 
    for (m = 0; m < (MB_SIZE>>1); m++) { 
      data->Cr[n][m] = 0; 
      data->Cb[n][m] = 0; 
    } 
  return; 
} 
 
/********************************************************************** 
 * 
 *	Name:        Clip 
 *	Description:    clips recontructed data 0-255 
 *	 
 *	Input:	        pointer to recon. data structure 
 *	Side effects:   data structure clipped 
 * 
 *	Date: 950718        Author: Karl.Lillevold@nta.no 
 * 
 ***********************************************************************/ 
 
void Clip(MB_Structure *data) 
{ 
  int m,n; 
 
  for (n = 0; n < 16; n++) { 
    for (m = 0; m < 16; m++) { 
      data->lum[n][m] = mmin(255,mmax(0,data->lum[n][m])); 
    } 
  } 
  for (n = 0; n < 8; n++) { 
    for (m = 0; m < 8; m++) { 
      data->Cr[n][m] = mmin(255,mmax(0,data->Cr[n][m])); 
      data->Cb[n][m] = mmin(255,mmax(0,data->Cb[n][m])); 
    } 
  } 
} 
 
/********************************************************************** 
 * 
 *	Name:        LoadArea 
 *	Description:    fills array with a square of image-data 
 * 
 *	Input:	       pointer to image and position, x and y size 
 *	Returns:       pointer to area 
 *	Side effects:  memory allocated to array 
 * 
 *	Date: 940203	Author: PGB 
 *                      Mod: KOL 
 * 
 ***********************************************************************/ 
 
 
unsigned char *LoadArea (unsigned char *im, int x, int y, 
                          int x_size, int y_size, int lx) 
{ 
  unsigned char *res = (unsigned char *) malloc (sizeof (char) * x_size * y_size); 
  unsigned char *in; 
  unsigned char *out; 
  int i = x_size; 
  int j = y_size; 
 
  in = im + (y * lx) + x; 
  out = res; 
 
  while (j--) 
  { 
    while (i--) 
      *out++ = *in++; 
    i = x_size; 
    in += lx - x_size; 
  }; 
  return res; 
} 
 
/********************************************************************** 
 * 
 *	Name:        SAD_Macroblock 
 *	Description:    fast way to find the SAD of one vector 
 * 
 *	Input:	        pointers to search_area and current block, 
 *                      Min_F1/F2/FR 
 *	Returns:        sad_f1/f2 
 *	Side effects: 
 * 
 *	Date: 940203        Author: PGB 
 *                      Mod:    KOL 
 * 
 ***********************************************************************/ 
 
 
int SAD_Macroblock (unsigned char *ii, unsigned char *act_block, 
                     int h_length, int Min_FRAME) 
{ 
  int i; 
  int sad = 0; 
  unsigned char *kk; 
 
  kk = act_block; 
  i = 16; 
  while (i--) 
  { 
    sad += (abs (*ii - *kk) + abs (*(ii + 1) - *(kk + 1)) 
            + abs (*(ii + 2) - *(kk + 2)) + abs (*(ii + 3) - *(kk + 3)) 
            + abs (*(ii + 4) - *(kk + 4)) + abs (*(ii + 5) - *(kk + 5)) 
            + abs (*(ii + 6) - *(kk + 6)) + abs (*(ii + 7) - *(kk + 7)) 
            + abs (*(ii + 8) - *(kk + 8)) + abs (*(ii + 9) - *(kk + 9)) 
          + abs (*(ii + 10) - *(kk + 10)) + abs (*(ii + 11) - *(kk + 11)) 
          + abs (*(ii + 12) - *(kk + 12)) + abs (*(ii + 13) - *(kk + 13)) 
        + abs (*(ii + 14) - *(kk + 14)) + abs (*(ii + 15) - *(kk + 15))); 
 
    ii += h_length; 
    kk += 16; 
    if (sad > Min_FRAME) 
      return INT_MAX; 
  } 
  return sad; 
} 
 
int SAD_Block (unsigned char *ii, unsigned char *act_block, 
                int h_length, int min_sofar) 
{ 
  int i; 
  int sad = 0; 
  unsigned char *kk; 
 
  kk = act_block; 
  i = 8; 
  while (i--) 
  { 
    sad += (abs (*ii - *kk) + abs (*(ii + 1) - *(kk + 1)) 
            + abs (*(ii + 2) - *(kk + 2)) + abs (*(ii + 3) - *(kk + 3)) 
            + abs (*(ii + 4) - *(kk + 4)) + abs (*(ii + 5) - *(kk + 5)) 
            + abs (*(ii + 6) - *(kk + 6)) + abs (*(ii + 7) - *(kk + 7))); 
 
    ii += h_length; 
    kk += 16; 
    if (sad > min_sofar) 
      return INT_MAX; 
  } 
  return sad; 
} 
 
int SAD_MB_Bidir (unsigned char *ii, unsigned char *aa, unsigned char *bb, 
                   int width, int min_sofar) 
{ 
  int i, sad = 0; 
  unsigned char *ll, *kk; 
  kk = aa; 
  ll = bb; 
  i = 16; 
  while (i--) 
  { 
    sad += (abs (*ii - ((*kk + *ll) >> 1)) + 
            abs (*(ii + 1) - ((*(kk + 1) + *(ll + 1)) >> 1)) + 
            abs (*(ii + 2) - ((*(kk + 2) + *(ll + 2)) >> 1)) + 
            abs (*(ii + 3) - ((*(kk + 3) + *(ll + 3)) >> 1)) + 
            abs (*(ii + 4) - ((*(kk + 4) + *(ll + 4)) >> 1)) + 
            abs (*(ii + 5) - ((*(kk + 5) + *(ll + 5)) >> 1)) + 
            abs (*(ii + 6) - ((*(kk + 6) + *(ll + 6)) >> 1)) + 
            abs (*(ii + 7) - ((*(kk + 7) + *(ll + 7)) >> 1)) + 
            abs (*(ii + 8) - ((*(kk + 8) + *(ll + 8)) >> 1)) + 
            abs (*(ii + 9) - ((*(kk + 9) + *(ll + 9)) >> 1)) + 
            abs (*(ii + 10) - ((*(kk + 10) + *(ll + 10)) >> 1)) + 
            abs (*(ii + 11) - ((*(kk + 11) + *(ll + 11)) >> 1)) + 
            abs (*(ii + 12) - ((*(kk + 12) + *(ll + 12)) >> 1)) + 
            abs (*(ii + 13) - ((*(kk + 13) + *(ll + 13)) >> 1)) + 
            abs (*(ii + 14) - ((*(kk + 14) + *(ll + 14)) >> 1)) + 
            abs (*(ii + 15) - ((*(kk + 15) + *(ll + 15)) >> 1))); 
 
    ii += width; 
    kk += width; 
    ll += width; 
    if (sad > min_sofar) 
      return INT_MAX; 
  } 
  return sad; 
} 
 
int SAD_MB_integer (int *ii, int *act_block, int h_length, int min_sofar) 
{ 
  int i, sad = 0, *kk; 
 
  kk = act_block; 
  i = 16; 
  while (i--) 
  { 
    sad += (abs (*ii - *kk) + abs (*(ii + 1) - *(kk + 1)) 
            + abs (*(ii + 2) - *(kk + 2)) + abs (*(ii + 3) - *(kk + 3)) 
            + abs (*(ii + 4) - *(kk + 4)) + abs (*(ii + 5) - *(kk + 5)) 
            + abs (*(ii + 6) - *(kk + 6)) + abs (*(ii + 7) - *(kk + 7)) 
            + abs (*(ii + 8) - *(kk + 8)) + abs (*(ii + 9) - *(kk + 9)) 
          + abs (*(ii + 10) - *(kk + 10)) + abs (*(ii + 11) - *(kk + 11)) 
          + abs (*(ii + 12) - *(kk + 12)) + abs (*(ii + 13) - *(kk + 13)) 
        + abs (*(ii + 14) - *(kk + 14)) + abs (*(ii + 15) - *(kk + 15))); 
 
    ii += h_length; 
    kk += 16; 
    if (sad > min_sofar) 
      return INT_MAX; 
  } 
  return sad; 
} 
 
/********************************************************************** 
 * 
 *	Name:        FindMB 
 *	Description:	Picks out one MB from picture 
 * 
 *	Input:        position of MB to pick out, 
 *        pointer to frame data, empty 16x16 array 
 *	Returns: 
 *	Side effects:	fills array with MB data 
 * 
 *	Date: 930119	Author: Karl Olav Lillevold 
 * 
 ***********************************************************************/ 
 
void FindMB (int x, int y, unsigned char *image, int MB[16][16]) 
 
{ 
  int n; 
  register int m; 
 
  for (n = 0; n < MB_SIZE; n++) 
    for (m = 0; m < MB_SIZE; m++) 
      MB[n][m] = *(image + x + m + (y + n) * pels); 
} 
 
/********************************************************************** 
 * 
 *	Name:        FindPred 
 *	Description:	Find the prediction block 
 * 
 *	Input:        position, vector, array for prediction 
 *        pointer to previous interpolated luminance, 
 * 
 *	Side effects:	fills array with prediction 
 * 
 *	Date: 930126        Author: Karl.Lillevold@nta.no 
 *            950208    Mod: Karl.Lillevold@nta.no 
 * 
 ***********************************************************************/ 
 
 
void FindPred (int x, int y, MotionVector * fr, unsigned char *prev, 
               int *pred, int bs, int comp) 
{ 
  int m, n; 
  int new_x, new_y; 
  int lx; 
 
  lx = (mv_outside_frame ? pels + (long_vectors ? 64 : 32) : pels); 
 
  new_x = x + fr->x; 
  new_y = y + fr->y; 
 
  new_x += ((comp & 1) << 3); 
  new_y += ((comp & 2) << 2); 
 
 
  /* Fill pred. data */ 
  for (n = 0; n < bs; n++) 
  { 
    for (m = 0; m < bs; m++) 
    { 
      /* Find interpolated pixel-value */ 
      *(pred + m + n * 16) = *(prev + (new_x + m) * 2 + fr->x_half + 
                               ((new_y + n) * 2 + fr->y_half) * lx * 2); 
    } 
  } 
  return; 
} 
 
/********************************************************************** 
 * 
 *	Name:        FindPredOBMC 
 *	Description:	Find the OBMC prediction block 
 * 
 *	Input:        position, vector, array for prediction 
 *        pointer to previous interpolated luminance, 
 * 
 *	Returns: 
 *	Side effects:	fills array with prediction 
 * 
 *	Date: 950209        Author: Karl.Lillevold@nta.no 
 * 
 ***********************************************************************/ 
 
 
void FindPredOBMC (int x, int y, MotionVector * MV[6][MBR + 1][MBC + 2], 
                   unsigned char *prev, int *pred, int comp, int PB) 
{ 
  int m, n; 
  int pc, pt, pb, pr, pl; 
  int nxc, nxt, nxb, nxr, nxl; 
  int nyc, nyt, nyb, nyr, nyl; 
  int xit, xib, xir, xil; 
  int yit, yib, yir, yil; 
  int vect, vecb, vecr, vecl; 
  int c8, t8, l8, r8; 
  int ti8, li8, ri8; 
  int xmb, ymb, lx; 
  MotionVector *fc, *ft, *fb, *fr, *fl; 
 
  int Mc[8][8] = { 
    {4, 5, 5, 5, 5, 5, 5, 4}, 
    {5, 5, 5, 5, 5, 5, 5, 5}, 
    {5, 5, 6, 6, 6, 6, 5, 5}, 
    {5, 5, 6, 6, 6, 6, 5, 5}, 
    {5, 5, 6, 6, 6, 6, 5, 5}, 
    {5, 5, 6, 6, 6, 6, 5, 5}, 
    {5, 5, 5, 5, 5, 5, 5, 5}, 
    {4, 5, 5, 5, 5, 5, 5, 4}, 
  }; 
  int Mt[8][8] = { 
    {2, 2, 2, 2, 2, 2, 2, 2}, 
    {1, 1, 2, 2, 2, 2, 1, 1}, 
    {1, 1, 1, 1, 1, 1, 1, 1}, 
    {1, 1, 1, 1, 1, 1, 1, 1}, 
    {0, 0, 0, 0, 0, 0, 0, 0}, 
    {0, 0, 0, 0, 0, 0, 0, 0}, 
    {0, 0, 0, 0, 0, 0, 0, 0}, 
    {0, 0, 0, 0, 0, 0, 0, 0}, 
  }; 
  int Mb[8][8] = { 
    {0, 0, 0, 0, 0, 0, 0, 0}, 
    {0, 0, 0, 0, 0, 0, 0, 0}, 
    {0, 0, 0, 0, 0, 0, 0, 0}, 
    {0, 0, 0, 0, 0, 0, 0, 0}, 
    {1, 1, 1, 1, 1, 1, 1, 1}, 
    {1, 1, 1, 1, 1, 1, 1, 1}, 
    {1, 1, 2, 2, 2, 2, 1, 1}, 
    {2, 2, 2, 2, 2, 2, 2, 2}, 
  }; 
  int Mr[8][8] = { 
    {0, 0, 0, 0, 1, 1, 1, 2}, 
    {0, 0, 0, 0, 1, 1, 2, 2}, 
    {0, 0, 0, 0, 1, 1, 2, 2}, 
    {0, 0, 0, 0, 1, 1, 2, 2}, 
    {0, 0, 0, 0, 1, 1, 2, 2}, 
    {0, 0, 0, 0, 1, 1, 2, 2}, 
    {0, 0, 0, 0, 1, 1, 2, 2}, 
    {0, 0, 0, 0, 1, 1, 1, 2}, 
  }; 
  int Ml[8][8] = { 
    {2, 1, 1, 1, 0, 0, 0, 0}, 
    {2, 2, 1, 1, 0, 0, 0, 0}, 
    {2, 2, 1, 1, 0, 0, 0, 0}, 
    {2, 2, 1, 1, 0, 0, 0, 0}, 
    {2, 2, 1, 1, 0, 0, 0, 0}, 
    {2, 2, 1, 1, 0, 0, 0, 0}, 
    {2, 2, 1, 1, 0, 0, 0, 0}, 
    {2, 1, 1, 1, 0, 0, 0, 0}, 
  }; 
 
  xmb = x / MB_SIZE + 1; 
  ymb = y / MB_SIZE + 1; 
 
  lx = (mv_outside_frame ? pels + (long_vectors ? 64 : 32) : pels); 
 
  c8 = (MV[0][ymb][xmb]->Mode == MODE_INTER4V ? 1 : 0); 
  c8 = (MV[0][ymb][xmb]->Mode == MODE_INTER4V_Q ? 1 : c8); 
 
  t8 = (MV[0][ymb - 1][xmb]->Mode == MODE_INTER4V ? 1 : 0); 
  t8 = (MV[0][ymb - 1][xmb]->Mode == MODE_INTER4V_Q ? 1 : t8); 
  ti8 = (MV[0][ymb - 1][xmb]->Mode == MODE_INTRA ? 1 : 0); 
  ti8 = (MV[0][ymb - 1][xmb]->Mode == MODE_INTRA_Q ? 1 : ti8); 
 
  l8 = (MV[0][ymb][xmb - 1]->Mode == MODE_INTER4V ? 1 : 0); 
  l8 = (MV[0][ymb][xmb - 1]->Mode == MODE_INTER4V_Q ? 1 : l8); 
  li8 = (MV[0][ymb][xmb - 1]->Mode == MODE_INTRA ? 1 : 0); 
  li8 = (MV[0][ymb][xmb - 1]->Mode == MODE_INTRA_Q ? 1 : li8); 
 
  r8 = (MV[0][ymb][xmb + 1]->Mode == MODE_INTER4V ? 1 : 0); 
  r8 = (MV[0][ymb][xmb + 1]->Mode == MODE_INTER4V_Q ? 1 : r8); 
  ri8 = (MV[0][ymb][xmb + 1]->Mode == MODE_INTRA ? 1 : 0); 
  ri8 = (MV[0][ymb][xmb + 1]->Mode == MODE_INTRA_Q ? 1 : ri8); 
 
  if (PB) 
  { 
    ti8 = li8 = ri8 = 0; 
  } 
  switch (comp + 1) 
  { 
 
    case 1: 
      vect = (ti8 ? (c8 ? 1 : 0) : (t8 ? 3 : 0)); 
      yit = (ti8 ? ymb : ymb - 1); 
      xit = xmb; 
 
      vecb = (c8 ? 3 : 0); 
      yib = ymb; 
      xib = xmb; 
 
      vecl = (li8 ? (c8 ? 1 : 0) : (l8 ? 2 : 0)); 
      yil = ymb; 
      xil = (li8 ? xmb : xmb - 1); 
 
      vecr = (c8 ? 2 : 0); 
      yir = ymb; 
      xir = xmb; 
 
      /* edge handling */ 
      if (ymb == 1) 
      { 
        yit = ymb; 
        vect = (c8 ? 1 : 0); 
      } 
      if (xmb == 1) 
      { 
        xil = xmb; 
        vecl = (c8 ? 1 : 0); 
      } 
      break; 
 
    case 2: 
      vect = (ti8 ? (c8 ? 2 : 0) : (t8 ? 4 : 0)); 
      yit = (ti8 ? ymb : ymb - 1); 
      xit = xmb; 
 
      vecb = (c8 ? 4 : 0); 
      yib = ymb; 
      xib = xmb; 
      vecl = (c8 ? 1 : 0); 
      yil = ymb; 
      xil = xmb; 
 
      vecr = (ri8 ? (c8 ? 2 : 0) : (r8 ? 1 : 0)); 
      yir = ymb; 
      xir = (ri8 ? xmb : xmb + 1); 
 
      /* edge handling */ 
      if (ymb == 1) 
      { 
        yit = ymb; 
        vect = (c8 ? 2 : 0); 
      } 
      if (xmb == pels / 16) 
      { 
        xir = xmb; 
        vecr = (c8 ? 2 : 0); 
      } 
      break; 
 
    case 3: 
      vect = (c8 ? 1 : 0); 
      yit = ymb; 
      xit = xmb; 
      vecb = (c8 ? 3 : 0); 
      yib = ymb; 
      xib = xmb; 
 
      vecl = (li8 ? (c8 ? 3 : 0) : (l8 ? 4 : 0)); 
      yil = ymb; 
      xil = (li8 ? xmb : xmb - 1); 
 
      vecr = (c8 ? 4 : 0); 
      yir = ymb; 
      xir = xmb; 
 
      /* edge handling */ 
      if (xmb == 1) 
      { 
        xil = xmb; 
        vecl = (c8 ? 3 : 0); 
      } 
      break; 
 
    case 4: 
      vect = (c8 ? 2 : 0); 
      yit = ymb; 
      xit = xmb; 
      vecb = (c8 ? 4 : 0); 
      yib = ymb; 
      xib = xmb; 
      vecl = (c8 ? 3 : 0); 
      yil = ymb; 
      xil = xmb; 
 
      vecr = (ri8 ? (c8 ? 4 : 0) : (r8 ? 3 : 0)); 
      yir = ymb; 
      xir = (ri8 ? xmb : xmb + 1); 
 
      /* edge handling */ 
      if (xmb == pels / 16) 
      { 
        xir = xmb; 
        vecr = (c8 ? 4 : 0); 
      } 
      break; 
 
    default: 
      fprintf (stderr, "Illegal block number in FindPredOBMC (pred.c)\n"); 
      exit (-1); 
      break; 
  } 
 
  fc = MV[c8 ? comp + 1 : 0][ymb][xmb]; 
 
  ft = MV[vect][yit][xit]; 
  fb = MV[vecb][yib][xib]; 
  fr = MV[vecr][yir][xir]; 
  fl = MV[vecl][yil][xil]; 
 
  nxc = 2 * x + ((comp & 1) << 4); 
  nyc = 2 * y + ((comp & 2) << 3); 
  nxt = nxb = nxr = nxl = nxc; 
  nyt = nyb = nyr = nyl = nyc; 
 
  nxc += 2 * fc->x + fc->x_half; 
  nyc += 2 * fc->y + fc->y_half; 
  nxt += 2 * ft->x + ft->x_half; 
  nyt += 2 * ft->y + ft->y_half; 
  nxb += 2 * fb->x + fb->x_half; 
  nyb += 2 * fb->y + fb->y_half; 
  nxr += 2 * fr->x + fr->x_half; 
  nyr += 2 * fr->y + fr->y_half; 
  nxl += 2 * fl->x + fl->x_half; 
  nyl += 2 * fl->y + fl->y_half; 
 
  /* Fill pred. data */ 
  for (n = 0; n < 8; n++) 
  { 
    for (m = 0; m < 8; m++) 
    { 
 
      /* Find interpolated pixel-value */ 
      pc = *(prev + nxc + 2 * m + (nyc + 2 * n) * lx * 2) * Mc[n][m]; 
      pt = *(prev + nxt + 2 * m + (nyt + 2 * n) * lx * 2) * Mt[n][m]; 
      pb = *(prev + nxb + 2 * m + (nyb + 2 * n) * lx * 2) * Mb[n][m]; 
      pr = *(prev + nxr + 2 * m + (nyr + 2 * n) * lx * 2) * Mr[n][m]; 
      pl = *(prev + nxl + 2 * m + (nyl + 2 * n) * lx * 2) * Ml[n][m]; 
 
      /* pc = *(prev + nxc + 2*m + (nyc + 2*n)*lx*2) * 8; pt = *(prev + 
       * nxt + 2*m + (nyt + 2*n)*lx*2) * 0;; pb = *(prev + nxb + 2*m + 
       * (nyb + 2*n)*lx*2) * 0; pr = *(prev + nxr + 2*m + (nyr + 
       * 2*n)*lx*2) * 0; pl = *(prev + nxl + 2*m + (nyl + 2*n)*lx*2) * 0;$ */ 
 
      *(pred + m + n * 16) = (pc + pt + pb + pr + pl + 4) >> 3; 
    } 
  } 
  return; 
} 
                        
/*********************************************************************** 
 * 
 *	Name:        MB_Reconstruct 
 *	Description:    Reconstructs the macroblock by using 
 *                  previously calculated prediction array 
 * 
 *	Input:	        pointers prediction diff and prediction arrays 
 *	Returns:        pointer to reconstructed MB data 
 *	Side effects:   allocates memory to MB_structure 
 * 
 *	Date: 97/06/23        Author: Berna Erol <bernae@ee.ubc.ca> 
 * 
 ***********************************************************************/ 
 
MB_Structure *MB_Reconstruct (MB_Structure * diff, MB_Structure * pred) 
{ 
  int i, j; 
  MB_Structure *recon = (MB_Structure *) malloc (sizeof (MB_Structure)); 
 
  /* Reconstruction */ 
  for (j = 0; j < MB_SIZE; j++) 
    for (i = 0; i < MB_SIZE; i++) 
      recon->lum[j][i] = pred->lum[j][i] + diff->lum[j][i]; 
 
  for (j = 0; j < MB_SIZE >> 1; j++) 
    for (i = 0; i < MB_SIZE >> 1; i++) 
    { 
      recon->Cr[j][i] = pred->Cr[j][i] + diff->Cr[j][i]; 
      recon->Cb[j][i] = pred->Cb[j][i] + diff->Cb[j][i]; 
    } 
 
  return recon; 
} 
