/************************************************************************ 
 * 
 *  coder.c, main coding engine of tmn (TMN encoder) 
 * 
 *  Copyright (C) 1997  University of BC, Canada 
 * 
 *  Contacts:  
 *  Michael Gallant                   <mikeg@ee.ubc.ca> 
 *  Guy Cote                          <guyc@ee.ubc.ca> 
 *  Berna Erol                        <bernae@ee.ubc.ca> 
 * 
 *  UBC Image Processing Laboratory   http://www.ee.ubc.ca/image 
 *  2356 Main Mall                    tel.: +1 604 822 4051 
 *  Vancouver BC Canada V6T1Z4        fax.: +1 604 822 5949 
 * 
 ************************************************************************/ 
 
/* 
 * Disclaimer of Warranty 
 * 
 * These software programs are available to the user without any 
 * license fee or royalty on an "as is" basis. The University of 
 * British Columbia disclaims any and all warranties, whether 
 * express, implied, or statuary, including any implied warranties 
 * or merchantability or of fitness for a particular purpose.  In no 
 * event shall the copyright-holder be liable for any incidental, 
 * punitive, or consequential damages of any kind whatsoever arising 
 * from the use of these programs. 
 * 
 * This disclaimer of warranty extends to the user of these programs 
 * and user's customers, employees, agents, transferees, successors, 
 * and assigns. 
 * 
 * The University of British Columbia does not represent or warrant 
 * that the programs furnished hereunder are free of infringement of 
 * any third-party patents. 
 * 
 * Commercial implementations of H.263, including shareware, are 
 * subject to royalty fees to patent holders.  Many of these patents 
 * are general enough such that they are unavoidable regardless of 
 * implementation design. 
 * 
*/ 
 
#include"sim.h" 

 extern int which_dct;
extern int lossless_codec;

/********************************************************************** 
 * 
 *	Name:         CodeOneEP 
 *	Description:  code one EP image  
 *	 
 *	Input:        pointer to image, prev_image, prev_recon, Q 
 *         
 *	Returns:      pointer to reconstructed image 
 *	Side effects: memory is allocated to recon image 
 * 
 *	Date: 950930  Author: Michael Gallant <mikeg@ee.ubc.ca> 
 * 
 ***********************************************************************/ 
 void CodeOneEP(PictImage *curr, PictImage *prev_enhance_image,  
               PictImage *prev_enhance_recon, int QP, int frameskip, Bits *bits, 
               Pict *pic, PictImage *curr_ref_recon, PictImage *curr_enhance_recon) 
{ 
  unsigned char *prev_ipol, *base_ipol, *pi, *bi; 
  PictImage *prev_recon=NULL, *base_recon=NULL, *pr_edge = NULL, *nr_edge = NULL; 
  MotionVector *MV[7][MBR+1][MBC+2]; 
  MotionVector ZERO = {0,0,0,0,0}; 
  MB_Structure *recon_data_ep = NULL; 
  MB_Structure *diff;  
  int *rcoeff, *coeff; 
  int *qcoeff_ep = NULL; 
  int Mode; 
  int CBP; 
  int newgob, advanced_temporarily_off = DEF_AIC_MODE; 
  int i,j,k; 
  int ep_prediction_type, MV_present = OFF; 
 
  /* buffer control vars */ 
  float QP_cumulative = (float)0.0; 
  int abs_mb_num = 0, QuantChangePostponed = 0; 
  int QP_new, QP_prev, QP_xmitted=QP; 
 
  MB_Structure *pred = (MB_Structure *)malloc(sizeof(MB_Structure));  
 
  /* advanced intra coding variables */ 
  int *store_rcoeff, *store_coeff, *pcoeff; 
 
  ZeroBits(bits); 
 
  /* Currently, the MV info is stored in 7 MV structs per MB. MV[0] 
   * stores the 16x16 MV. MV[1] through MV[4] store the 8x8 MVs. MV[6] 
   * stores the PB delta MV or the forward predicted B MV. MV[5] 
   * stores the backward predicted MV for true-B pictures. */  
 
  InitializeMV(MV); 
 
  /* Mark PMV's outside the frame. */ 
  for (i = 1; i < (pels>>4)+1; i++)  
  { 
    for (j = 0; j < 7; j++) 
    { 
      MarkVec(MV[j][0][i]);   
    } 
    MV[0][0][i]->Mode = MODE_INTRA; 
  } 
 
  /* Zero out PMV's outside the frame */ 
  for (i = 0; i < (lines>>4)+1; i++)  
  { 
    for (j = 0; j < 7; j++)  
    { 
      ZeroVec(MV[j][i][0]); 
      ZeroVec(MV[j][i][(pels>>4)+1]); 
    } 
    MV[0][i][0]->Mode = MODE_INTRA; 
    MV[0][i][(pels>>4)+1]->Mode = MODE_INTRA; 
  } 
 
  GenerateFrameAndInterpolatedImages(prev_enhance_recon, pic, &prev_ipol, &prev_recon, &pi, &pr_edge); 
  GenerateFrameAndInterpolatedImages(curr_ref_recon, pic, &base_ipol, &base_recon, &bi, &nr_edge); 
 
  if (adv_pred) 
  { 
    /* Turn off advanced coding since there can be only 1  
     * motion vector for B frames of IPB. */ 
    advanced_temporarily_off = YES; 
    overlapping_MC = OFF; 
    adv_pred = OFF; 
    use_4mv = OFF; 
  } 
 
  MotionEstimatePicture( curr->lum,prev_recon->lum, base_recon->lum,          
                         prev_ipol, base_ipol, pic->seek_dist, MV, 
                         pic->use_gobsync, EI_EP_PICTURE_ESTIMATION); 
 
  QP_new = QP_xmitted = QP_prev = QP; 
 
  /* Copy the passed value of QP */ 
 
  for ( j = 0; j < lines/MB_SIZE; j++)  
  { 
    newgob = 0; 
 
    if (j == 0)  
    { 
      if (advanced_intra_coding) 
      { 
        /* store the coeff for the frame */ 
        if ((store_coeff=(int *)calloc(384*(pels/MB_SIZE)*(lines/MB_SIZE), sizeof(int))) == 0)  
        { 
          fprintf(stderr,"coder(): Couldn't allocate store_coeff.\n"); 
          exit(-1); 
        } 
        if ((store_rcoeff=(int *)calloc(384*(pels/MB_SIZE)*(lines/MB_SIZE), sizeof(int))) == 0)  
        { 
          fprintf(stderr,"coder(): Couldn't allocate store_rcoeff.\n"); 
          exit(-1); 
        } 
      } 
 
      pic->QUANT = QP_new; 
 
      bits->header += CountBitsPicture(pic); 
 
      QP_xmitted = QP_prev = QP_new; 
    } 
    else if (pic->use_gobsync && j%pic->use_gobsync == 0)  
    { 
      bits->header += CountBitsGOB(j,QP_new,pic); 
 
      /* insert gob sync */ 
      QP_xmitted = QP_prev = QP_new; 
      newgob = 1; 
    } 
  
    for ( i = 0; i < pels/MB_SIZE; i++)  
    { 
      pic->MB = i + j * (pels/MB_SIZE); 
       
      /* store the QP for every macroblock */ 
      quant_map[j+1][i+1] = QP_xmitted; 
 
      if ((rcoeff = (int *)malloc(sizeof(int)*384)) == NULL)  
      { 
        fprintf(stderr,"MB_Coder: Could not allocate space for rcoeff\n"); 
        exit(-1); 
      }   
 
      /* Predict true B-MB */ 
      diff = Predict_EP( curr, prev_recon, prev_ipol, base_recon,  
                         base_ipol, pred, i*MB_SIZE, j*MB_SIZE, 
                         MV, &ep_prediction_type, &Mode, pic->RTYPE); 
 
      if (EP_INTRA_PREDICTION == ep_prediction_type) 
      { 
        FillLumBlock(i*MB_SIZE, j*MB_SIZE, curr, diff); 
        FillChromBlock(i*MB_SIZE, j*MB_SIZE, curr, diff); 
      } 
 
      if ((qcoeff_ep=(int *)malloc(sizeof(int)*384)) == 0)  
      { 
        fprintf(stderr,"coder(): Couldn't allocate qcoeff_ep.\n"); 
        exit(-1); 
      } 
      coeff = MB_Encode(diff); 
      if (advanced_intra_coding)  
      { 
        if (!(Mode == MODE_INTRA || Mode == MODE_INTRA_Q))  
        { 
          for (k=0;k<6;k++)  
          {     
            /* store default coeff if non-intra macroblock */ 
 	    if (!which_dct) {
	      store_coeff[(i + j * pels/MB_SIZE) * 384 + k * 64] = 1024; 
	      store_rcoeff[(i + j * pels/MB_SIZE) * 384 + k * 64] = 1024; 
	    } else {
	      store_coeff[(i + j * pels/MB_SIZE) * 384 + k * 64] = DEFAULT_DC_BINDCT;
	      store_rcoeff[(i + j * pels/MB_SIZE) * 384 + k * 64] = DEFAULT_DC_BINDCT;
	    }         
	  } 
          for (k=0;k<6;k++) 
            Quant_blk(coeff,qcoeff_ep,QP_xmitted, Mode,k); 
          CBP = FindCBP(qcoeff_ep, Mode, 64); 
          if (CBP == 0 && (Mode == MODE_INTER || Mode == MODE_INTER_Q))  
            ZeroMBlock(diff); 
          else 
          {   
            for (k=0;k<6;k++) 
            { 
              Quant_blk(coeff,qcoeff_ep,QP_xmitted, Mode,k); 
              Dequant(qcoeff_ep, rcoeff, QP_xmitted, Mode,k); 
            } 
            MB_Decode(rcoeff, diff); 
          } 
        } 
        else  
        { 
          if ((pcoeff=(int *)malloc(sizeof(int)*384)) == 0)  
          { 
            fprintf(stderr,"coder(): Couldn't allocate pcoeff.\n"); 
            exit(-1); 
          } 
 
          /* store the quantized DCT coefficients */ 
          memcpy( (void *) (store_coeff + (i + j*pels/MB_SIZE)*384), (void *) coeff, sizeof(int) * 384); 
 
          /* Do Intra mode prediction */ 
          pic->Intra_Mode = Choose_Intra_Mode(pcoeff, store_coeff, i, j, newgob); 
 
          for (k=0;k<6;k++)  
          {  
            Intra_AC_DC_Encode(coeff, store_rcoeff, pic->Intra_Mode, i, j, newgob,k); 
            Quant_blk(coeff,pcoeff,QP_xmitted,Mode,k); 
            Dequant(pcoeff, rcoeff, QP_xmitted, Mode,k); 
            Intra_AC_DC_Decode(rcoeff, store_rcoeff, pic->Intra_Mode, i, j, newgob,k); 
          } 
          MB_Decode(rcoeff, diff); 
          CBP = FindCBP(pcoeff,Mode,64); 
        }     
      } 
      else 
      {   
        for (k=0;k<6;k++) 
          Quant_blk(coeff,qcoeff_ep,QP_xmitted, Mode,k); 
        CBP = FindCBP(qcoeff_ep, Mode, 64); 
        if (CBP == 0 && (Mode == MODE_INTER || Mode == MODE_INTER_Q))  
          ZeroMBlock(diff); 
        else 
        { 
          for (k=0;k<6;k++)   
            Dequant(qcoeff_ep, rcoeff, QP_xmitted, Mode,k);  
          MB_Decode(rcoeff, diff); 
        } 
      } 
 
      recon_data_ep = MB_Recon_EP( prev_recon, prev_ipol, diff,  
                                   base_recon, base_ipol,  
                                   i*MB_SIZE,j*MB_SIZE,MV, 
                                   ep_prediction_type, pic->RTYPE); 
      Clip(recon_data_ep); 
      free(diff); 
      free(coeff); 
 
      if (!CBP) 
        intra_refresh[j+1][i+1] += 1; 
 
 
      if ((CBP==0) && (EP_FORWARD_PREDICTION == ep_prediction_type) && 
          (Mode == MODE_INTER) && (0 == pic->DQUANT) && 
          (EqualVec(MV[0][j+1][i+1],&ZERO)) )  
      { 
        coded_map[j+1][i+1] = 0; 
        quant_map[j+1][i+1] = 0; 
         
        CountBitsScalMB(Mode, 1, CBP, 0, pic, bits, ep_prediction_type, 0); 
      } 
      else  
      { 
        if ( (EP_FORWARD_PREDICTION == ep_prediction_type) || 
             ( (EP_BIDIRECTIONAL_PREDICTION == ep_prediction_type) && 
               ( !(EqualVec(MV[0][j+1][i+1],&ZERO)) || (0 != CBP) ) ) ) 
        { 
          MV_present = ON; 
        } 
        else 
        { 
          MV_present = OFF; 
        } 
 
        CountBitsScalMB(Mode,0,CBP,0,pic,bits,ep_prediction_type, MV_present); 
 
        if (MODE_INTER == Mode || MODE_INTER_Q == Mode) 
        { 
          coded_map[j+1][i+1] = 1; 
          quant_map[j+1][i+1] = QP_xmitted; 
          bits->no_inter++; 
 
          if (MV_present) 
          { 
            CountBitsVectors( MV, bits, i, j, Mode, newgob, pic,  
                              ep_prediction_type); 
          } 
        } 
        else  
        { 
          /* MODE_INTRA or MODE_INTRA_Q */ 
          coded_map[j+1][i+1] = 2; 
          quant_map[j+1][i+1] = QP_xmitted; 
          intra_refresh[j+1][i+1] = 0; 
          bits->no_intra++; 
        } 
          
        if ( (Mode == MODE_INTRA || Mode == MODE_INTRA_Q) && advanced_intra_coding ) 
        { 
          Scan(pcoeff,pic->Intra_Mode); 
          CountBitsCoeff(pcoeff, Mode, CBP, bits, 64); 
        } 
        else if (CBP || Mode == MODE_INTRA || Mode == MODE_INTRA_Q) 
        { 
          Scan(qcoeff_ep,0); 
          CountBitsCoeff(qcoeff_ep, Mode, CBP, bits, 64); 
        } 
      } 
      QP_prev = QP_xmitted; 
     
      abs_mb_num++; 
      QP_cumulative += QP_xmitted;      
#ifdef PRINTQ  
 
      /* most useful when quantizer changes within a picture */ 
      if (QuantChangePostponed) 
        fprintf(stdout,"@%2d",QP_xmitted); 
      else 
        fprintf(stdout," %2d",QP_xmitted); 
#endif 
 
      ReconImage(i,j,recon_data_ep,curr_enhance_recon); 
  
      free(qcoeff_ep); 
      free(recon_data_ep); 
 
      if (advanced_intra_coding && (Mode == MODE_INTRA || Mode == MODE_INTRA_Q))   
        free(pcoeff); 
    } 
#ifdef PRINTQ 
    fprintf(stdout,"\n"); 
#endif 
  } 
 
  pic->QP_mean = QP_cumulative/(float)abs_mb_num; 
 
  /* Free memory */ 
  free(pred); 
  free(prev_recon); 
  free(base_recon); 
  FreeImage(pr_edge); 
  FreeImage(nr_edge); 
  free(pi); 
  free(bi); 
 
  for (j = 0; j < (lines>>4)+1; j++) 
    for (i = 0; i < (pels>>4)+2; i++)  
      for (k = 0; k < 7; k++) 
        free(MV[k][j][i]); 
 
  if (advanced_intra_coding) 
  { 
    free(store_coeff); 
    free(store_rcoeff); 
  } 
 
  if (advanced_temporarily_off) 
  { 
    overlapping_MC = ON; 
    adv_pred = ON; 
    use_4mv = ON; 
    advanced_temporarily_off = NO; 
  } 
 
  return; 
} 
 
 
/*********************************************************************** 
 * 
 *	Name:           Predict_EP 
 *	Description:    Predicts macroblock in EP frame prediction 
 * 
 *	Input:	        pointers to current frame, previous recon. previous 
 *                      interpolated, base recon, base interpolated, 
 *                      pos. in image, MV data, and MB prediction type. 
 *	Returns:        pointer to differential MB data after prediction 
 *	Side effects: 
 * 
 *	Date: 971001    Author: Michael Gallant ---mikeg@ee.ubc.ca 
 * 
 ***********************************************************************/ 
MB_Structure *Predict_EP( PictImage *curr_image, PictImage *prev_recon, unsigned char *prev_ipol,  
                         PictImage *base_recon, unsigned char *base_ipol, MB_Structure *pred_macroblock,  
                         int x, int y, MotionVector *MV[7][MBR+1][MBC+2],  
                         int *prediction_type, int *mode, int RTYPE) 
{ 
  int curr[16][16]; 
  MB_Structure *forward_pred = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MB_Structure *bidir_pred = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MB_Structure *upward_pred = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MB_Structure *pred_error = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MotionVector *forward_MV; 
  MotionVector ZERO = {0,0,0,0,0}; 
  int SADbidir, SADforw, SADup, SADmin; 
  int dx, dy, xmb, ymb, i, j; 
 
  xmb = x / MB_SIZE + 1; 
  ymb = y / MB_SIZE + 1; 
 
  forward_MV = MV[0][ymb][xmb]; 
 
  /* Find MB in current image */ 
  FindMB (x, y, curr_image->lum, curr); 
 
  /* Forward prediction. */ 
  FindPred (x, y, forward_MV, prev_ipol, &forward_pred->lum[0][0], 16, 0); 
 
  FindPred (x, y, &ZERO, base_ipol, &upward_pred->lum[0][0], 16, 0); 
   
  for (j = 0; j < MB_SIZE; j++) 
  { 
    for (i = 0; i < MB_SIZE; i++) 
    { 
      /* Bidirectional prediction. */ 
      bidir_pred->lum[j][i] = (forward_pred->lum[j][i] + 
                               upward_pred->lum[j][i]) / 2; 
    } 
  } 
 
  SADforw = SAD_MB_integer (&curr[0][0], &forward_pred->lum[0][0], 16, INT_MAX); 
 
  SADbidir = SAD_MB_integer (&curr[0][0], &bidir_pred->lum[0][0], 16, INT_MAX); 
  SADbidir += 100; 
 
  SADup = SAD_MB_integer (&curr[0][0], &upward_pred->lum[0][0], 16, INT_MAX); 
  SADup -= 50; 
 
  /* Prediction direction decision. */ 
  if ((SADup <= SADforw) && (SADup <= SADbidir)) 
  { 
    *prediction_type = EP_UPWARD_PREDICTION; 
    SADmin = SADup; 
  }  
  else if ((SADforw < SADup) && (SADforw <= SADbidir)) 
  { 
    *prediction_type = EP_FORWARD_PREDICTION; 
    SADmin = SADforw; 
  }  
  else 
  { 
    *prediction_type = EP_BIDIRECTIONAL_PREDICTION; 
    SADmin = SADbidir; 
  } 
 
  *(mode) = ChooseMode( curr_image->lum, x, y, SADmin); 
                                 
  if (MODE_INTRA == *(mode) || MODE_INTRA_Q == *(mode))  
  { 
    *prediction_type = EP_INTRA_PREDICTION; 
  } 
 
  switch (*prediction_type) 
  { 
    case EP_FORWARD_PREDICTION: 
 
      /* Translate MV to chrominance-resolution (downsampled). */ 
      dx = 2 * forward_MV->x + forward_MV->x_half; 
      dy = 2 * forward_MV->y + forward_MV->y_half; 
      dx = (dx % 4 == 0 ? dx >> 1 : (dx >> 1) | 1); 
      dy = (dy % 4 == 0 ? dy >> 1 : (dy >> 1) | 1); 
 
      /* Predict B as if P. */ 
      DoPredChrom_P (x, y, dx, dy, curr_image, prev_recon, forward_pred, pred_error, RTYPE); 
 
      for (j = 0; j < MB_SIZE; j++) 
      { 
        for (i = 0; i < MB_SIZE; i++) 
        {  
          pred_error->lum[j][i] = *(curr_image->lum + x + i + (y + j) * pels) - forward_pred->lum[j][i]; 
          pred_macroblock->lum[j][i] = forward_pred->lum[j][i]; 
        } 
      } 
      for (j = 0; j < MB_SIZE >> 1; j++) 
      { 
        for (i = 0; i < MB_SIZE >> 1; i++) 
        { 
          pred_macroblock->Cr[j][i] = forward_pred->Cr[j][i]; 
          pred_macroblock->Cb[j][i] = forward_pred->Cb[j][i]; 
        } 
      } 
 
      forward_MV->Mode = MODE_INTER; 
 
      break; 
 
    case EP_UPWARD_PREDICTION:     
 
      /* Predict B as if P. */ 
      DoPredChrom_P (x, y, 0, 0, curr_image, base_recon, upward_pred, pred_error, RTYPE); 
 
      for (j = 0; j < MB_SIZE; j++) 
      { 
        for (i = 0; i < MB_SIZE; i++) 
        { 
          pred_error->lum[j][i] = *(curr_image->lum + x + i + (y + j) * pels) - upward_pred->lum[j][i]; 
          pred_macroblock->lum[j][i] = upward_pred->lum[j][i]; 
        } 
      } 
 
      for (j = 0; j < MB_SIZE >> 1; j++) 
      { 
        for (i = 0; i < MB_SIZE >> 1; i++) 
        { 
          pred_macroblock->Cr[j][i] = upward_pred->Cr[j][i]; 
          pred_macroblock->Cb[j][i] = upward_pred->Cb[j][i]; 
        } 
      } 
 
      forward_MV->Mode = MODE_INTER; 
 
      /* Set forward MV data to 0 for future MV prediction. */ 
      forward_MV->x = forward_MV->x_half = 0; 
      forward_MV->y = forward_MV->y_half = 0; 
 
      break; 
 
    case EP_BIDIRECTIONAL_PREDICTION:   
 
      /* Translate MV to chrominance-resolution (downsampled). */ 
      dx = 2 * forward_MV->x + forward_MV->x_half; 
      dy = 2 * forward_MV->y + forward_MV->y_half; 
      dx = (dx % 4 == 0 ? dx >> 1 : (dx >> 1) | 1); 
      dy = (dy % 4 == 0 ? dy >> 1 : (dy >> 1) | 1); 
 
      DoPredChrom_P (x, y, dx, dy, curr_image, prev_recon, forward_pred, pred_error, RTYPE); 
 
      /* Zero MV for upward prediction */ 
      DoPredChrom_P (x, y, 0, 0, curr_image, base_recon, upward_pred, pred_error, RTYPE); 
 
      for (j = 0; j < MB_SIZE; j++) 
      { 
        for (i = 0; i < MB_SIZE; i++) 
        { 
          pred_error->lum[j][i] = *(curr_image->lum + x + i + (y + j) * pels) - 
                                  bidir_pred->lum[j][i]; 
          pred_macroblock->lum[j][i] = bidir_pred->lum[j][i]; 
        } 
      } 
      for (j = 0; j < MB_SIZE >> 1; j++) 
      { 
        for (i = 0; i < MB_SIZE >> 1; i++) 
        { 
          pred_macroblock->Cr[j][i] = (forward_pred->Cr[j][i] + 
                                       upward_pred->Cr[j][i]) / 2; 
          pred_macroblock->Cb[j][i] = (forward_pred->Cb[j][i] + 
                                       upward_pred->Cb[j][i]) / 2; 
          pred_error->Cr[j][i] = *(curr_image->Cr + (x>>1) + i + ((y>>1) + j) * cpels) - 
                                  pred_macroblock->Cr[j][i]; 
          pred_error->Cb[j][i] = *(curr_image->Cb + (x>>1) + i + ((y>>1) + j) * cpels) - 
                                  pred_macroblock->Cb[j][i]; 
        } 
      } 
 
      forward_MV->Mode = MODE_INTER; 
 
      break; 
 
    case EP_INTRA_PREDICTION: 
     
 
      /* Set forward MV data to 0 for future MV prediction. */ 
      forward_MV->x = forward_MV->x_half = 0; 
      forward_MV->y = forward_MV->y_half = 0; 
 
      forward_MV->Mode = MODE_INTRA; 
 
      break; 
 
    default: 
     
      fprintf (stderr, "Illegal scalable prediction type in MB_Predict_EP (pred.c)\n"); 
      exit (-1); 
  } 
 
 
  free (forward_pred); 
  free (bidir_pred); 
  free (upward_pred); 
 
  return pred_error; 
} 
 
/********************************************************************** 
 * 
 *	Name:           MB_Recon_EP 
 *	Description:	Reconstructs EP MB 
 * 
 *	Input:	        pointers to decoded residual, previous recon, previous 
 *                      interpolated, base recon, base interpolated, 
 *                      pos. in image, MV data, and MB prediction type. 
 *	Returns:        pointer to reconstructed MB data after motion compensation 
 *	Side effects:   allocates memory to MB_structure 
 * 
 *	Date: 971101	Author: Michael Gallant --- mikeg@ee.ubc.ca 
 * 
 ***********************************************************************/ 
MB_Structure *MB_Recon_EP( PictImage *prev_recon, unsigned char *prev_ipol,  
                           MB_Structure *diff, PictImage *base_recon,  
                           unsigned char *base_ipol, int x_curr, int y_curr, 
                           MotionVector *MV[7][MBR+1][MBC+2], 
                           int prediction_type, int RTYPE) 
{ 
  MB_Structure *forward = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MB_Structure *upward = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MB_Structure *recon_data = (MB_Structure *) malloc (sizeof (MB_Structure)); 
  MotionVector *forward_MV; 
  MotionVector ZERO = {0,0,0,0,0}; 
  int dxf, dyf; 
  int i, j; 
 
  switch (prediction_type) 
  { 
 
    case EP_FORWARD_PREDICTION: 
 
      forward_MV = MV[0][y_curr / MB_SIZE + 1][x_curr / MB_SIZE + 1]; 
 
      if (forward_MV->Mode == MODE_INTER || forward_MV->Mode == MODE_INTER_Q) 
      { 
        /* Inter 16x16 */ 
        ReconLumBlock_P (x_curr, y_curr, forward_MV, prev_ipol, &diff->lum[0][0], 16, 0); 
        dxf = 2 * forward_MV->x + forward_MV->x_half; 
        dyf = 2 * forward_MV->y + forward_MV->y_half; 
        dxf = (dxf % 4 == 0 ? dxf >> 1 : (dxf >> 1) | 1); 
        dyf = (dyf % 4 == 0 ? dyf >> 1 : (dyf >> 1) | 1); 
        ReconChromBlock_P (x_curr, y_curr, dxf, dyf, prev_recon, diff, RTYPE); 
      } 
 
      break; 
 
    case EP_UPWARD_PREDICTION: 
 
      /* Inter 16x16 */ 
      ReconLumBlock_P (x_curr, y_curr, &ZERO, base_ipol, &diff->lum[0][0], 16, 0); 
      ReconChromBlock_P (x_curr, y_curr, 0, 0, base_recon, diff, RTYPE); 
 
      break; 
 
    case EP_BIDIRECTIONAL_PREDICTION: 
 
      forward_MV = MV[0][y_curr / MB_SIZE + 1][x_curr / MB_SIZE + 1]; 
 
      if ((forward_MV->Mode == MODE_INTER || forward_MV->Mode == MODE_INTER_Q)) 
      { 
        /* Forward prediction. */ 
        FindPred (x_curr, y_curr, forward_MV, prev_ipol, &forward->lum[0][0], 16, 0); 
 
        /* Backward prediction. */ 
        FindPred (x_curr, y_curr, &ZERO, base_ipol, &upward->lum[0][0], 16, 0); 
 
        for (j = 0; j < MB_SIZE; j++) 
        { 
          for (i = 0; i < MB_SIZE; i++) 
          { 
            diff->lum[j][i] += (forward->lum[j][i] + 
                                upward->lum[j][i]) / 2; 
          } 
        } 
 
        dxf = 2 * forward_MV->x + forward_MV->x_half; 
        dyf = 2 * forward_MV->y + forward_MV->y_half; 
        dxf = (dxf % 4 == 0 ? dxf >> 1 : (dxf >> 1) | 1); 
        dyf = (dyf % 4 == 0 ? dyf >> 1 : (dyf >> 1) | 1); 
 
        FindChromBlock_P (x_curr, y_curr, dxf, dyf, prev_recon, forward, RTYPE); 
 
        FindChromBlock_P (x_curr, y_curr, 0, 0, base_recon, upward, RTYPE); 
 
        for (j = 0; j < MB_SIZE / 2; j++) 
        { 
          for (i = 0; i < MB_SIZE / 2; i++) 
          { 
            diff->Cr[j][i] += (forward->Cr[j][i] + 
                               upward->Cr[j][i]) / 2; 
            diff->Cb[j][i] += (forward->Cb[j][i] + 
                               upward->Cb[j][i]) / 2; 
          } 
        } 
      } 
 
      break; 
 
    case EP_INTRA_PREDICTION: 
     
      break; 
 
    default: 
 
      fprintf (stderr, "Illegal scalable prediction type in MB_Recon_EP (pred.c)\n"); 
      exit (-1); 
 
      break; 
 
  } 
 
  memcpy (recon_data, diff, sizeof (MB_Structure)); 
 
  free (forward); 
  free (upward); 
 
  return recon_data; 
} 
