/************************************************************************ 
 * 
 *  coder.c, main coding engine of tmn (TMN encoder) 
 * 
 *  Copyright (C) 1997  University of BC, Canada 
 * 
 *  Contacts:  
 *  Michael Gallant                   <mikeg@ee.ubc.ca> 
 *  Guy Cote                          <guyc@ee.ubc.ca> 
 *  Berna Erol                        <bernae@ee.ubc.ca> 
 * 
 *  UBC Image Processing Laboratory   http://www.ee.ubc.ca/image 
 *  2356 Main Mall                    tel.: +1 604 822 4051 
 *  Vancouver BC Canada V6T1Z4        fax.: +1 604 822 5949 
 * 
 *  Copyright (C) 1995, 1996  Telenor R&D, Norway 
 *   
 *  Contacts:  
 *  Robert Danielsen                  <Robert.Danielsen@nta.no> 
 * 
 *  Telenor Research and Development  http://www.nta.no/brukere/DVC/ 
 *  P.O.Box 83                        tel.:   +47 63 84 84 00 
 *  N-2007 Kjeller, Norway            fax.:   +47 63 81 00 76 
 *   
 ************************************************************************/ 
 
/* 
 * Disclaimer of Warranty 
 * 
 * These software programs are available to the user without any 
 * license fee or royalty on an "as is" basis. The University of 
 * British Columbia disclaims any and all warranties, whether 
 * express, implied, or statuary, including any implied warranties 
 * or merchantability or of fitness for a particular purpose.  In no 
 * event shall the copyright-holder be liable for any incidental, 
 * punitive, or consequential damages of any kind whatsoever arising 
 * from the use of these programs. 
 * 
 * This disclaimer of warranty extends to the user of these programs 
 * and user's customers, employees, agents, transferees, successors, 
 * and assigns. 
 * 
 * The University of British Columbia does not represent or warrant 
 * that the programs furnished hereunder are free of infringement of 
 * any third-party patents. 
 * 
 * Commercial implementations of H.263, including shareware, are 
 * subject to royalty fees to patent holders.  Many of these patents 
 * are general enough such that they are unavoidable regardless of 
 * implementation design. 
 * 
*/ 
 
#include"sim.h" 
 
 
/********************************************************************** 
 * 
 *	Name:         UpsampleReferenceLayerPicture 
 *	Description:  Upsamples the lower resolution base layer image for 
 *                    prediction of enhancement layer images 
 *	 
 *	Input:        base layer image 
 *         
 *	Returns:       
 *	Side effects: allocates memory for enhancement layer (increased 
 *                    resolution) image 
 * 
 *	Date: 970930  Author: Michael Gallant <mikeg@ee.ubc.ca> 
 * 
 ***********************************************************************/ 
 
PictImage *UpsampleReferenceLayerPicture(PictImage *base_image) 
{ 
 
  PictImage *upsampled_image; 
  int size; 
 
  size = enhancement_pels*enhancement_lines; 
 
  if ((upsampled_image = (PictImage *) malloc (sizeof (PictImage))) == NULL) 
  { 
    fprintf (stderr, "Couldn't allocate (PictImage *)\n"); 
    exit (-1); 
  } 
  if ((upsampled_image->lum = (unsigned char *) malloc (sizeof (char) * size)) == NULL) 
  { 
    fprintf (stderr, "Couldn't allocate memory for luminance\n"); 
    exit (-1); 
  } 
  if ((upsampled_image->Cr = (unsigned char *) malloc (sizeof (char) * size / 4)) == NULL) 
  { 
    fprintf (stderr, "Couldn't allocate memory for Cr\n"); 
    exit (-1); 
  } 
  if ((upsampled_image->Cb = (unsigned char *) malloc (sizeof (char) * size / 4)) == NULL) 
  { 
    fprintf (stderr, "Couldn't allocate memory for Cb\n"); 
    exit (-1); 
  } 
 
  UpsampleComponent(base_image->lum, upsampled_image->lum,  
                    base_pels, base_lines); 
  UpsampleComponent(base_image->Cr, upsampled_image->Cr,  
                    base_pels/2, base_lines/2); 
  UpsampleComponent(base_image->Cb, upsampled_image->Cb,  
                    base_pels/2, base_lines/2); 
 
  return (upsampled_image); 
} 
 
/********************************************************************** 
 * 
 *	Name:         UpsampleComponent 
 *	Description:  Upsamples Y, Cb, Cr components in one or both spatial 
 *                    dimensions 
 *	 
 *	Input:        base layer and enhancement layer image and dimensions 
 *                    of base layer image 
 *         
 *	Returns:       
 *	Side effects:  
 * 
 *	Date: 970930  Author: Michael Gallant <mikeg@ee.ubc.ca> 
 * 
 ***********************************************************************/ 
 
void UpsampleComponent (unsigned char *base, unsigned char *enhanced, 
                        int horiz, int vert) 
{ 
  int i,j; 
  unsigned char *base_next, *enhanced_next, *enhanced_origin; 
 
  enhanced_origin = enhanced; 
 
  switch (scalability_mode) 
  { 
    case SPATIAL_SCALABILITY_H: 
 
      /* Rows */ 
      for( j=0; j<vert; j++ ) 
      { 
        /* First column of rows */ 
        *enhanced++ = *base;   
        for( i=1; i<horiz; i++ ) 
        { 
          *enhanced++ = (3* *base +    *(base+1) + 2) >> 2; 
          *enhanced++ = (   *base + 3* *(base+1) + 2) >> 2; 
          base++; 
        } 
        /* Last column of rows */ 
        *enhanced++ = *base++; 
      } 
 
      break; 
 
    case SPATIAL_SCALABILITY_V: 
       
      /* First row */ 
      for( i=0 ; i<horiz; i++ )       
      { 
        *enhanced++ = *base++; 
      } 
       
      enhanced_next = enhanced + horiz; 
      base          = base - horiz; 
      base_next     = base + horiz; 
 
      /* Rows */ 
      for( j=0; j<vert-1; j++ ) 
      { 
         
        for( i=0; i<horiz; i++ ) 
        { 
          *enhanced++      = (3* *base +    *(base_next) + 2) >> 2;   
          *enhanced_next++ = (   *base + 3* *(base_next) + 2) >> 2;   
          base++; 
          base_next++; 
        } 
        enhanced      = enhanced + horiz; 
        enhanced_next = enhanced + horiz; 
      } 
 
      /* Last row */ 
      for( i=0 ; i<horiz; i++ )       
      { 
        *enhanced++ = *base++; 
      } 
       
      break; 
 
    case SPATIAL_SCALABILITY_HV: 
 
      /* Top left corner pel */ 
      *enhanced++  = *base;    
      /* First row */ 
      for( i=1 ; i<horiz; i++ )       
      { 
        *enhanced++ = (3* *base +    *(base+1) + 2) >> 2; 
        *enhanced++ = (   *base + 3* *(base+1) + 2) >> 2; 
        base++ ; 
      } 
      /* Top right corner pel */ 
      *enhanced++ = *base++;    
 
      enhanced_next = enhanced + (horiz<<1); 
      base          = base - horiz; 
      base_next     = base + horiz; 
       
      /* Rows */ 
      for( j=0; j<vert-1; j++ ) 
      { 
        /* First column of rows */ 
        *enhanced++       = (3* *base +    *(base_next) + 2) >> 2;   
        *enhanced_next++  = (   *base + 3* *(base_next) + 2) >> 2; 
        for( i=1; i<horiz; i++ ) 
        { 
          *enhanced++      = (9* *base + 3* *(base+1) + 3* *base_next +     
                                 *(base_next+1) + 8) >> 4; 
          *enhanced++      = (3* *base + 9* *(base+1) +    *base_next +  
                              3* *(base_next+1) + 8) >> 4; 
          *enhanced_next++ = (3* *base +    *(base+1) + 9* *base_next +  
                              3* *(base_next+1) + 8) >> 4; 
          *enhanced_next++ = (   *base + 3* *(base+1) + 3* *base_next +  
                              9* *(base_next+1) + 8) >> 4; 
          base++; 
          base_next++; 
        } 
        /* Last column of rows */ 
        *enhanced++      = (3* *base +    *(base_next) + 2) >> 2;   
        *enhanced_next++ = (   *base + 3* *(base_next) + 2) >> 2;   
 
        enhanced      = enhanced + (horiz<<1); 
        enhanced_next = enhanced + (horiz<<1); 
        base++; 
        base_next++; 
      } 
       
      /* Bottom left corner pel */ 
      *enhanced++  = *base;    
      /* Last row */ 
      for( i=1; i<horiz; i++ )                                 
      { 
        *enhanced++ = (3* *base +    *(base+1) + 2) >> 2 ; 
        *enhanced++ = (   *base + 3* *(base+1) + 2) >> 2 ; 
        base++ ; 
      } 
      /* Bottom right corner pel */ 
      *enhanced = *base; 
 
      break; 
 
    default: 
 
      break; 
  } 
} 
 
/********************************************************************** 
 * 
 *	Name:           GenerateFrameAndInterpolatedImages 
 *	Description:     
 * 
 *	Input:		reconstructed image 
 *	Side effects:   sets out_recon_ipol, out_recon via  
 *                      recon_interpolated and recon_frame 
 *      Return:          
 * 
 *	Date:970926     Michael Gallant <mikeg@ee.ubc.ca> 
 * 
 ***********************************************************************/ 
void GenerateFrameAndInterpolatedImages(PictImage *in_recon, Pict *pic,  
                                        unsigned char **out_recon_ipol,  
                                        PictImage **out_recon,  
                                        unsigned char **recon_interpolated, 
                                        PictImage **recon_frame) 
{ 
 
  int B; 
  unsigned char *recon_lum_frame = NULL; 
 
  /* interpolate image */ 
  if (mv_outside_frame)  
  { 
    if (long_vectors)  
    { 
      /* If the Extended Motion Vector range is used, motion vectors 
         may point further out of the picture than in the normal range, 
         and the edge images will have to be made larger */ 
      B = 16; 
    } 
    else  
    { 
      /* normal range */ 
      B = 8; 
    } 
 
    /* Prepare to store an edge image, including luminance AND  
     * chrominance components. */ 
    *(recon_frame) = InitImage((pels+4*B)*(lines+4*B)); 
 
    MakeEdgeImage(in_recon->lum,(*recon_frame)->lum + (pels+4*B)*2*B + 2*B,pels,lines,2*B); 
 
    *(recon_interpolated) = InterpolateImage((*recon_frame)->lum, pels+4*B, lines+4*B,  
                                          pic->RTYPE); 
    /* prev_ipol points to position in pi where (0,0) of 
     * original resolution would be. */ 
    *(out_recon_ipol) = *(recon_interpolated) + (2*pels + 8*B) * 4*B + 4*B;  
     
    /* Make edge image of non-interpolated, reconstructed, 
     * previous image, luminace and chrominance components. */ 
    MakeEdgeImage(in_recon->Cr,(*recon_frame)->Cr + (pels/2 + 2*B)*B + B,pels/2,lines/2,B); 
    MakeEdgeImage(in_recon->Cb,(*recon_frame)->Cb + (pels/2 + 2*B)*B + B,pels/2,lines/2,B); 
     
    (*out_recon) = (PictImage *)malloc(sizeof(PictImage)); 
    (*out_recon)->lum = (*recon_frame)->lum + (pels + 4*B)*2*B + 2*B; 
    (*out_recon)->Cr = (*recon_frame)->Cr + (pels/2 + 2*B)*B + B; 
    (*out_recon)->Cb = (*recon_frame)->Cb + (pels/2 + 2*B)*B + B; 
  } 
  else  
  { 
    /* P-pictures. */ 
    *(recon_interpolated) = InterpolateImage(in_recon->lum,pels,lines,pic->RTYPE); 
    *(out_recon_ipol) = *(recon_interpolated); 
    *(out_recon) = in_recon; 
  } 
} 
 
/********************************************************************** 
 * 
 *	Name:        ReconImage 
 *	Description:	Puts together reconstructed image 
 *	 
 *	Input:        position of curr block, reconstructed 
 *        macroblock, pointer to recontructed image 
 *	Returns: 
 *	Side effects: 
 * 
 *	Date: 930123        Author: Karl.Lillevold@nta.no 
 * 
 ***********************************************************************/ 
 
void ReconImage (int i, int j, MB_Structure *data, PictImage *recon) 
{ 
  int n; 
  register int m; 
 
  int x_curr, y_curr; 
 
  x_curr = i * MB_SIZE; 
  y_curr = j * MB_SIZE; 
 
  /* Fill in luminance data */ 
  for (n = 0; n < MB_SIZE; n++) 
    for (m= 0; m < MB_SIZE; m++) { 
      *(recon->lum + x_curr+m + (y_curr+n)*pels) = data->lum[n][m]; 
    } 
 
  /* Fill in chrominance data */ 
  for (n = 0; n < MB_SIZE>>1; n++) 
    for (m = 0; m < MB_SIZE>>1; m++) { 
      *(recon->Cr + (x_curr>>1)+m + ((y_curr>>1)+n)*cpels) = data->Cr[n][m]; 
      *(recon->Cb + (x_curr>>1)+m + ((y_curr>>1)+n)*cpels) = data->Cb[n][m]; 
    } 
  return; 
} 
 
/********************************************************************** 
 * 
 *	Name:        InterpolateImage 
 *	Description:    Interpolates a complete image for easier half 
 *                      pel prediction 
 *	 
 *	Input:	        pointer to image structure 
 *	Returns:        pointer to interpolated image 
 *	Side effects:   allocates memory to interpolated image 
 * 
 *	Date: 950207        Author: Karl.Lillevold@nta.no 
 * 
 ***********************************************************************/ 
 
 
unsigned char *InterpolateImage(unsigned char *image, int width, int height, int RTYPE) 
{ 
  unsigned char *ipol_image, *ii, *oo; 
  int i,j; 
 
  ipol_image = (unsigned char *)malloc(sizeof(char)*width*height*4); 
  ii = ipol_image; 
  oo = image; 
 
  /* main image */ 
  for (j = 0; j < height-1; j++) { 
    for (i = 0; i  < width-1; i++) { 
      *(ii + (i<<1)) = *(oo + i); 
      *(ii + (i<<1)+1) = (*(oo + i) + *(oo + i + 1) + 1 - RTYPE)>>1; 
      *(ii + (i<<1)+(width<<1)) = (*(oo + i) + *(oo + i + width) + 1 - RTYPE)>>1; 
      *(ii + (i<<1)+1+(width<<1)) = (*(oo+i) + *(oo+i+1) +  
         *(oo+i+width) + *(oo+i+1+width) + 2 - RTYPE)>>2; 
    } 
    /* last pels on each line */ 
    *(ii+ (width<<1) - 2) = *(oo + width - 1); 
    *(ii+ (width<<1) - 1) = *(oo + width - 1); 
    *(ii+ (width<<1)+ (width<<1)-2) = (*(oo+width-1)+*(oo+width+width-1)+1-RTYPE)>>1; 
    *(ii+ (width<<1)+ (width<<1)-1) = (*(oo+width-1)+*(oo+width+width-1)+1-RTYPE)>>1; 
    ii += (width<<2); 
    oo += width; 
  } 
 
  /* last lines */ 
  for (i = 0; i < width-1; i++) { 
    *(ii+ (i<<1)) = *(oo + i);     
    *(ii+ (i<<1)+1) = (*(oo + i) + *(oo + i + 1) + 1 - RTYPE)>>1; 
    *(ii+ (width<<1)+ (i<<1)) = *(oo + i);     
    *(ii+ (width<<1)+ (i<<1)+1) = (*(oo + i) + *(oo + i + 1) + 1 - RTYPE)>>1; 
           
  } 
 
  /* bottom right corner pels */ 
  *(ii + (width<<1) - 2) = *(oo + width -1); 
  *(ii + (width<<1) - 1) = *(oo + width -1); 
  *(ii + (width<<2) - 2) = *(oo + width -1); 
  *(ii + (width<<2) - 1) = *(oo + width -1); 
 
  return ipol_image; 
} 
 
/********************************************************************** 
 * 
 *	Name:        MakeEdgeImage 
 *	Description:    Copies edge pels for use with unrestricted 
 *                      motion vector mode 
 *	 
 *	Input:	        pointer to source image, destination image 
 *                      width, height, edge 
 *	Returns:        
 *	Side effects: 
 * 
 *	Date: 950219        Author: Karl.Lillevold@nta.no 
 * 
 ***********************************************************************/ 
 
void MakeEdgeImage(unsigned char *src, unsigned char *dst, int width, 
           int height, int edge) 
{ 
  int i,j; 
  unsigned char *p1,*p2,*p3,*p4; 
  unsigned char *o1,*o2,*o3,*o4; 
 
  /* center image */ 
  p1 = dst; 
  o1 = src; 
  for (j = 0; j < height;j++) { 
    memcpy(p1,o1,width); 
    p1 += width + (edge<<1); 
    o1 += width; 
  } 
 
  /* left and right edges */ 
  p1 = dst-1; 
  o1 = src; 
  for (j = 0; j < height;j++) { 
    for (i = 0; i < edge; i++) { 
      *(p1 - i) = *o1; 
      *(p1 + width + i + 1) = *(o1 + width - 1); 
    } 
    p1 += width + (edge<<1); 
    o1 += width; 
  }     
     
  /* top and bottom edges */ 
  p1 = dst; 
  p2 = dst + (width + (edge<<1))*(height-1); 
  o1 = src; 
  o2 = src + width*(height-1); 
  for (j = 0; j < edge;j++) { 
    p1 = p1 - (width + (edge<<1)); 
    p2 = p2 + (width + (edge<<1)); 
    for (i = 0; i < width; i++) { 
      *(p1 + i) = *(o1 + i); 
      *(p2 + i) = *(o2 + i); 
    } 
  }     
 
  /* corners */ 
  p1 = dst - (width+(edge<<1)) - 1; 
  p2 = p1 + width + 1; 
  p3 = dst + (width+(edge<<1))*(height)-1; 
  p4 = p3 + width + 1; 
 
  o1 = src; 
  o2 = o1 + width - 1; 
  o3 = src + width*(height-1); 
  o4 = o3 + width - 1; 
  for (j = 0; j < edge; j++) { 
    for (i = 0; i < edge; i++) { 
      *(p1 - i) = *o1; 
      *(p2 + i) = *o2; 
      *(p3 - i) = *o3; 
      *(p4 + i) = *o4;  
    } 
    p1 = p1 - (width + (edge<<1)); 
    p2 = p2 - (width + (edge<<1)); 
    p3 = p3 + width + (edge<<1); 
    p4 = p4 + width + (edge<<1); 
  } 
} 
 
/********************************************************************** 
 * 
 *	Name:		ReadImage 
 *	Description:	Reads one qcif image from disk 
 * 
 *	Input:		filename of sequence, frame no. to be read, 
 *			headerlength of sequence 
 *	Returns:	Pointer to start of raw YUV-data 
 *	Side effects:	Memory allocated to image-data 
 * 
 *	Date: 940108	Author:	Karl.Lillevold@nta.no 
 * 
 *      Modified:       guyc@ee.ubc.ca 8/97 to read yuv files if necessary 
 * 
 ***********************************************************************/ 
 
unsigned char * 
 ReadImage (char *filename, int frame_no, int headerlength) 
{ 
  FILE *im_file = NULL; 
  int im_size = pels * lines * 3 / 2; 
  unsigned char *qcif; 
  int status; 
  char YUVname[255], tmp[255]; 
  FILE *YUVfile; 
 
 
  if ((qcif = (unsigned char *) malloc (sizeof (char) * im_size)) == NULL) 
  { 
    fprintf (stderr, "Couldn't allocate memory to image\n"); 
    exit (-1); 
  } 
  if ((im_file = fopen (filename, "rb")) == NULL) 
  { 
 
    /* Try using separate file names (yuv files) */ 
    sprintf (YUVname, filename, frame_no); 
    strcpy (tmp, YUVname); 
    sprintf (YUVname, "%s.yuv", tmp); 
 
    if ((YUVfile = fopen (YUVname, "rb")) == NULL) 
    { 
      fprintf (stderr, "Unable to open image_file yuv: %s\n", YUVname); 
      exit (-1); 
    } 
    fprintf (stdout, "Reading image no: %d\n", frame_no); 
    if ((status = fread (qcif, sizeof (char), im_size, YUVfile)) != im_size) 
    { 
      fclose (YUVfile); 
      fprintf (stderr, "Error in reading yuv image from %s \n", YUVname); 
      exit (-1); 
    } 
    fclose (YUVfile); 
 
  } else 
  { 
 
    rewind (im_file); 
    /* Find the correct image */ 
    status = fseek (im_file, (headerlength + frame_no * im_size), 0); 
    if (status != 0) 
    { 
      fprintf (stderr, "Error in seeking image no: %d\n", frame_no); 
      fprintf (stderr, "From file: %s\n", filename); 
      exit (-1); 
    } 
    /* Read image */ 
    fprintf (stdout, "Reading image no: %d\n", frame_no); 
    if ((status = fread (qcif, sizeof (char), im_size, im_file)) != im_size) 
    { 
      fprintf (stderr, "Error in reading image no: %d\n", frame_no); 
      fprintf (stderr, "From file: %s\n", filename); 
      exit (-1); 
    } 
    fclose (im_file); 
  } 
  return (qcif); 
} 
 
 
/********************************************************************** 
 * 
 *	Name:		FillImage 
 *	Description:	fills Y, Cb and Cr of a PictImage struct 
 * 
 *	Input:		pointer to raw image 
 * 
 *	Returns:	pointer to filled PictImage 
 *	Side effects:	allocates memory to PictImage 
 *                      raw image is freed 
 * 
 *	Date: 940109	Author:	Karl.Lillevold@nta.no 
 * 
 ***********************************************************************/ 
 
PictImage * 
 FillImage (unsigned char *in) 
{ 
  PictImage *Pict; 
 
  Pict = InitImage (pels * lines); 
 
  memcpy (Pict->lum, in, pels * lines); 
  memcpy (Pict->Cb, (in + pels * lines), pels * lines / 4); 
  memcpy (Pict->Cr, (in + pels * lines + pels * lines / 4), pels * lines / 4); 
 
  free (in); 
  return (Pict); 
} 
 
 
/********************************************************************** 
 * 
 *	Name:		WriteImage 
 *	Description:	Writes PictImage struct to disk 
 * 
 *	Input:		pointer to image data to be stored, filename 
 *			to be used on the disk, image size 
 *	Returns: 
 *	Side effects: 
 * 
 *	Date: 930115	Author: Karl.Lillevold@nta.no 
 * 
 ***********************************************************************/ 
 
void 
 WriteImage (PictImage * image, char *filename) 
{ 
  int status; 
  FILE *f_out; 
 
  /* Opening file */ 
  if ((f_out = fopen (filename, "ab")) == NULL) 
  { 
    fprintf (stderr, "%s%s\n", "Error in opening file: ", filename); 
    exit (-1); 
  } 
  /* Writing lum to file */ 
  if ((status = fwrite (image->lum, sizeof (char), pels * lines, f_out)) != pels * lines) 
  { 
    fprintf (stderr, "%s%s\n", "Error in writing to file: ", filename); 
    exit (-1); 
  } 
  /* Writing Cb to file */ 
  if ((status = fwrite (image->Cb, sizeof (char), pels * lines / 4, f_out)) != pels * lines / 4) 
  { 
    fprintf (stderr, "%s%s\n", "Error in writing to file: ", filename); 
    exit (-1); 
  } 
  /* Writing Cr to file */ 
  if ((status = fwrite (image->Cr, sizeof (char), pels * lines / 4, f_out)) != pels * lines / 4) 
  { 
    fprintf (stderr, "%s%s\n", "Error in writing to file: ", filename); 
    exit (-1); 
  } 
  fclose (f_out); 
  return; 
} 
 
 
/********************************************************************** 
 * 
 *	Name:		InitImage 
 *	Description:	Allocates memory for structure of 4:2:0-image 
 * 
 *	Input:	        image size 
 *	Returns:	pointer to new structure 
 *	Side effects:	memory allocated to structure 
 * 
 *	Date: 930115  	Author: Karl.Lillevold@nta.no 
 * 
 ***********************************************************************/ 
 
PictImage * 
 InitImage (int size) 
{ 
  PictImage *new; 
 
  if ((new = (PictImage *) malloc (sizeof (PictImage))) == NULL) 
  { 
    fprintf (stderr, "Couldn't allocate (PictImage *)\n"); 
    exit (-1); 
  } 
  if ((new->lum = (unsigned char *) malloc (sizeof (char) * size)) == NULL) 
  { 
    fprintf (stderr, "Couldn't allocate memory for luminance\n"); 
    exit (-1); 
  } 
  if ((new->Cr = (unsigned char *) malloc (sizeof (char) * size / 4)) == NULL) 
  { 
    fprintf (stderr, "Couldn't allocate memory for Cr\n"); 
    exit (-1); 
  } 
  if ((new->Cb = (unsigned char *) malloc (sizeof (char) * size / 4)) == NULL) 
  { 
    fprintf (stderr, "Couldn't allocate memory for Cb\n"); 
    exit (-1); 
  } 
  return (new); 
} 
 
 
/********************************************************************** 
 * 
 *	Name:		FreeImage 
 *	Description:	Frees memory allocated to structure of 4:2:0-image 
 * 
 *	Input:		pointer to structure 
 *	Returns: 
 *	Side effects:	memory of structure freed 
 * 
 *	Date: 930115	Author: Karl.Lillevold@nta.no 
 * 
 ***********************************************************************/ 
 
void 
 FreeImage (PictImage * image) 
{ 
  free (image->lum); 
  free (image->Cr); 
  free (image->Cb); 
  free (image); 
} 
