/* 
 **********************************************************************
 *
 * File Name:		seqpsnr.c
 * 
 * Designed by:		Jie Liang
 *
 * Created Date:	10/07/2000
 *
 * Contents:
 *			Calculate the PSNR between two qcif or cif files.
 *
 * Usage:
 *			seqpsnr type imgfile1 frame1
 *			where:
 *                              type:     0 for qcif, 1 for cif;
 *				imgfile1: input file 1,
 *				imgfile2: input file 2,
 *				
 *			
 * Copyright (C) 2000 Department of Electrical and Computer Engineering 
 * The Johns Hopkins University. All right reserved. 
 *
 *********************************************************************
 */

/*
 *********************************************************************
 *
 *   Modification History:
 *
 *   Date	Programmer	Description
 *-------------------------------------------------------------------
 *
 * 10/07/2000	Jie Liang	Created.
 * 
 * $Log$
 *
 *********************************************************************
 */

static const char rcsid[] = 
	"$Id$";

/*
 ********************************************************************
 *
 *	Include Files
 *
 ********************************************************************
 */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

/*
 *******************************************************************
 *
 * 	Local Constants
 *
 *******************************************************************
 */

/*
 *******************************************************************
 *
 *	Local Macros
 *
 *******************************************************************
 */

/*
 *******************************************************************
 *
 *	Local Enumerated Types
 *
 *******************************************************************
 */

/*
 *******************************************************************
 *
 *	Local Data Types
 *
 *******************************************************************
 */
typedef struct statistics { 
  double psnr; 
  int    mad; 
} STATISTICS; 

/*
 *******************************************************************
 *
 *	Variables
 *
 *******************************************************************
 */

/*
 *******************************************************************
 *
 *	Local Function Prototypes
 *
 *******************************************************************
 */
void getpsnr( unsigned char *imgptr1, unsigned char *imgptr2, 
		unsigned int hsize, unsigned int vsize, STATISTICS *outptr);

/*
 ***********************************************************************
 *
 * Function Name:		main
 *
 * Descriptions:
 *
 * Input Parameters:
 *				argc: argument count;
 *				argv: array of input arguments.
 * Output Parameters:		0:    successful,
 *				-1:   error.
 * Limitations:
 *
 ************************************************************************
 */
 
int main(int argc, char *argv[])
{
  FILE 	*imgfile[2], *outfile[3];
  int	filetype, hsize, vsize;
  unsigned int	i, framecnt;
  unsigned char *imgdata[2] = {0, 0};
  STATISTICS y_sta, u_sta, v_sta;
  int tmp1, tmp2;
  double y_ave = 0, u_ave = 0, v_ave = 0;

  if (argc != 7) {
	printf(
	        "\nCalculate the PSNR of two qcif/cif files:\n"
		"Usage:\n seqpsnr type imgfile1 imgfile2 outfile1 outfile2 outfile3\n"
		"Arguments:\n"
		"\ttype:\t\t0 forqcif 1 for cif;\n"
		"\timgfile1:\tinput file 1,\n"
		"\timgfile2:\tinput file 2,\n"
		"\toufile1:\toutput file for Y PSNR,\n"
		"\toufile2:\toutput file for U PSNR,\n"
		"\toufile3:\toutput file for V PSNR,\n"
		"Examples:\n"
		"	./seqpsnr 0 glasgo.qcif glasgo_rec.qcif y.m u.m v.m\n\n"
	);	
	return(-1);
  }
 
  filetype = atoi(argv[1]);
  if ((filetype < 0) || (filetype > 1)) {
	printf("Filetype shoudl be 0 or 1.\n");
	return(-1);
  }
  if (filetype) {
    hsize = 352;
    vsize = 288;
  } else {
    hsize = 176;
    vsize = 144;
  }
 
  /* Open Input files */
  for (i = 0; i < 2; i++){
    imgfile[i] = fopen(argv[i + 2], "r");
    if (imgfile[i] == NULL)  {
      printf("Input file not found.\n");
      return(-1);
    }
  }

  /* create output files */
  for (i = 0; i < 3; i++){
    outfile[i] = fopen(argv[i + 4], "w");
    if (outfile[i] == NULL)  {
      printf("Couldn't create output file.\n");
      return(-1);
    }
  }

  //allocate image buffer
  for (i = 0; i < 2; i++) {	
    imgdata[i] = (unsigned char *) malloc(hsize * vsize);
    if (imgdata[i] == NULL) {
      printf("Memory allocation error.\n");
      return(-1);
    }
  }

  /* read in image frames */

  fprintf(outfile[0], "y_psnr = [ ");
  fprintf(outfile[1], "u_psnr = [ ");
  fprintf(outfile[2], "v_psnr = [ ");
  
  framecnt = 0;
  while (1) {
    ///////////////////
    //read Y components
    ///////////////////
    tmp1 = (int) fread(imgdata[0], sizeof(unsigned char), hsize * vsize, imgfile[0]);
    tmp2 = (int) fread(imgdata[1], sizeof(unsigned char), hsize * vsize, imgfile[1]);
    if (!tmp1 || !tmp2){
      break;
    }

    getpsnr(imgdata[0], imgdata[1], hsize, vsize, &y_sta);

    ///////////////////
    //read U components
    ///////////////////
    tmp1 = (int) fread(imgdata[0], sizeof(unsigned char), hsize * vsize / 4, imgfile[0]);
    tmp2 = (int) fread(imgdata[1], sizeof(unsigned char), hsize * vsize / 4, imgfile[1]);
    if (!tmp1 || !tmp2){
      break;
    }
    getpsnr(imgdata[0], imgdata[1], hsize / 2, vsize / 2, &u_sta);

    ///////////////////
    //read V components
    ///////////////////
    tmp1 = (int) fread(imgdata[0], sizeof(unsigned char), hsize * vsize / 4, imgfile[0]);
    tmp2 = (int) fread(imgdata[1], sizeof(unsigned char), hsize * vsize / 4, imgfile[1]);
    if (!tmp1 || !tmp2){
      break;
    }
    getpsnr(imgdata[0], imgdata[1], hsize / 2, vsize / 2, &v_sta);

    fprintf(stdout, "%f  %f  %f\t\t%d  %d  %d\n", y_sta.psnr, u_sta.psnr, v_sta.psnr, y_sta.mad, u_sta.mad, v_sta.mad );

    fprintf(outfile[0], "%f  ", y_sta.psnr);
    fprintf(outfile[1], "%f  ", u_sta.psnr);
    fprintf(outfile[2], "%f  ", v_sta.psnr);
    
    //prepare for average.
    y_ave += y_sta.psnr;
    u_ave += u_sta.psnr;
    v_ave += v_sta.psnr;
    
    framecnt++;
    if (framecnt % 5 == 0) {
      fprintf(outfile[0], "...\n");  //Matlab format
      fprintf(outfile[1], "...\n");
      fprintf(outfile[2], "...\n");
   }

  }

  fprintf(stdout, "The above are PSNR of Y, U, V, and MAD of Y, U, V.\n");

  fprintf(outfile[0], "];\n ");
  fprintf(outfile[1], "];\n ");
  fprintf(outfile[2], "];\n ");

  //calculate average PSNR
  y_ave = y_ave / framecnt;
  u_ave = u_ave / framecnt;
  v_ave = v_ave / framecnt;

  fprintf(stdout, "Average:\n%f, %f, %f\n", y_ave, u_ave, v_ave);
  fprintf(outfile[0], "%%Average:   %f, %f, %f\n", y_ave, u_ave, v_ave);

  fclose(imgfile[0]);
  fclose(imgfile[1]);

  fclose(outfile[0]);
  fclose(outfile[1]);
  fclose(outfile[2]);

  free(imgdata[0]);
  free(imgdata[1]);


  return(0);   
}


/*
 *******************************************************************************
 *
 * Function Name:	getpsnr
 *
 * Descriptions:
 *			Calculate the Peak SNR between two images.
 * Input Parameters:
 *			imgptr1:	pointer to the first image,
 *			imgptr2:	pointer to the second image,
 *			hsize:		Horizontal size,
 *			vsize:		Vertical size,
 *			maxval:		Maximum pixel value, such as 255.
 * Output Parameters:
 *			psnr:		Peak SNR.
 * Limitations:
 *
 ********************************************************************************
 */
void getpsnr( unsigned char *imgptr1, unsigned char *imgptr2, 
		unsigned int hsize, unsigned int vsize, STATISTICS *outptr) {
  double mse, psnr, err;
  int    i, pixels, mad;
  
  pixels = hsize * vsize;
  mse = 0;
  mad = 0;

  for (i =0; i < pixels; i++) {
    err = *(imgptr1 + i) - *(imgptr2 + i);

    //test
    //    if (err) {
    //fprintf(stdout, "%d --> %d\n", *(imgptr1 + i), *(imgptr2 + i));
    //    }

    mse = mse + err * err; 

    if (abs(err) > mad ) {
      mad = abs(err);
    }
  }
  mse = mse / pixels;

  outptr->psnr = 10 * log10( 255 * 255 / mse);
  outptr->mad  = mad;
  
  return;
}


/*
 *************************************************************************
 *
 * End of $Source$
 *
 **************************************************************************
 */
