//=============================================================================
// File: gnutbinarymsgbc.h
// Author: Andre Dufour
//
// Class type: abstract base
// Description: superclass for all gnutella binary messages (i.e. those 
//              exchanged once a gnutella connection has been established
//              between peers).
//=============================================================================

#ifndef __GNUTBINARYMSGBC_H__
#define __GNUTBINARYMSGBC_H__

#include "ignutmsg.h"
#include "gnut_types.h"
#include "gnutmsgparser.h"
#include "gnutcoord.h"

//=============================================================================

// TODO: AD - don't inline member functions of this struct.
struct GnutDescHdr
{
    GnutMsgParser::EGnutMsgType         payload_desc_;
    DescriptorID_t                      id_;
    Byte_t                              ttl_;
    Byte_t                              hops_;
    NodeAddr_t                          initialSender_;

    // Build empty structure to be filled later.
    GnutDescHdr(void)
    : payload_desc_(GnutMsgParser::GNUT_MSG_LAST),
      ttl_(0),
      hops_(0),
      initialSender_(0)
    { }

    // Build structure from user data
    GnutDescHdr(NodeAddr_t aInitialSender, DescriptorID_t aId, 
                GnutMsgParser::EGnutMsgType aType, 
                Byte_t aTtl, Byte_t aHops)
    : payload_desc_(aType), id_(aId), ttl_(aTtl), hops_(aHops),
      initialSender_(aInitialSender)
    { }

    GnutDescHdr& operator=(const GnutDescHdr& aHdr)
    {
        if (&aHdr != this)
        {
            this->id_               = aHdr.id_; 
            this->payload_desc_     = aHdr.payload_desc_;
            this->ttl_              = aHdr.ttl_;
            this->hops_             = aHdr.hops_;
            this->initialSender_    = aHdr.initialSender_;
        }

        return *this;
    }
};
//=============================================================================
class GnutBinaryMsgBC : public IGnutMsg
{
protected:
    // = FOUNDATION

    // Build structure from user input
    GnutBinaryMsgBC(NodeAddr_t aInitialSender,
                    DescriptorID_t aId, GnutMsgParser::EGnutMsgType aType, 
                    Byte_t aTtl, Byte_t aHops, const GnutCoord& aCoord,
                    double aError);

    // Build structure from packet.
    GnutBinaryMsgBC(const void* aMem);

    // Make class abstact
    virtual ~GnutBinaryMsgBC(void) = 0;


    // = IGnutMsg

    virtual void writeToPacket(void* aMem) const;
    

    // = ACCESS
public:
    // Get length of the binarymsg header in packet.
    virtual size_t getSize(void) const;

    const GnutDescHdr& getHeader(void) const { return hdr_; }

    const GnutCoord& getCoordinates(void) const { return coord_; }

    double getError(void) const { return error_; }

    double getTimeSent(void) const { return timeSent_; }


private:
    // = DATA

    GnutDescHdr         hdr_;
    // Gnutella common header

    GnutCoord           coord_;
    // The Vivaldi coordinates of the node sending the message

    double              error_;
    // The error on the coordinates as estimated by the originating node

    double              timeSent_;
    // The time the message was first sent

    // = FOUNDATION

    // Disable copy constructor and assignment operator
    GnutBinaryMsgBC(const GnutBinaryMsgBC&);
    GnutBinaryMsgBC& operator=(const GnutBinaryMsgBC&);
};
//=============================================================================
#endif /* __GNUTBINARYMSGBC_H__ */

