//=============================================================================
// File: gnutcoord.cc
// Author: Andre Dufour
//=============================================================================

#include "gnutcoord.h"
#include "gnut_util.h"
#include <cstdlib>
#include <cmath>

//=============================================================================
GnutCoord::GnutCoord(void)
: mX(0.0),
  mY(0.0),
  mHeight(0.0)
{ }
//=============================================================================
GnutCoord::GnutCoord(double aX, double aY, double aHeight)
: mX(aX),
  mY(aY),
  mHeight(aHeight)
{ }
//=============================================================================
GnutCoord::GnutCoord(double aLength)
{
    double x      = rand();
    double y      = rand();
    double height = rand();

    double length = sqrt( pow(x,2) + pow(y,2) + pow(height,2) );
    double scale  = aLength / length;

    mX      = x * scale;
    mY      = y * scale;
    mHeight = height * scale;
}
//=============================================================================
GnutCoord::GnutCoord(double aLength, const GnutCoord& aDirection)
{
    double x, y, height;

    if (aDirection.abs() != 0.0)
    {
        x      = aDirection.mX;
        y      = aDirection.mY;
        height = aDirection.mHeight;
    }
    else
    {
        // If the direction is (0,0,0), we pick an arbitrary direction.
        // This is like GnutCoord(double aLength)
        x      = rand();
        y      = rand();
        height = rand();
    }

    double length = sqrt( pow(x,2) + pow(y,2) + pow(height,2) );
    double scale  = aLength / length;

    mX      = x * scale;
    mY      = y * scale;
    mHeight = height * scale;
}
//=============================================================================
GnutCoord::~GnutCoord(void)
{ }
//=============================================================================
double
GnutCoord::abs(void) const
{
    return sqrt( pow(mX,2) + pow(mY,2) + pow(mHeight,2) );
}
//=============================================================================
GnutCoord& 
GnutCoord::operator+=(const GnutCoord& aRhs)
{
    this->mX      += aRhs.mX;
    this->mY      += aRhs.mY;
    this->mHeight += aRhs.mHeight;

    return *this;
}
//=============================================================================
GnutCoord& 
GnutCoord::operator-=(const GnutCoord& aRhs)
{
    this->mX      -= aRhs.mX;
    this->mY      -= aRhs.mY;

    // Height is special. The difference between two nodes has a height
    // component which is the sum of the heights.
    this->mHeight += aRhs.mHeight;

    return *this;
}
//=============================================================================
GnutCoord&
GnutCoord::operator*=(double aFactor)
{
    mX      *= aFactor;
    mY      *= aFactor;
    mHeight *= aFactor;

    return *this;
}
//=============================================================================
// Non class-member operators
//=============================================================================
GnutCoord operator+(const GnutCoord& aLhs, const GnutCoord& aRhs)
{
    return GnutCoord(aLhs.mX + aRhs.mX,
                     aLhs.mY + aRhs.mY,
                     aLhs.mHeight + aRhs.mHeight);
}
//=============================================================================
GnutCoord operator-(const GnutCoord& aLhs, const GnutCoord& aRhs)
{
    // Height is special. The difference between two nodes has a height
    // component which is the sum of the heights.

    return GnutCoord(aLhs.mX - aRhs.mX,
                     aLhs.mY - aRhs.mY,
                     aLhs.mHeight + aRhs.mHeight);
}
//=============================================================================
GnutCoord operator*(const GnutCoord& aCoord, double aFactor)
{
    return GnutCoord(aCoord.mX * aFactor, 
                     aCoord.mY * aFactor,  
                     aCoord.mHeight * aFactor);
}
//=============================================================================
GnutCoord operator*(double aFactor, const GnutCoord& aCoord)
{
    return GnutCoord(aCoord.mX * aFactor, 
                     aCoord.mY * aFactor,  
                     aCoord.mHeight * aFactor);
}
//=============================================================================
bool operator==(const GnutCoord& aLhs, const GnutCoord& aRhs)
{
    return aLhs.mX == aRhs.mX &&
           aLhs.mY == aRhs.mY &&
           aLhs.mHeight == aRhs.mHeight;
}
//=============================================================================
bool operator!=(const GnutCoord& aLhs, const GnutCoord& aRhs)
{
    return aLhs.mX != aRhs.mX ||
           aLhs.mY != aRhs.mY ||
           aLhs.mHeight != aRhs.mHeight;
}
//=============================================================================
std::ostream& operator<<(std::ostream& aOs, const GnutCoord& aCoord) 
{
    return aOs << "(" << aCoord.mX << ", " << aCoord.mY << ", " 
               << aCoord.mHeight << ")";
}
//=============================================================================
