// File: gnutellaagent.cc
// Original by Qi He
// Modified by Andre Dufour

#include "gnutellaagent.h"
#include "gnut_types.h"
#include "gnut_util.h"
#include "basicstat.h"
#include "gnutellaapp.h"
#include "peeragent.h"
#include "gnut_util.h"
#include "pendingreqentry.h"
#include "conntimer.h"
#include "gc.h"
#include "gnutmsgparser.h"
#include "gnutbinarymsgbc.h"
#include "gnutpayload.h"
#include "gnutpingmsg.h"
#include "gnutquerymsg.h"
#include "gnutqueryhitmsg.h"
#include "gnutokmsg.h"
#include "gnutrejmsg.h"
#include "gnutbootcacheupdatemsg.h"
#include "gnutlegacyconnectmsg.h"
#include "gnutbootstrapmsg.h"
#include "ultrapeer.h"
#include "gnutstats.h"
#include "ivivaldimanager.h"
#include "gnutcoord.h"

//============================================================================
const NodeAddr_t GnutellaAgent::ALL_NEIGHBOURS = -1;
//============================================================================
/****  GnutellaAgent ****/
GnutellaAgent::
GnutellaAgent(NodeAddr_t aAddr)
: PeerAgent(aAddr),
  conn_timer_(this),
  gapp_(0),
  gc_(new GC(this)), // TODO: AD shouldn't we have a gc in the other c'tor too?
  rate_limit_(0),
  use_prio_(0),
  bkblock_(aAddr),
  fwblock_(aAddr),
  rcvRate_(aAddr),
  sec_(0),
  secRcv_(0)
{
    bind ("rate_limit_", &rate_limit_);
    bind ("use_prio_", &use_prio_);
}
//============================================================================
GnutellaAgent::
GnutellaAgent(GnutellaApp* aApp)
: PeerAgent(aApp), 
  conn_timer_(this),
  gapp_(aApp),
  gc_(0), // TODO: AD this was the default. is it right?
  rate_limit_(0),
  use_prio_(0),
  bkblock_(aApp->addr_),
  fwblock_(aApp->addr_),
  rcvRate_(aApp->addr_),
  sec_(0),
  secRcv_(0)
{
    bind ("rate_limit_", &rate_limit_);
    bind ("use_prio_", &use_prio_);
}
//============================================================================
GnutellaAgent::~GnutellaAgent(void)
{
    // TODO: AD make destructors for these guys too.
    GnutUtil::gnutTrace("GnutellaAgent::~GnutellaAgent()\n");
    (void)cleanup();
    delete gc_;
}
//============================================================================
// upcalls from NSSocket
/* passive connection established */
void GnutellaAgent::upcall_passconn(Socket *sock) {
  GNUT_ASSERT(gapp_ != 0);
  if(gapp_->isBootserver_) {
    NodeAddr_t src = sock->peer_.addr_;
    SockMap_t::iterator si = lsocks_.find(src);
    if(si==lsocks_.end()) {
        SockEntry *newentry = new SockEntry(sock, src, SOCK_BOOT);
        lsocks_.insert(SockMap_t::value_type(src, *newentry));
    } 
  }
}

//============================================================================
/*outstanding socket connection request accepted */
void GnutellaAgent::upcall_connected(Socket *sock) {
  NodeAddr_t src = sock->peer_.addr_;

  SockMap_t::iterator si = lsocks_.find(src);
  if(si!=lsocks_.end() && si->second.type_==SOCK_BOOT_WAIT) {
    si->second.type_ = SOCK_BOOT;
    lime_bootstrap(sock);
    return;
  }

  gnutella_req(sock);
  PendingConns_t::iterator centry = conn_pending_.find(sock->peer_.addr_);
  if(centry==conn_pending_.end()) {
    debug_warning("WARNING: upcall_connected() for non pending socket on %d\n", app_->addr_);
    return; 
  } 

  debug_info("upcall_connected() for pending socket with %d on %d\n", sock->peer_.addr_, app_->addr_);

  centry->second.state_ = TRANSPORT_CONNECTED;
}

//============================================================================
/* socket being closed by peer */
void GnutellaAgent::upcall_closing(Socket *sock) {
  SockMap_t::iterator si = lsocks_.find(sock->peer_.addr_);

  if(si!=lsocks_.end()) {
    SockEntry *cur;

    for(SockMap_t::iterator i=lsocks_.begin(); i!=lsocks_.end(); i++) {
      if(i->second.peer_ == sock->peer_.addr_) {
	lsocks_.erase(i);
	gapp_->ConnectionClosed(sock->peer_.addr_);
	break;
      }
    }
  }
}

//============================================================================
/* socket ready for sending */
void GnutellaAgent::upcall_send(Socket *sock) {
  SockMap_t::iterator si = lsocks_.find(sock->peer_.addr_);
  Socket *tsock=NULL;

  if(si!=lsocks_.end()) {
    SockEntry *cur;
    
    for(SockMap_t::iterator i=lsocks_.begin(); i!=lsocks_.end(); i++) {
      tsock = i->second.sock_;

      if(tsock != sock && !tsock->blocked(SOCK_READ)) {
	i->second.sock_->recv();
      }
    }
  }
}

//============================================================================
/* data packet received */
int GnutellaAgent::upcall_recv(Socket *sock, PacketData *pktdata, Handler *) 
{
    if (gapp_ == 0 || app_ == 0)
    {
        NodeAddr_t src = sock->peer_.addr_;
        GnutUtil::gnutTrace("Null app in %p. Returning from upcall_recv."
                            " after getting msg from %d\n", this, src);
        return TCL_OK;
    }

    GNUT_ASSERT(gapp_ != 0);
    GNUT_ASSERT(app_ != 0);

    //continue only if peer is currently online 
    //(bootstrap server is never in ONLINE state)
    if(app_->state_==PS_OFFLINE && !gapp_->isBootserver_) 
    {
        GnutUtil::gnutTrace("%d is offline and dropping pkt\n", gapp_->addr_);
        return pktdata->size();
    }  

    int block = FALSE;
    GnutMsgParser& parser = GnutMsgParser::instance();
    int res;
    long cur = (long)NOW;


    if(sec_!=cur) 
    {
        rcvRate_.increment2(secRcv_);
        sec_ = cur;
        secRcv_ = 0;
    }
    secRcv_ ++;

    NodeAddr_t src = sock->peer_.addr_;
    GnutDescHdr header;
    GnutPayload payload;
    GnutCoord remote_coord;
    double remote_error;
    double time_sent;
    const double current_time = GnutUtil::now();
    GnutMsgParser::EGnutMsgType msg = parser.parse(pktdata->data(), &header, 
                                                   &payload, &remote_coord,
                                                   &remote_error, &time_sent);
    GNUT_ASSERT(current_time > time_sent);


    NodeAddr_t initialSender = 0;
    if (GnutMsgParser::isBinaryMsg(msg))
    {
        initialSender = header.initialSender_;
    }
    else
    {
        // Non binary messages don't have the header structure filled out.
        // The initial sender is just the immediate neighbour.
        initialSender = src;
    }
    // Update the vivaldi coordinates!
    GnutUtil::gnutTrace("Updating coordinates for node %d\n", gapp_->addr_);
    gapp_->notifyRecv(initialSender, remote_coord, current_time - time_sent, remote_error);

    GnutStats::instance().setPeerCoordError(
        gapp_->addr_, gapp_->mVivaldi->getLocalError());

//    //bootstrap server should only recv BOOTSTRAP msg
//    if (gapp_->isBootserver_ && msg!=MSG_BOOTSTRAP && msg!=MSG_BOOTCACHE_UPDATE) 
//    {
//        return pktdata->size();	
//    }


    GnutUtil::gnutTrace("%d is%sthe bootserver\n", 
                        gapp_->addr_, 
                        gapp_->isBootserver_ ? " " : " not "); 

    GNUT_ASSERT(!gapp_->isBootserver_ || 
                msg == GnutMsgParser::GNUT_BOOTSTRAP || 
                msg == GnutMsgParser::GNUT_BOOTCACHE_UPDATE);
	
    switch(msg) 
    {
    case GnutMsgParser::GNUT_BOOTSTRAP:
        gapp_->BootstrapRequest(src);
        break;

    case GnutMsgParser::GNUT_BOOTCACHE_UPDATE:
        gapp_->BootcacheUpdate(src);
        break;

    case GnutMsgParser::GNUT_BOOTSTRAP_RES:
        {
            const BootstrapRes& bsr = payload.getBootstrapRes();
            gapp_->BootstrapResult(bsr.cnt_, bsr.servents_); 
            payload.freeBootstrapRes();
        }
        break;

    case GnutMsgParser::GNUT_LEGACY_CONNECT:
        res = gapp_->ConnectionRequest(src, sock, remote_coord, remote_error, time_sent);

        // if connection request granted by the app, 
        // add the socket to socket list
        if(res) 
        {
            SockMap_t::iterator si = lsocks_.find(src);
            if(si == lsocks_.end()) 
            {
                SockEntry *newentry = new SockEntry(sock, src, SOCK_LEGACY);
                lsocks_.insert(SockMap_t::value_type(src, *newentry));
            } 
            else 
            {
                debug_info("requested conn to %d from %d already exists\n", 
                           app_->addr_, src);
            }
        }
        break;
    
    case GnutMsgParser::GNUT_ULTRA_CONNECT:
        res = gapp_->UltraConnRequest(src, sock, remote_coord, remote_error, time_sent);

        if(res) 
        {
            SockMap_t::iterator si = lsocks_.find(src);
            if(si==lsocks_.end()) 
            {
                SockEntry *newentry = new SockEntry(sock, src, SOCK_ULTRA);
                lsocks_.insert(SockMap_t::value_type(src, *newentry));
            } 
            else 
            {
                debug_info("requested conn to %d from %d already exists\n", 
                           app_->addr_, src);
            }
        }
        break;

    case GnutMsgParser::GNUT_LEAF_CONNECT:
        res = gapp_->LeafConnRequest(src, sock, remote_coord, remote_error, time_sent);

        if(res) 
        {
            SockMap_t::iterator si = lsocks_.find(src);
            if(si==lsocks_.end()) 
            {
                SockEntry *newentry = new SockEntry(sock, src, SOCK_LEAF);
                lsocks_.insert(SockMap_t::value_type(src, *newentry));
            } 
            else 
            {
                debug_info("requested conn to %d from %d already exists\n", 
                           app_->addr_, src);
            }
        }
        break;

    case GnutMsgParser::GNUT_OK:
    case GnutMsgParser::GNUT_ULTRA_OK:
    case GnutMsgParser::GNUT_LEAF_OK:
        {
            for(PendingConns_t::iterator i = conn_pending_.begin(); 
                i!=conn_pending_.end(); i++) 
            {
                if(i->second.peer_==src) 
                {
                    conn_pending_.erase(i);
                    break; // for
                }
            }

            SockMap_t::iterator si = lsocks_.find(src);
            if(si == lsocks_.end()) 
            {
                // TODO: AD uglyness!!!
                SockEntry *newentry = new SockEntry(sock, src, 
                                    msg == GnutMsgParser::GNUT_OK ?
                                        SOCK_LEGACY
                                    : msg == GnutMsgParser::GNUT_ULTRA_OK ?
                                        SOCK_ULTRA
                                    : SOCK_LEAF);
                lsocks_.insert(SockMap_t::value_type(src, *newentry));
            }

            // TODO: AD uglyness!!!
            gapp_->ConnectSucceeded(src,
                                    msg == GnutMsgParser::GNUT_OK ? 
                                        EGnutAppLegacy 
                                    : msg == GnutMsgParser::GNUT_ULTRA_OK ? 
                                        EGnutAppUltrapeer
                                    : EGnutAppLeaf, remote_coord, 
                                    remote_error); 
        }
        break;

    case GnutMsgParser::GNUT_REJ:
        {
            PendingConns_t::iterator pi;
            pi = conn_pending_.find(src);
            if(pi!=conn_pending_.end()) 
            {
                gapp_->ConnectionRejected(src);

                for(pi=conn_pending_.begin(); pi!=conn_pending_.end(); pi++) 
                {
                    if(pi->second.peer_==src) 
                    {
                        conn_pending_.erase(pi);

                        sock->close();
                        break;
                    }
                }
            }
        }
        break;

    case GnutMsgParser::GNUT_PING:
        if(find_desc(header.id_)!=NULL) 
            break;  //do not forward if DescID already seen

        if(find_ping(header.id_, 1)!=-1)
            break;  //do not forward if it originates from me

        if(header.ttl_ > header.hops_) 
        {
            block = forward(sock, pktdata, header);
            if(block) 
            {
                return 0;
            }
        }
        gapp_->PingRequest(src, header.ttl_, header.id_); 
        break;

    case GnutMsgParser::GNUT_PONG:
        // PONG has to be in descriptor cache or outstanding PING cache 
        if(find_desc(header.id_)==NULL && find_ping(header.id_, 0)==-1)
        {
            payload.freePongRes();
            break;
        }

        if(header.ttl_ > header.hops_) 
        {
            block = backroute(pktdata, header);
        }

        gapp_->PongReply(src, payload.getPongRes());
        if(find_desc(header.id_)==NULL) 
        {
            pongcnt++;
            if(pongcnt%100==0)
                debug_info("pong cnt %d\n",pongcnt);
        }

        payload.freePongRes();

        break;

    case GnutMsgParser::GNUT_QUERY:
        if(find_desc(header.id_)!=NULL)
            break;

        if(find_query(header.id_, 1)!=-1)
            break;

        if(header.ttl_ > header.hops_) 
        {
            block = forward(sock, pktdata, header);
            if(block) 
            {
                return 0;
            }
        }
        {
            GnutQueryMsg query(pktdata->data());
            gapp_->QueryRequest(src, 0, header.id_, query);
        }
        break;

    case GnutMsgParser::GNUT_QUERYHIT:
        if(find_desc(header.id_)==NULL && find_query(header.id_, 0)==-1)
            break;

        if(find_desc(header.id_)!=NULL && header.ttl_ > header.hops_) 
            backroute(pktdata, header);

        //we didn't forward the corresponding Query request
        //we must be receiving it because we initiated the Query message
        if(find_desc(header.id_)==NULL) 
        {
            GnutUtil::gnutError("Query hit %f arrived at destination %d\n",
                                header.id_, gapp_->addr_);
            GNUT_ASSERT(header.initialSender_ = gapp_->addr_);
            GnutQueryHitMsg hit(pktdata->data());
            gapp_->QueryHitReply(src, hit);
            nqueryhit++;
        }

      break;

    case GnutMsgParser::GNUT_ULTRA_LEAF_DIR:
        {
            // TODO: AD ugly!
            Ultrapeer* upeer = dynamic_cast<Ultrapeer*>(gapp_);
            if (upeer == 0)
            {
                break;
            }

            for(PendingConns_t::iterator i = conn_pending_.begin(); 
                i != conn_pending_.end(); i++) 
            {
                if(i->second.peer_==src) 
                {
                    conn_pending_.erase(i);
                    break; // for
                }
            }

            upeer->UltraLeafGuidance(src); 
        }
        break;

    default: 
        GNUT_ASSERT(0);
        debug_warning("WARNING: Unknown Message Type\n");
        break;
    }; // switch(msg)

    return pktdata->size();  
}

//============================================================================
int 
GnutellaAgent::
command(int argc, const char*const* argv) 
{
    Tcl &tcl = Tcl::instance();

    const char* const cmd = argv[1];

    if (argc == 3) 
    {
        if(strcmp(cmd, "set-app") == 0) 
        {
            GnutellaApp* app = reinterpret_cast<GnutellaApp*>
                                            (tcl.lookup(argv[2]));
            if(app != NULL) 
            {
                gapp_ = app;
                app_ = gapp_;
                return TCL_OK;
            }
            return TCL_ERROR;
        }
    }
    else if (argc == 2) 
    {
        if(strcmp(cmd, "statistics") == 0) 
        {
            statistics();
            return TCL_OK;
        }
        else if (strcmp(cmd, "cleanup") == 0)
        {
            return cleanup();
        }
    }

    return TCL_ERROR;
}

//============================================================================
void GnutellaAgent::statistics() {
     if (!gapp_->isBootserver_) {
       bkblock_.print(0);
       fwblock_.print(1);
       gapp_->degree_.print(2);
       rcvRate_.print(3);
     }
}

//============================================================================
// APIs to GnutellaApp
/* bootstrap request */
void GnutellaAgent::Bootstrap(NodeAddr_t peer) {
  Socket *cur=NULL;

  SockMap_t::iterator si = lsocks_.find(peer);
  if(si==lsocks_.end()) { 
  Tcl &tcl = Tcl::instance();

#ifdef PDNS
  tcl.evalf("[Simulator instance] create-sock %d %s -1 1 %d %d", app_->addr_, name(), use_prio_, rate_limit_); 
  cur = (Socket *)tcl.lookup(tcl.result());
#else
  tcl.evalf("[Simulator instance] create-sock %d %s -1 0 %d %d", app_->addr_, name(), use_prio_, rate_limit_); 
  cur = (Socket *)tcl.lookup(tcl.result());
#endif
  if(cur==NULL) {
    debug_warning("WARNING: unable to create-sock to %d\n", peer);
    return;
  }

  cur->connect(peer, LISTEN_PORT);
  SockEntry *newentry = new SockEntry(cur, peer, SOCK_BOOT_WAIT);
  lsocks_.insert(SockMap_t::value_type(peer, *newentry));
  }
}

//============================================================================
/* respond to a bootstrap request*/
void GnutellaAgent::Bootstrap_Reply(NodeAddr_t peer, int cnt, NodeAddr_t *iplist) {
  SockMap_t::iterator si = lsocks_.find(peer);
  if(si==lsocks_.end()) {
    debug_warning("WARNING: error sending bootstrap through unconnected peer\n");
    return;
  }

  GnutBootstrapResMsg bsr(cnt, iplist,
                          gapp_->mVivaldi->getCoordinates(),
                          gapp_->mVivaldi->getLocalError());
  PacketData* pkt = GnutMsgParser::instance().newpacket(bsr);

  si->second.sock_->send(bsr.getSize(), pkt);
}

//============================================================================
/* request to connect to a peer*/
int GnutellaAgent::Connect(NodeAddr_t peer, double timeout) {
  Socket *cur=NULL;
  PendingConnEntry *newconn=NULL;

  SockMap_t::iterator si = lsocks_.find(peer);
  PendingConns_t::iterator pi = conn_pending_.find(peer);

  if(si!=lsocks_.end()) {
    GnutUtil::gnutTrace("existing socket from %d to %d not removed\n", app_->addr_, peer);
    return FALSE;
  }

  if(pi==conn_pending_.end()) {
    GnutUtil::gnutTrace("About to create socket\n");
    Tcl &tcl = Tcl::instance();
#ifdef PDNS
    tcl.evalf("[Simulator instance] create-sock %d %s -1 1 %d %d", gapp_->addr_, name(), use_prio_, rate_limit_);
#else
    tcl.evalf("[Simulator instance] create-sock %d %s -1 0 %d %d", gapp_->addr_, name(), use_prio_, rate_limit_);
#endif

    cur = (Socket *)tcl.lookup(tcl.result());
    if(cur==NULL) {
      GnutUtil::gnutTrace("WARNING: unable to create-sock to\n", peer);
      debug_warning("WARNING: unable to create-sock to\n", peer);
      return FALSE;
    }

    GnutUtil::gnutTrace("About to connect socket with %d:%d\n", peer, LISTEN_PORT);
    cur->connect(peer, LISTEN_PORT);
    newconn = new PendingConnEntry(peer, timeout, NOW, NOT_CONNECTED);
  } else {
    return FALSE;
  }

  //add to pending connection list
  conn_pending_.insert(PendingConns_t::value_type(peer, *newconn));
  if(conn_pending_.size()==1)
    conn_timer_.resched(CONN_TIMEOUT);
  return TRUE;
}
//============================================================================
/* disconnect a peer (peers) */
void GnutellaAgent::Disconnect(NodeAddr_t peer) {
  SockMap_t::iterator i;

  if(peer == ALL_NEIGHBOURS) { //disconnect all
    GnutUtil::gnutTrace("GnutellaAgent disconnecting from all peers. Had %d"
                        " connections\n", lsocks_.size());

    for(i=lsocks_.begin(); i != lsocks_.end(); i++) {
      if(i->second.type_!=SOCK_BOOT_WAIT)
	      i->second.sock_->close();
    }

    // TODO: AD - fix this ugly code later...
    while(!lsocks_.empty()) {
      i = lsocks_.begin();
#ifdef LEAF_GUIDANCE
      delete i->second.sock_;
#endif /* LEAF_GUIDANCE */
      lsocks_.erase(i);
    }
  }
  else
  {
      //disconnect a specific peer
      i = lsocks_.find(peer);
      if(i!=lsocks_.end()) {
        i->second.sock_->close();
        lsocks_.erase(i);
      }
  }
}

//============================================================================
/* send PING to a peer */
void GnutellaAgent::Ping(NodeAddr_t peer, int ttl) {

  SockMap_t::iterator si = lsocks_.find(peer);
  if(si==lsocks_.end()) {
    debug_warning("WARNING: %d error pinging unconnected peer %d\n", app_->addr_, peer);
    return;
  } 

  GnutMsgParser& parser = GnutMsgParser::instance();
  
  DescriptorID_t guid = parser.getNewGuid();

  GnutPingMsg ping(gapp_->addr_, guid, ttl, 0,
                   gapp_->mVivaldi->getCoordinates(),
                   gapp_->mVivaldi->getLocalError());
  PacketData* pkt = parser.instance().newpacket(ping);

  GNUT_ASSERT(pkt);

  if(pkt) {
    si->second.sock_->send(ping.getSize(), pkt);
    PendingReqEntry *newquery = new PendingReqEntry(ping.getHeader().id_, NOW, REQ_PING);
    pending_req_.insert(pending_req_.end(), *newquery);
  }
}

//============================================================================
/* respond to a PING with a specific descriptor ID */
void 
GnutellaAgent::
Pong(NodeAddr_t peer, int cnt, const NodeAddr_t *iplist, DescriptorID_t id) {

  SockMap_t::iterator si = lsocks_.find(peer);
  if(si==lsocks_.end()) {
    //sending PONG through unconnected peer could happen
    return;
  }

  for(int i=0; i<cnt; i++) {
    GnutPongMsg pong(gapp_->addr_, id, INIT_TTL, 0, 0 /* port */, iplist[i],
                     gapp_->mVivaldi->getCoordinates(),
                     gapp_->mVivaldi->getLocalError());

    PacketData* pkt = GnutMsgParser::instance().newpacket(pong);
    if(pkt) 
      si->second.sock_->send(pong.getSize(), pkt);
  }
}

//============================================================================
/* send a Query */
void 
GnutellaAgent::
Query(NodeAddr_t peer) 
{
  SockMap_t::iterator si;
  PacketData* pkt = 0;

  GnutMsgParser& parser = GnutMsgParser::instance();
  DescriptorID_t guid = parser.getNewGuid();

  GnutStats::instance().newQuery(guid, GnutUtil::now());
  
  GnutQueryMsg query(gapp_->addr_, guid, INIT_TTL, 0,
                     gapp_->mVivaldi->getCoordinates(),
                     gapp_->mVivaldi->getLocalError());

  // global
  nquery++;

  if(peer == ALL_NEIGHBOURS) 
  { //send to all connected peers
    for(si = lsocks_.begin(); si != lsocks_.end(); si++) 
    {
      pkt = parser.newpacket(query);

      GNUT_ASSERT(pkt);

      if(pkt) 
      {
        si->second.sock_->send(query.getSize(), pkt);
      }
    }

    PendingReqEntry *newquery = new PendingReqEntry(query.getHeader().id_, NOW, REQ_QUERY);
    pending_req_.insert(pending_req_.end(), *newquery);
    return;
  }

  //or, send to a specific peer
  si = lsocks_.find(peer);
  if(si==lsocks_.end()) 
  {
    debug_warning("WARNING: error sending Query to unconnected peer %d\n", peer);
    return;
  }

  pkt = parser.newpacket(query);  
  if(pkt) 
  {
    si->second.sock_->send(query.getSize(), pkt);

    PendingReqEntry *newquery = new PendingReqEntry(query.getHeader().id_, NOW, REQ_QUERY);
    pending_req_.insert(pending_req_.end(), *newquery);
  }
}

//============================================================================
/* respond to a Query with QueryHit */
void GnutellaAgent::QueryHit(NodeAddr_t peer, DescriptorID_t id, 
                             const GnutQueryMsg& aQuery) 
{
  SockMap_t::iterator si = lsocks_.find(peer);
  if(si==lsocks_.end()) 
  {
    // sending QueryHit through unconnected peer could happen
    GnutUtil::gnutError("Agent %d dropping query hit for unconnected peer %d and query hit with id %f\n", gapp_->addr_, peer, id);
    return;
  }

  GnutQueryHitMsg hit(gapp_->addr_, id, INIT_TTL, 0, aQuery.getTimeSent(),
                      gapp_->mVivaldi->getCoordinates(),
                      gapp_->mVivaldi->getLocalError());
  PacketData* pkt = GnutMsgParser::instance().newpacket(hit);
  if(pkt)
  {
    int res = si->second.sock_->send(hit.getSize(), pkt);
    if(res != -1) 
    {
      GnutUtil::gnutError("Send failure sending query hit in response to query %f in peer %d. Trying to send to peer %d\n", id, gapp_->addr_, peer);
      squeryhit++;
    }
  }
}

//============================================================================
//called within GnutellaAgent
/* compose and send LimeWire-like bootstrap request */
void GnutellaAgent::lime_bootstrap(Socket *sock) 
{
  GnutBootstrapMsg bs(gapp_->mVivaldi->getCoordinates(),
                      gapp_->mVivaldi->getLocalError());
  PacketData* pkt = GnutMsgParser::instance().newpacket(bs);

  sock->send(bs.getSize(), pkt);
}

//============================================================================
/* agent send out cache update messages to connected bootstrap servers*/
void GnutellaAgent::UpdateBootcache() {
  for(SockMap_t::iterator si = lsocks_.begin(); si != lsocks_.end(); si++) {
	if(si->second.type_==SOCK_BOOT) {
      GnutBootcacheUpdateMsg bum(gapp_->mVivaldi->getCoordinates(),
                                 gapp_->mVivaldi->getLocalError());
  	  PacketData* pkt = GnutMsgParser::instance().newpacket(bum);
      
	  si->second.sock_->send(bum.getSize(), pkt);
	}
  }
}

//============================================================================
/* compose and send Gnutella Connection OK msg */
void GnutellaAgent::gnutella_ok(Socket *sock) 
{
    GnutOkMsg ok(gapp_->mVivaldi->getCoordinates(),
                 gapp_->mVivaldi->getLocalError());
    PacketData* pkt = GnutMsgParser::instance().newpacket(ok);
    sock->send(ok.getSize(), pkt);
}

//============================================================================
/* compose and send Gnutella connection rejection msg */
void GnutellaAgent::gnutella_reject(Socket *sock) 
{
    GnutRejMsg rej(gapp_->mVivaldi->getCoordinates(),
                   gapp_->mVivaldi->getLocalError());
    PacketData* pkt = GnutMsgParser::instance().newpacket(rej);
    sock->send(rej.getSize(), pkt);
}

//============================================================================
/* compose and send Gnutella Connection Request msg */
void GnutellaAgent::gnutella_req(Socket *sock) 
{
    GnutLegacyConnectMsg con(gapp_->mVivaldi->getCoordinates(),
                             gapp_->mVivaldi->getLocalError());
    PacketData* pkt = GnutMsgParser::instance().newpacket(con);
    sock->send(con.getSize(), pkt);
  
    if(conn_pending_.size() == 1)
    {
        conn_timer_.resched(CONN_TIMEOUT);
    }
}
//============================================================================
/* Conn timeout handler, notify GnutellaApp of requests that have timed out*/
void GnutellaAgent::conn_timeout() 
{
    double ctime = NOW;
    for(PendingConns_t::iterator pi = conn_pending_.begin(); 
        pi != conn_pending_.end(); pi++) 
    {
        if(ctime - pi->second.start_ > pi->second.timeout_ 
           && pi->second.state_==TRANSPORT_CONNECTED)
        {
            gapp_->ConnectionTimeout(pi->second.peer_);

            // TODO: AD - IMPORTANT: ensure that this is ok.
            conn_pending_.erase(pi);
        }
    }
}
//============================================================================
/* find the incoming socket matching a descriptor ID*/
inline Socket* GnutellaAgent::find_desc(DescriptorID_t aDesc) 
{
  DescMap_t::const_iterator di = desc_cache_.find(aDesc);
  if (di != desc_cache_.end())
  {
      return di->second.sock_;
  }
  else
  {
      return NULL;
  }
}

//============================================================================
/* find the outstanding Query matching a descriptor ID */
int GnutellaAgent::find_query(DescriptorID_t descid, int poll) {
  int cnt = -1;
  double ctime = NOW, tmp, delay;

  for(ReqList_t::iterator di=pending_req_.begin(); di !=pending_req_.end(); di++) {
    if(di->type_==REQ_QUERY && di->id_ == descid) 
    {
      cnt = di->replycnt_;

      if(poll) //poll only, do not update replycnt_
      {
        return cnt;
      }

      if(cnt==0) 
      {
        delay = ctime - di->tstamp_;
        tmp = (double)nthru * Avg_Rsp + delay;
        nthru++;
        Avg_Rsp = tmp/(double)nthru;
        GnutUtil::gnutTrace("QueryHit %d %d %d %d %f %f %d %d\n", 
                            nquery, nqueryhit, nthru, squeryhit, 
                            Avg_Rsp, delay, rquery, gapp_->lpeers_.size());
        fflush(NULL);
      }

      di->replycnt_++;
#ifdef WHAT_THE_HECK
      if(di->replycnt_ > 20) //remove Query if there are already many replies
        pending_req_.erase(di);
#endif
      return cnt;
    }
  }
  return -1;
}

//============================================================================
/* find the outstanding Ping matching a descriptor ID */
int GnutellaAgent::find_ping(DescriptorID_t aId, int poll) 
{
    int cnt = -1;
    for(ReqList_t::iterator di=pending_req_.begin(); di !=pending_req_.end(); di++) 
    {
        if(di->type_==REQ_PING && di->id_ == aId) 
        {
            cnt = di->replycnt_;
            if(!poll)
            {
                di->replycnt_++;
            }
            return cnt;
        }
    }
    return -1;
}
//============================================================================
/* forward a PING or Query message */
int 
GnutellaAgent::
forward(Socket* incoming, PacketData* data, const GnutDescHdr& hdr) 
{
  for(SockMap_t::iterator si = lsocks_.begin(); si!=lsocks_.end(); si++) {
    if(si->second.sock_!=incoming) {
      if(!si->second.sock_->blocked(SOCK_WRITE)) {
        si->second.sock_->send(data->size(), new PacketData(*data));
       }
       else {
        GnutUtil::gnutError("Socket blocked when forwarding message with id %f in peer %d\n", hdr.id_, gapp_->addr_);
        fwblock_.increment(); 
      } 
    }
  }

  DescEntry* newentry = new DescEntry(hdr.id_, incoming, NOW);
  desc_cache_[hdr.id_] = *newentry;
  return FALSE;
}

//============================================================================
/* route a PONG or QueryHit back to the initiator */
int 
GnutellaAgent::
backroute(PacketData *data, const GnutDescHdr& header) 
{
  Socket *sock = find_desc(header.id_);

  if(sock!=NULL) 
  {
    SockMap_t::iterator si = lsocks_.find(sock->peer_.addr_);
    if(si!=lsocks_.end()) 
    {
      if(sock->blocked(SOCK_WRITE)) 
      {
        GnutUtil::gnutError("Socket blocked when backrouting message with id %f in peer %d\n", header.id_, gapp_->addr_);
        bkblock_.increment();
      }
      else
      {
//        GnutUtil::gnutError("Socket NOT blocked when backrouting message with id %f in peer %d\n", header.id_, gapp_->addr_);
      }
          

      PacketData* newdata = new PacketData(*data);
      sock->send(data->size(), newdata);
      GnutUtil::gnutError("Found socket to backroute message %f in peer %d\n", header.id_, gapp_->addr_);
    }
    else
    {
        GnutUtil::gnutError("Failed to find socket to backroute message %f in peer %d\n", header.id_, gapp_->addr_);
    }
  } 
  return FALSE;
}

//============================================================================
/* Garbage collection of descriptor ID and pending connection */
void 
GnutellaAgent::
gc() 
{
    double curtime = NOW;
  
    DescMap_t::iterator di;
    while(!desc_cache_.empty()) 
    {
        di = desc_cache_.begin();
        if(curtime - di->second.tstamp_ > DESC_TIMEOUT) 
        {
            desc_cache_.erase(di);
        } 
        else
            break;
    }

    ReqList_t::iterator qi;
    while(!pending_req_.empty()) 
    {
        qi = pending_req_.begin();
        if(curtime - qi->tstamp_ > DESC_TIMEOUT) 
        {
            if(qi->replycnt_==0) 
            {
                nforceremove++;	
            }
            pending_req_.erase(qi);

        } 
        else
            break;
    }
}
//============================================================================
void 
GnutellaAgent::gnutella_leaf_guidance(Socket*)
{
    GNUT_ASSERT(0);
}
//============================================================================
int
GnutellaAgent::cleanup(void)
{
    Disconnect(ALL_NEIGHBOURS);
    gapp_ = 0;
    app_  = 0;
    return TCL_OK;
}
//============================================================================
static class GnutellaAgentClass: public TclClass {
public:
  GnutellaAgentClass(): TclClass("SocketApp/PeerAgent/GnutellaAgent") {}
  TclObject* create(int argc, const char*const* argv) {
    return (new GnutellaAgent(atoi(argv[4])));
  }
}class_gnutellaagent;
//============================================================================

