//=============================================================================
// File: gnutstats.cc
// Author: Andre Dufour
//=============================================================================
#include "gnutstats.h"
#include <cstring>
#include <cstdlib>
#include <errno.h>
#include <limits>
#include <sys/stat.h>
#include <sys/types.h>
#include "gnut_util.h"
#include "gnutqueryrecord.h"

// Kinda ugly, but I'm ok with it for now.
// Eventually, I should write a runtime expression to figure out the right
// length. Anyway, the strings that use this don't persist very long and 
// are normally only created once, at the end of the simulation.
#define GNUT_STAT_FILENAME_LEN 500

//=============================================================================
GnutStats* GnutStats::mInstance = 0;
//=============================================================================
// Static
GnutStats*
GnutStats::
initialize(unsigned int         aNumPeers, 
           const char* const    aTopologyId, 
           const char* const    aRootDir,
           long                 aRandomSeed)
{
    // TODO: AD
    GnutUtil::gnutTrace("Initializing GnutStats: %s %s %ld\n",
                        aTopologyId, aRootDir, aRandomSeed);

    GNUT_ASSERT(mInstance == 0);
    mInstance = new GnutStats(aNumPeers, aTopologyId, aRootDir, aRandomSeed);
    return mInstance;
}
//=============================================================================
// Static
GnutStats& 
GnutStats::
instance(void)
{
    GNUT_ASSERT(mInstance != 0);
    return *mInstance;
}
//=============================================================================
int
GnutStats::
dumpStats(void) const
{
    int rc = TCL_ERROR;
    if ((rc = dumpPeerFile()) != TCL_OK)
    {
        return rc;
    }

    return dumpTimeFiles();
}
//=============================================================================
int
GnutStats::
command(int argc, const char* const* argv)
{
    if (argc == 2)
    {
        if (strcmp(argv[1], "dump-stats") == 0)
        {
            return dumpStats();
        }
        else if (strcmp(argv[1], "snapshot") == 0)
        {
            takeSnapshot(GnutUtil::now());
            return TCL_OK;
        }
    }

    return TCL_ERROR;
}
//=============================================================================
GnutStats::
GnutStats(unsigned int      aNumPeers,
          const char* const aTopologyId,
          const char* const aRootDir,
          long              aRandomSeed)
:  mTopologyId(0),
   mRootDir(0),
   mRandomSeed(aRandomSeed)
{ 
    // TODO: assume memory is always available. Bad.

    size_t len = strlen(aTopologyId) + 1; // don't forget the null!
    mTopologyId = new char[len];
    strncpy(mTopologyId, aTopologyId, len);

    len = strlen(aRootDir) + 1;
    mRootDir = new char[len];
    strncpy(mRootDir, aRootDir, len);
}
//=============================================================================
GnutStats::
~GnutStats(void)
{ 
    delete[] mTopologyId;
    delete[] mRootDir;
}
//=============================================================================
void
GnutStats::newQuery(DescriptorID_t aId, double aTimeSent)
{
    // TODO: AD sooooo inefficient!
    GnutQueryRecord rec(aTimeSent);
    mQueries[aId] = rec;
    GnutUtil::gnutTrace("New query %f being sent at time %f\n", aId, aTimeSent);
}
//=============================================================================
void 
GnutStats::
newNodeTime(NodeAddr_t aNode, DescriptorID_t aId, double aTime)
{
    mQueries[aId].newNodeTime(aNode, aTime);
    GnutUtil::gnutTrace("New node time %f being recorded for query %f\n", 
                        aTime, aId);
}
//=============================================================================
void 
GnutStats::newHitTime(DescriptorID_t aId, double aTime)
{
    GnutUtil::gnutTrace("New query hit recorded for query %f at time %f\n",
                        aId, aTime);
    mQueries[aId].newHitTime(aTime);
}

//=============================================================================
void 
GnutStats::newConnDumpTime(double aTime)
{
    GnutUtil::gnutTrace("New conn dump time recorded at time %f\n", aTime);
    mDumpTimeList.push_back(aTime);
}
//=============================================================================
void 
GnutStats::setPeerConns(NodeAddr_t aNode, unsigned int aNumConns)
{
    mPeerRecs[aNode].mNumConns = aNumConns;
}
//=============================================================================
void
GnutStats::setPeerCoordError(NodeAddr_t aNode, double aError)
{
    mPeerRecs[aNode].mError = aError;
}
//=============================================================================
void 
GnutStats::setPeerState(NodeAddr_t aNode, EGnutApp aRole)
{
    mPeerRecs[aNode].mRole = aRole;
}
//=============================================================================
void
GnutStats::setPeerOnline(NodeAddr_t aNode, bool aOnline)
{
    mPeerRecs[aNode].mOnline = aOnline;
}
//=============================================================================
void
GnutStats::setPeerHasContent(NodeAddr_t aNode, bool aHasContent)
{
    mPeerRecs[aNode].mHasContent = aHasContent;
}
//=============================================================================
void
GnutStats::takeSnapshot(double aTime)
{
    PeerSummary sum(aTime);

    GnutUtil::gnutTrace("Processing snapshot command\n");

    // TODO: AD - remove!
    GnutUtil::gnutTrace("peer_dirname = %s/%s\n", mRootDir, mTopologyId);

    std::list<double> errorList;

    for(PeerRecMap::const_iterator iter = mPeerRecs.begin();
        iter != mPeerRecs.end();
        iter++)
    {
        if (iter->second.mOnline)
        {
            switch (iter->second.mRole)
            {
            case EGnutAppLegacy:
                sum.mNumLegacy++;
                break;

            case EGnutAppLeaf:
                sum.mNumLeaf++;
                break;

            case EGnutAppUltrapeer:
                sum.mNumUltra++;
                break;

            default:
                GNUT_ASSERT(0);
                break;
            } // switch (iter->mRole)

            sum.mConnections.push_back(iter->second.mNumConns);

            errorList.push_back(iter->second.mError);

            if (iter->second.mHasContent)
            {
                sum.mNumHasContent++;
            }
        }
    }

    errorList.sort();

    unsigned long middle_of_list = errorList.size() / 2; 
    unsigned long item_count = 0;

    for (std::list<double>::const_iterator iter = errorList.begin(); ;
         iter++, item_count++)
    {
        if (item_count == middle_of_list)
        {
            sum.mMedianError = *iter;
            break;
        }
    }

    mPeerSummaries.push_back(sum);
}


//=============================================================================
int
GnutStats::
dumpPeerFile(void) const
{
    // TODO: AD - would be much nicer with C++ style strings. Oh well :(

    char peer_dirname[GNUT_STAT_FILENAME_LEN];
    char peer_filename[GNUT_STAT_FILENAME_LEN];
    char dump_conn_filename[GNUT_STAT_FILENAME_LEN];

    snprintf(peer_dirname, GNUT_STAT_FILENAME_LEN, "%s/%s",
             mRootDir, mTopologyId);

    GnutUtil::gnutTrace("peer_dirname = %s\n", peer_dirname);


    snprintf(peer_filename, GNUT_STAT_FILENAME_LEN, "%s/%s/peers/%ld.txt",
             mRootDir, mTopologyId, mRandomSeed);
    snprintf(dump_conn_filename, GNUT_STAT_FILENAME_LEN, 
             "%s/%s/peers/%lddump.txt",
             mRootDir, mTopologyId, mRandomSeed);

    if (mkdir(peer_dirname, S_IRUSR | S_IWUSR | S_IXUSR) != 0 &&
        errno != EEXIST)
    {
        GnutUtil::gnutTrace("Unable to create directory %s\n", peer_dirname);
        return TCL_ERROR;
    }

    snprintf(peer_dirname, GNUT_STAT_FILENAME_LEN, "%s/%s/peers",
             mRootDir, mTopologyId);

    if (mkdir(peer_dirname, S_IRUSR | S_IWUSR | S_IXUSR) != 0 &&
        errno != EEXIST)
    {
        GnutUtil::gnutTrace("Unable to create directory %s\n", peer_dirname);
        return TCL_ERROR;
    }

    FILE* peerfile = fopen(peer_filename, "w");
    FILE* dumpfile = fopen(dump_conn_filename, "w");

    if (peerfile != NULL && dumpfile != NULL)
    {
        for (PeerSummaryVector::const_iterator iter = 
                mPeerSummaries.begin();
             iter != mPeerSummaries.end();
             iter++)
        {
            fprintf(peerfile, "%f %f %ld %ld",
                    iter->mTime,
                    iter->mMedianError,
                    iter->mNumLegacy +
                     iter->mNumLeaf +
                     iter->mNumUltra,
                    iter->mNumHasContent);

//            for (std::vector<unsigned long>::const_iterator conn_iter =
//                    iter->mConnections.begin();
//                 conn_iter != iter->mConnections.end();
//                 conn_iter++)
//            {
//                fprintf(peerfile, "%lu ", *conn_iter);
//            }
            
            fprintf(peerfile, "\n");
        }

        for (ConnDumpTimeList::const_iterator dumpIter = mDumpTimeList.begin();
             dumpIter != mDumpTimeList.end(); dumpIter++)
        {
            fprintf(dumpfile, "%f\n", *dumpIter);
        }

        fclose(peerfile);
        fclose(dumpfile);
        GnutUtil::gnutTrace("Dump of peers file succeeded\n");
    }
    else
    {

        GnutUtil::gnutTrace("Unable to open file %s for writing\n", 
                            peer_filename);
        return TCL_ERROR;
    }

    return TCL_OK;
}
//=============================================================================
int
GnutStats::
dumpTimeFiles(void) const 
{
    char dirname[GNUT_STAT_FILENAME_LEN];
    char filename[GNUT_STAT_FILENAME_LEN];

    // Create root directory
    snprintf(dirname, GNUT_STAT_FILENAME_LEN, "%s/%s/times",
             mRootDir, mTopologyId);

    if (mkdir(dirname, S_IRUSR | S_IWUSR | S_IXUSR) != 0 &&
        errno != EEXIST)
    {
        GnutUtil::gnutTrace("Unable to make dir %s\n", dirname);
        return TCL_ERROR;
    }

    for (QueryMap::const_iterator qiter = mQueries.begin();
         qiter != mQueries.end();
         qiter++)
    {
        // ---- Hit times

        snprintf(dirname, GNUT_STAT_FILENAME_LEN, 
                 "%s/%s/times/%f",
                 mRootDir, mTopologyId, qiter->second.getTimeSent());

        if (mkdir(dirname, S_IRUSR | S_IWUSR | S_IXUSR) != 0 &&

            errno != EEXIST)
        {
            GnutUtil::gnutTrace("Unable to make dir %s\n", dirname);
            return TCL_ERROR;
        }

        snprintf(dirname, GNUT_STAT_FILENAME_LEN, 
                 "%s/%s/times/%f/hit_times",
                 mRootDir, mTopologyId, qiter->second.getTimeSent());

        if (mkdir(dirname, S_IRUSR | S_IWUSR | S_IXUSR) != 0 &&

            errno != EEXIST)
        {
            GnutUtil::gnutTrace("Unable to make dir %s\n", dirname);
            return TCL_ERROR;
        }

        char filename[GNUT_STAT_FILENAME_LEN];
        // query_number.txt is the filename
        snprintf(filename, GNUT_STAT_FILENAME_LEN,
                 "%s/%f.txt",
                 dirname, qiter->first);

        FILE* hitfile = NULL;
        if ((hitfile = fopen(filename, "w")) == NULL)
        {
            GnutUtil::gnutTrace("Unable to open hitfile %s\n", filename);
            return TCL_ERROR;
        }

        const GnutQueryRecord::TimeVector& hit_times = 
                                    qiter->second.getHitTimes();

        for (GnutQueryRecord::TimeVector::const_iterator hiter = 
                                                    hit_times.begin();
             hiter != hit_times.end();
             hiter++)
        {
            fprintf(hitfile, "%f ", *hiter);
        }

        fclose(hitfile);
    
        // ---- Node times

        snprintf(dirname, GNUT_STAT_FILENAME_LEN, 
                 "%s/%s/times/%f/node_times",
                 mRootDir, mTopologyId, qiter->second.getTimeSent());

        if (mkdir(dirname, S_IRUSR | S_IWUSR | S_IXUSR) != 0 &&
            errno != EEXIST)
        {
            GnutUtil::gnutTrace("Unable to make directory %s\n", dirname);
            return TCL_ERROR;
        }

        // query_number.txt is the filename
        snprintf(filename, GNUT_STAT_FILENAME_LEN,
                 "%s/%f.txt",
                 dirname, qiter->first);

        FILE* nodefile = NULL;
        if ((nodefile = fopen(filename, "w")) == NULL)
        {
            GnutUtil::gnutTrace("Unable to open nodefile %s for writing\n",
                                filename);
            return TCL_ERROR;
        }

        const GnutQueryRecord::TimeVector& node_times = qiter->
                                    second.getNodeTimes();

        for (GnutQueryRecord::TimeVector::const_iterator niter = 
                                                node_times.begin();
             niter != node_times.end();
             niter++)
        {
            fprintf(nodefile, "%f ", *niter);
        }

        fclose(nodefile);
    }
                    
    GnutUtil::gnutTrace("Dump of time files succeeded\n");
    return TCL_OK;
}   
//=============================================================================
static class GnutStatsClass: public TclClass {
public:
    GnutStatsClass(): TclClass("GnutStats") {}
    TclObject* create(int argc, const char*const* argv) {
        GNUT_ASSERT(argc == 8);
        return (GnutStats::initialize(atoi(argv[4]), argv[5], argv[6], 
                                      atol(argv[7])));
    }
} class_gnutstats;
//=============================================================================

#undef GNUT_STAT_FILENAME_LEN
