// File: ultraagent.cc
// Original by Qi He
// Modified by Andre Dufour

#include "ultraagent.h"
#include <cstring>
#include "gnut_types.h"
#include "gnutellaagent.h"
#include "gnutellaapp.h"
#include "pendingreqentry.h"
#include "gnut_util.h"
#include "conntimer.h"
#include "gnutultraconnmsg.h"
#include "gnutultraokmsg.h"
#include "gnutpingmsg.h"
#include "gnutquerymsg.h"
#include "gnutmsgparser.h"
#include "gnutbinarymsgbc.h"
#include "gnutultraleafdirmsg.h"
#include "gnutstats.h"
#include "ivivaldimanager.h"

//=============================================================================
/* PeerAgent for Ultrapper Application*/
UltraAgent::UltraAgent(NodeAddr_t addr): GnutellaAgent(addr) 
{ }
//=============================================================================
UltraAgent::UltraAgent(GnutellaApp *app): GnutellaAgent(app) 
{ }
//=============================================================================
UltraAgent::~UltraAgent(void)
{ 
    GnutUtil::gnutTrace("UltraAgent::~UltraAgent()\n");
}
//=============================================================================
/* send PING to a peer */
void UltraAgent::Ping(NodeAddr_t peer, int ttl) 
{
    GnutMsgParser& parser = GnutMsgParser::instance();

    DescriptorID_t guid = parser.getNewGuid();
    
    GnutPingMsg ping(gapp_->addr_, guid, ttl, 0, 
                     gapp_->mVivaldi->getCoordinates(),
                     gapp_->mVivaldi->getLocalError());
    PacketData* pkt = parser.newpacket(ping);

    SockMap_t::iterator si = lsocks_.find(peer);
    if(si==lsocks_.end()) 
    {
        debug_warning("WARNING: %d error pinging unconnected peer %d\n", 
                      app_->addr_, peer);
        delete pkt;
        return;
    } 

    GNUT_ASSERT(pkt);
    
    if(pkt) 
    {
        si->second.sock_->send(ping.getSize(), pkt);
        PendingReqEntry *newquery = new PendingReqEntry(ping.getHeader().id_, NOW, REQ_PING);
        pending_req_.insert(pending_req_.end(), *newquery);
    }
}

//=============================================================================
/* send a Query */
void UltraAgent::Query(NodeAddr_t peer) 
{
    SockMap_t::iterator si;
    PacketData* pkt = 0;

    GnutMsgParser& parser = GnutMsgParser::instance();

    DescriptorID_t guid = parser.getNewGuid();
    GnutQueryMsg query(gapp_->addr_, guid, INIT_TTL, 0,
                       gapp_->mVivaldi->getCoordinates(),
                       gapp_->mVivaldi->getLocalError());
 
    GnutUtil::gnutTrace("%d sending query\n", gapp_->addr_);
    GnutStats::instance().newQuery(guid, GnutUtil::now());

    // TODO: AD - global var: fix this!!!
    nquery++;

    if(peer == GnutellaAgent::ALL_NEIGHBOURS) 
    { //send to all connected peers
        for(si = lsocks_.begin(); si != lsocks_.end(); si++) 
        {
            if (si->second.type_==SOCK_LEAF) 
            {

                //send to a leaf peer only with certain probability
                //QRP is not implemented yet
                float hprob = (float)rand()/(float)RAND_MAX;

                if(hprob < QRP_HITPROB) 
                {
                    continue;
                }
            }

            pkt = parser.newpacket(query);
            GNUT_ASSERT(pkt);

            if(pkt) 
            {
                si->second.sock_->send(query.getSize(), pkt);
            }
        }

        PendingReqEntry *newquery = new PendingReqEntry(query.getHeader().id_, NOW, REQ_QUERY);
        pending_req_.insert(pending_req_.end(), *newquery);
        return;
    }

    //or, send to a specific peer
    si = lsocks_.find(peer);
    if(si==lsocks_.end()) 
    {
        debug_warning("WARNING: error sending Query to unconnected peer %d\n", peer);
        return;
    }

    if (si->second.type_==SOCK_LEAF)
        return;

    pkt = parser.newpacket(query);  

    GNUT_ASSERT(pkt);

    if(pkt) 
    {
        si->second.sock_->send(query.getSize(), pkt);

        PendingReqEntry *newquery = new PendingReqEntry(query.getHeader().id_, NOW, REQ_QUERY);
        pending_req_.insert(pending_req_.end(), *newquery);
    }
}

//=============================================================================

/* forward a PING or Query message */
int 
UltraAgent::forward(Socket *incoming, PacketData *data, const GnutDescHdr& hdr) 
{

    for(SockMap_t::iterator si = lsocks_.begin(); si!=lsocks_.end(); si++) 
    {
        if(si->second.sock_!=incoming && si->second.type_!=SOCK_LEAF) 
        {
            if(si->second.sock_->blocked(SOCK_WRITE)) 
            {
                fwblock_.increment();
            }
        }
    }

    for(SockMap_t::iterator si = lsocks_.begin(); si!=lsocks_.end(); si++) 
    {
        if(si->second.sock_!=incoming && !si->second.type_!=SOCK_LEAF) 
        {
            if(!si->second.sock_->blocked(SOCK_WRITE)) 
            {
                PacketData* newdata = new PacketData(*data);
                si->second.sock_->send(data->size(), newdata);
            } 
        }
    }

    DescEntry* newentry = new DescEntry(hdr.id_, incoming, NOW);
    desc_cache_[hdr.id_] = *newentry;
    return FALSE;
}
//=============================================================================
/* compose and send Ultrapeer Connection Request msg */
void UltraAgent::gnutella_req(Socket *sock) 
{
    GnutUltraConnMsg con(gapp_->mVivaldi->getCoordinates(),
                         gapp_->mVivaldi->getLocalError());
    PacketData* pkt = GnutMsgParser::instance().newpacket(con);
    sock->send(con.getSize(), pkt);
  
    if(conn_pending_.size()==1)
        conn_timer_.resched(CONN_TIMEOUT);
}

//=============================================================================
/* compose and send Gnutella Connection OK msg */
void UltraAgent::gnutella_ok(Socket *sock) 
{
    GnutUltraOkMsg ok(gapp_->mVivaldi->getCoordinates(),
                      gapp_->mVivaldi->getLocalError());
    PacketData* pkt = GnutMsgParser::instance().newpacket(ok);
    sock->send(ok.getSize(), pkt);
}
//=============================================================================
// Compose and send ultrapeer leaf guidance message
void UltraAgent::gnutella_leaf_guidance(Socket *sock) 
{
    GnutUltraLeafDirMsg dir(gapp_->mVivaldi->getCoordinates(),
                            gapp_->mVivaldi->getLocalError());
    PacketData* pkt = GnutMsgParser::instance().newpacket(dir);
    sock->send(dir.getSize(), pkt);
}
//=============================================================================
static class UltraAgentClass: public TclClass {
public:
  UltraAgentClass(): TclClass("SocketApp/PeerAgent/GnutellaAgent/UltraAgent") {}
  TclObject* create(int argc, const char*const* argv) {
    return (new UltraAgent(atoi(argv[4])));
  }
}class_ultraagent;
//=============================================================================


