//=============================================================================
// File: vivaldimanager.cc
// Author: Andre Dufour
//=============================================================================

#include "vivaldimanager.h"
#include "gnutcoord.h"
#include "gnut_util.h"
#include <cmath>
#include <iostream>

//=============================================================================
const double VivaldiManager::DISPLACEMENT_FACTOR = 0.25;
// Recommended value in Vivaldi paper

const double VivaldiManager::MOVING_AVERAGE_SAMPLE_WEIGHT = 1.0;
// No value specified in paper!!!

//=============================================================================
VivaldiManager::VivaldiManager(void)
: mError(IVivaldiManager::NEW_NODE_ERROR)
{ }
//=============================================================================
VivaldiManager::~VivaldiManager(void)
{ }
//=============================================================================
GnutCoord 
VivaldiManager::getCoordinates(void) const
{
    return mCoordinates;
}
//=============================================================================
void 
VivaldiManager::updateCoordinates(
    const GnutCoord& aRemoteCoord,
    double aRtt,
    double aRemoteError)
{
    GnutUtil::gnutTrace("Updating coordinates. Remote error = %f\n", 
                        aRemoteError);

#if 0
std::cout << "My coordinates: " << mCoordinates << std::endl 
     << "Remote coord:   " << aRemoteCoord << std::endl
     << "Rtt:            " << aRtt << std::endl;
     fflush(NULL);
#endif 

    GNUT_ASSERT(aRtt > 0.0);
    GNUT_ASSERT(aRemoteError > 0.0);

    const double step = mError / (mError + aRemoteError);

    const GnutCoord coord_diff = mCoordinates - aRemoteCoord;

    const double rel_error = fabs(coord_diff.abs() - aRtt) / aRtt;

    GnutUtil::gnutTrace("Relative error = %f\n", rel_error);
    GnutUtil::gnutTrace("abs coord diff = %f\n", coord_diff.abs());
    GnutUtil::gnutTrace("Local error before modifcation = %f\n", mError);

    mError = rel_error * MOVING_AVERAGE_SAMPLE_WEIGHT * step
             + mError  * (1.0 - MOVING_AVERAGE_SAMPLE_WEIGHT * step);

    GnutUtil::gnutTrace("Local error after modifcation = %f\n", mError);
    GNUT_ASSERT(mError > 0.0);

    const double delta = DISPLACEMENT_FACTOR * step;

    mCoordinates = mCoordinates + delta * (aRtt - coord_diff.abs()) * 
                   GnutCoord(1, coord_diff); 
}
//=============================================================================
void 
VivaldiManager::resetCoordinates(void)
{ 
    mCoordinates = GnutCoord();
    mError = IVivaldiManager::NEW_NODE_ERROR;
}
//=============================================================================
double 
VivaldiManager::getLocalError(void) const
{
    return mError;
}
//=============================================================================
double 
VivaldiManager::estimateRtt(const GnutCoord& aRemoteCoord) const
{
    return (mCoordinates - aRemoteCoord).abs();
}
//=============================================================================

