// File: gnutellaapp.cc
// Original by Qi He
// Modified by Andre Dufour

#include "gnutellaapp.h"
#include "gnut_types.h"
#include "pingtimer.h"
#include "watchdog.h"
#include "activitycontroller.h"
#include "gnutellaagent.h"
#include "smpbootserver.h"
#include "gnut_util.h"
#include "peersys.h"
#include "gc.h"

//=============================================================================
/**** GnutellaApp ****/
GnutellaApp::
GnutellaApp(NodeAddr_t aAddr)
: PeerApp(aAddr),
  //ping_interval_(PING_INTERVAL),
  //watch_interval_(CONN_INTERVAL),
  isBootserver_(FALSE),
  max_deg_(MAX_DEGREE),
  ping_timer_(pingListener_),
  watchDog_(watchdogListener_),
  gagent_(0),
  degree_(aAddr),
  smpBoot_(FALSE),
  bootSrv_(0),
  watchdogListener_(*this, &GnutellaApp::watchdogExpired),
  pingListener_(*this, &GnutellaApp::pingExpired)
{
  bind("isBootserver_", &isBootserver_);
  bind("max_deg_", &max_deg_);
  
  // TODO: AD
  printf("Creating gnutellaapp %p for node %d\n", this, aAddr);
}

//=============================================================================
GnutellaApp::
~GnutellaApp(void)
{
    // TODO: AD
    printf("Destroying gnutellaapp %d (%p) (in C++)\n", addr_, this);
}
//=============================================================================
/* user command interface */
/* join: outsys->insys */
void GnutellaApp::join() {
  // TODO: AD
  printf("%d told to join\n", addr_);
  
  //ping_interval_ = PING_INTERVAL;

// TODO: AD test
// TODO: AD this causes the hated back in time error. WTF?
//if (state_ == PS_OUTSYS)
//{
//    state_ = PS_OFFLINE;
//}

  // TODO: AD

  watchDog_.resched();
  ping_timer_.resched();

  // TODO: AD
  bootstrap();
}
//=============================================================================
int
GnutellaApp::
cleanup()
{
    // TODO: AD
    printf("Cleaning up gnutellaapp %d (in C++)\n", addr_);
    setState(PS_OFFLINE); // Required for integrity of system.

    printf("About to destroy in TCL\n", addr_);
    fflush(NULL);
    Tcl::instance().evalf("%s cleanup_app", name());

    return TCL_OK;
}
//=============================================================================
/* leave: insys->outsys */
void GnutellaApp::leave() {
  if (state_!= PS_OUTSYS)
	setState(PS_OFFLINE);
  if (ping_timer_.status()==TIMER_PENDING)
    ping_timer_.cancel();

  if (watchDog_.status()==TIMER_PENDING)
    watchDog_.cancel();

  if (ac_ && ac_->status_==TIMER_PENDING)
    ac_->cancel();
}

//=============================================================================
void GnutellaApp::stop() {
  if (state_ != PS_OUTSYS)
	setState(PS_OFFLINE); 

  if (ping_timer_.status()==TIMER_PENDING)
    ping_timer_.cancel();

  if (watchDog_.status()==TIMER_PENDING)
    watchDog_.cancel();

  if (ac_ && ac_->status_==TIMER_PENDING)
    ac_->cancel();
}

//=============================================================================
void GnutellaApp::disconnect(NodeAddr_t node) 
{
    gagent_->Disconnect(node);
    if (node==-1) 
    {
        lpeers_.clear();
        if (smpBoot_) 
        {
            if (bootSrv_)
            {
                bootSrv_->RemovePeer(addr_, this);
            }
        }
    }
    else 
    {
        PeerMap_t::iterator pi = lpeers_.find(node);
        if (pi!=lpeers_.end()) 
        {
            lpeers_.erase(pi);
        }
        if (watchDog_.status() == TIMER_IDLE)
        {
            // If the watchdog had been turned off because we had all the 
            // connections we needed, turn it back on.
            // TODO: AD
            printf("%d can handle more connections. Re-enabling watchdog\n", addr_);
            watchDog_.resched();
        }
    }
    degree_.increment(lpeers_.size());
}

//=============================================================================
/* share files */
void GnutellaApp::share() {
}

//=============================================================================
/* maintenance of peer relationship */
void GnutellaApp::maintenance() {
}

//=============================================================================
/* initiate a query */
void GnutellaApp::search(char *criteria) {
  gagent_->Query(-1, 100, criteria); 
}

//=============================================================================
/* GnutellaApp internal operations */
/* proactively change its own status */
void GnutellaApp::setState(PeerState_t state) 
{
    PeerState_t old_state = state_;
    double tmp, ctime = NOW;

    state_ = state;

    // TODO: AD
    char* debug_state = 0;
    switch (state)
    {
    case PS_OFFLINE:
      debug_state = "offline";
      break;
    case PS_ACTIVE:
      debug_state = "active";
      break;
    case PS_IDLE:
      debug_state = "idle";
      break;
    case PS_OUTSYS:
      debug_state = "out_sys";
      break;
    default:
      debug_state = "???";
      break;
    }

    debug_stat("%d going %s\n", addr_, debug_state);
      

    //the following code is for membership accounting purpose
    if (state_ == PS_OFFLINE && old_state != PS_OFFLINE) 
    {
        if (!isFreeloader_) 
        {
            if (Last_Chg1 !=-1) 
            {
                tmp = (ctime - Last_Chg1) * (double)Na1 + Avg_Na1 * TSysTime1;
                TSysTime1 += (ctime - Last_Chg1);
                Avg_Na1 = tmp/TSysTime1;

                /* periodic averaging */
                tmp = (ctime - Last_Chg1) * (double)Na1 + pAvg_Na1 * pTime1;
                pTime1 += (ctime - Last_Chg1);
                pAvg_Na1 = tmp/pTime1;

                Na1--;
            }
            else 
            {
                debug_warning("WARNING: shouldn't get here (setState:1)\n");
            }

            Last_Chg1 = ctime;

        } 
        else 
        {
            if (Last_Chg2 !=-1) 
            {
                tmp = (ctime - Last_Chg2) * (double)Na2 + Avg_Na2 * TSysTime2;
                TSysTime2 += (ctime - Last_Chg2);
                Avg_Na2 = tmp/TSysTime2;

                /* periodic averaging */
                tmp = (ctime - Last_Chg2) * (double)Na2 + pAvg_Na1 * pTime2;
                pTime2 += (ctime - Last_Chg2);
                pAvg_Na2 = tmp/pTime2;

                Na2 --;
            }
            else 
            {
                debug_warning("WARNING: shouldn't get here (setState:2)\n");
            }  

            Last_Chg2 = ctime;
        }

        if (ping_timer_.status()==TIMER_PENDING)
            ping_timer_.cancel();

        if (watchDog_.status()==TIMER_PENDING)
            watchDog_.cancel();

        //disconnect() is required for correct functioning though
        disconnect(-1);

        debug_stat("Membership %d\t%d\n",  Na1, Na2);
        gagent_->bkblock_.offline();
        gagent_->fwblock_.offline();
        gagent_->rcvRate_.offline();
        degree_.offline();

        fflush(NULL);
    }
    else if (state_ != PS_OFFLINE && old_state == PS_OFFLINE) 
    {
        if (!isFreeloader_) 
        {
            if (Last_Chg1 !=-1) 
            {
                tmp = (ctime - Last_Chg1 ) * (double)Na1 + Avg_Na1 * TSysTime1;
                TSysTime1 += (ctime - Last_Chg1);
                Avg_Na1 = tmp/TSysTime1;

                tmp = (ctime - Last_Chg1) * (double)Na1 + pAvg_Na1 * pTime1;
                pTime1 += (ctime - Last_Chg1);
                pAvg_Na1 = tmp/pTime1;

            }
            Last_Chg1 = ctime; 
        }
        else 
        {
            if (Last_Chg2 !=-1) 
            {
                tmp = (ctime - Last_Chg2 ) * (double)Na2 + Avg_Na2 * TSysTime2;
                TSysTime2 += (ctime - Last_Chg2);
                Avg_Na2 = tmp/TSysTime2;

                tmp = (ctime - Last_Chg2) * (double)Na2 + pAvg_Na1 * pTime2;
                pTime2 += (ctime - Last_Chg2);
                pAvg_Na2 = tmp/pTime2;

            } 

            Last_Chg2 = ctime; 
        }

        if (!isFreeloader_)
            Na1 ++;
        else
            Na2 ++;

//        ping_timer_.resched(ping_interval_);
        ping_timer_.resched();

        // watchDog_.resched(watch_interval_);

        debug_stat("Membership %d\t%d\n",  Na1, Na2);
        gagent_->bkblock_.online();
        gagent_->fwblock_.online();
        gagent_->rcvRate_.online();
        degree_.online();
        fflush(NULL);
    }
    
    fflush(NULL);

    if (pTime1 > SMP_INTERVAL) 
    {
        debug_info("Membership non-freeloader %f\t%d\t%d\n", 
                   pAvg_Na1, Na1, Na2);
        pAvg_Na1 = 0;
        pTime1 = 0;
    }

    if (pTime2 > SMP_INTERVAL) 
    {
        debug_info("Membership freeloader %f\t%d\t%d\n", pAvg_Na2, Na1, Na2);
        pAvg_Na2 = 0;
        pTime2 = 0;
    }
}

//=============================================================================
/* bootstrap */
void GnutellaApp::bootstrap() 
{
    if (smpBoot_) 
    {
        if (bootSrv_)
        {
            smp_bootstrap();
        }
    }
    else if (bserver_list_.size() > 0) 
    {
        BootServerRec bi = bserver_list_.front();
        gagent_->Bootstrap(bi.addr_);
        bserver_list_.push_back(bi);
        bserver_list_.pop_front();
    }
}
//=============================================================================
/* use SmpBootServer for bootstrapping */
void GnutellaApp::smp_bootstrap() 
{
    BootstrapRes_t* res = bootSrv_->BootstrapRequest(addr_, this, EPeerLegacy);

    if (res != NULL)
    {
        BootstrapResult(res->cnt_, res->servents_);
        if (res->servents_)
        {
            free(res->servents_);
        }

        free(res);
    }
}
//=============================================================================
/* tries to connect to some known servents */
void GnutellaApp::connect(bool aTryBootserver, bool* aReloadTimer /* = 0 */) 
{
    int cnt = 0; 

    // TODO: AD
    printf("%d is aware of %u servents in connect\n", addr_, servent_cache_.size());

    int limit = max_deg_ - lpeers_.size();

    for (ServentMap_t::const_iterator i = servent_cache_.begin(); cnt < limit && i != servent_cache_.end(); ++i ) 
    {
        NodeAddr_t peer = i->addr_;

        // TODO: AD
        debug_stat("%d trying to connect to peer %d\n", addr_, peer);

        PeerMap_t::iterator pi = lpeers_.find(peer);
        if (pi==lpeers_.end()) 
        {
            if (gagent_->Connect(peer, CONN_TIMEOUT))
            {
                // successfully connected
            	cnt++;
            }
        }
    }

    if (aTryBootserver && cnt < limit)
    {
        printf("%d gonna boostrap\n", addr_);
        bootstrap();
    }
    if (aReloadTimer != 0)
    {
        if (cnt == limit)
        {
            *aReloadTimer = false;
        }
        else
        {
            *aReloadTimer = true;
        }
    }

//    if (trybootserver && !smpBoot_ && cnt < limit)   //not enough known servents
//        bootstrap();
}
//=============================================================================
/* periodically send PING to connected peers */
void GnutellaApp::ping() {
  double ctime; 

  debug_stat("%d sending ping\n", addr_);

  ctime = NOW;
  for(PeerMap_t::iterator i = lpeers_.begin(); i != lpeers_.end(); ++i) {
    if (ctime - i->second.lstamp_ > PingTimer::PING_INTERVAL) {
      gagent_->Ping(i->second.peer_, INIT_TTL);
      i->second.lstamp_ = ctime;
    }
  }
}

//=============================================================================
// upcalls from GnutellaAgent
/* Connection request to a peer succeeded */
void GnutellaApp::ConnectSucceeded(NodeAddr_t peer) 
{
    PeerMap_t::iterator pi = lpeers_.find(peer);
    if (pi==lpeers_.end()) 
    {
        PeerEntry *newentry = new PeerEntry(peer, NOW, FALSE);
        lpeers_.insert(PeerMap_t::value_type(peer, *newentry));

        degree_.increment(lpeers_.size());
        if (lpeers_.size()==1) 
        {
       	    // ping_timer_.resched(ping_interval_);
            // watchDog_.resched(watch_interval_);
            if (gagent_->gc_ && gagent_->gc_->status()==TIMER_IDLE) 
                gagent_->gc_->resched(DESC_TIMEOUT/2);
        }
    }

    if (state_ == PS_OUTSYS)
    {
        state_ = PS_OFFLINE;
    }

    if (state_ == PS_OFFLINE) 
    {
        printf("%d setState\n", __LINE__);
        setState(PS_IDLE);
        if (ac_)
            ac_->expire(NULL);
    }
}

//=============================================================================
/* Bootstrap request received*/
void GnutellaApp::BootstrapRequest(NodeAddr_t peer) {

  debug_info("bootstrap request received from %d\n", peer);

  if (isBootserver_) {

    // TODO: AD
    debug_stat("bootstrap request received from %d\n", peer);

    int i=0, size = servent_cache_.size();
    int cnt = MAX_BOOTREPLY>size ? size: MAX_BOOTREPLY;

    NodeAddr_t *addrs = (NodeAddr_t *)malloc(sizeof(NodeAddr_t)*cnt);
    for(ServentMap_t::iterator fi = servent_cache_.begin(); 
        fi != servent_cache_.end(); ++fi) {
      addrs[i] = fi->addr_;
      i++;
      if (i>=MAX_BOOTREPLY)
        break;
    }

    gagent_->Bootstrap_Reply(peer, cnt, addrs);
    if (find_servent(peer))
      remove_servent(peer);
    
    int prob = (int)((double)rand()/(double)RAND_MAX * 100.0);
     
    if (prob < PUBLISH_PROB) {
      ServentRec *newsrv;
      newsrv = new ServentRec(LISTEN_PORT, peer, 0, 0);
      servent_cache_.push_back(*newsrv);
    }

    while(servent_cache_.size() > KNOWNCACHE_LIMIT)
       servent_cache_.pop_front();

    free(addrs);
  }
}

//=============================================================================
/* Bootstrap results received */
void GnutellaApp::BootstrapResult(int cnt, NodeAddr_t *res) {
  int i;
  
  for(i=0; i<cnt; i++) {
    NodeAddr_t cur = res[i];

    if (!find_servent(cur) && cur != addr_) {
      ServentRec *newsrv;
      newsrv = new ServentRec(LISTEN_PORT, cur, 0, 0);
      servent_cache_.push_back(*newsrv);
    }
  }
  connect(false);
}

//=============================================================================
/* Bootstrap results received */
void GnutellaApp::BootcacheUpdate(NodeAddr_t peer) {
   if (!isBootserver_)
	return;

   if (!find_servent(peer))
	return;

   remove_servent(peer);
   ServentRec *newsrv;
   newsrv = new ServentRec(LISTEN_PORT, peer, 0, 0);
   servent_cache_.push_back(*newsrv);
}

//=============================================================================
/* connection request received*/
int GnutellaApp::ConnectionRequest(NodeAddr_t peer, Socket *sock) {

  debug_stat("%d recvd connection request from %d\n", addr_, peer);
  PeerMap_t::iterator i = lpeers_.find(peer);

  if (avail(EPeerAny) && i==lpeers_.end()) {
    gagent_->gnutella_ok(sock);

    PeerEntry *newentry = new PeerEntry(peer, NOW, FALSE);
    lpeers_.insert(PeerMap_t::value_type(peer, *newentry));
    degree_.increment(lpeers_.size());
    if (lpeers_.size()==1) {
      if (state_ == PS_OUTSYS)
        state_ = PS_OFFLINE;

      if (state_ == PS_OFFLINE) {
        printf("%d setState\n", __LINE__);
        setState(PS_IDLE);
	    if (ac_)
	      ac_->expire(NULL);
      }
//      ping_timer_.resched(ping_interval_);
      // watchDog_.resched(watch_interval_);
    }

    if (!find_servent(peer)) {
      ServentRec *srec = new ServentRec(LISTEN_PORT, peer, 0, 0);
      servent_cache_.push_back(*srec);
    }

    debug_stat("%d will accept conn req from %d\n", addr_, peer);
    return 1;
  } else {
    gagent_->gnutella_reject(sock);
    return 0;
  }
}

//=============================================================================
int GnutellaApp::UltraConnRequest(NodeAddr_t peer, Socket *socket) {
  printf("%d will reject ultra conn request from %d\n", addr_, peer);
  return FALSE;
}
//=============================================================================
int GnutellaApp::LeafConnRequest(NodeAddr_t peer, Socket *socket) {
  printf("%d will reject leaf conn request from %d\n", addr_, peer);
  return FALSE;
}
//=============================================================================
/* connection request rejected */
void GnutellaApp::ConnectionRejected(NodeAddr_t peer) {
  debug_info("connection request from %d to %d rejected\n", addr_, peer);

  if (find_servent(peer)) 
    remove_servent(peer);
}

//=============================================================================
/* connection failed, probably at network level */
void GnutellaApp::ConnectionFailed(NodeAddr_t peer) {
  debug_info("connection to %d failed\n", peer);

  if (find_servent(peer))
    remove_servent(peer);
}

//=============================================================================
/* connection request timed out */
void GnutellaApp::ConnectionTimeout(NodeAddr_t peer) {
  debug_info("connection from %d to %d timed out\n", addr_, peer);

  if (find_servent(peer))
    remove_servent(peer);
}

//=============================================================================
/* connection closed by peer */
void GnutellaApp::ConnectionClosed(NodeAddr_t peer) {
  debug_info("connection closed by peer %d\n", peer);

  for(PeerMap_t::iterator i=lpeers_.begin(); i != lpeers_.end(); ++i) {
    if (i->second.peer_ == peer) {
      lpeers_.erase(i);
      degree_.increment(lpeers_.size());
      return;
    }
  }
}

//=============================================================================
/* PING request received */
void GnutellaApp::PingRequest(NodeAddr_t peer, int ttl, char *id) {
   int i=0;
   int cnt = servent_cache_.size();

   // TODO: AD
   debug_stat("PingRequest from %d to %d\n", peer, addr_);

   int *sbytes = (int *)malloc(sizeof(int)*cnt);
   int *sfiles = (int *)malloc(sizeof(int)*cnt);
   NodeAddr_t *addrs = (NodeAddr_t *)malloc(sizeof(NodeAddr_t)*cnt);
   for(ServentMap_t::iterator si = servent_cache_.begin(); si!=servent_cache_.end(); si++) {
     sbytes[i] = si->byte_shared_;
     addrs[i] = si->addr_;
     sfiles[i] = si->file_shared_;
     i++;   }

   // TODO: AD
   debug_stat("%d sending pong to %d\n", addr_, peer);

   gagent_->Pong((int)peer, INIT_TTL, cnt, addrs, sbytes, sfiles, id);

   if (sbytes)
     free(sbytes);

   if (sfiles)
     free(sfiles);

}

//=============================================================================
/* PONG reply received */
void GnutellaApp::PongReply(NodeAddr_t peer, int ttl, char* payload) {
  tPong *pong = (tPong *)payload;

  // TODO: AD
  debug_stat("%d got a pong from %d\n", addr_, pong->addr_);

  if (lpeers_.size() < max_deg_ && pong->addr_!= addr_) {
    PeerMap_t::iterator pi = lpeers_.find(pong->addr_);
    if (pi==lpeers_.end()) {
      
      // TODO: AD
      debug_stat("%d attempting to connect to %d after pong\n", addr_, 
                                                                pong->addr_);
      gagent_->Connect(pong->addr_, CONN_TIMEOUT);
    }
  }

  if (find_servent(pong->addr_))
    return;

  if (pong->addr_!= addr_) {
    ServentRec *newsrv = new ServentRec(pong->port_, pong->addr_, 
                                        pong->byte_shared_, pong->file_shared_);
    servent_cache_.push_back(*newsrv);
    while(servent_cache_.size() >= KNOWNCACHE_LIMIT)
        servent_cache_.pop_front();
  }
}

//=============================================================================
/* Query Request received */
void GnutellaApp::QueryRequest(NodeAddr_t peer, int ttl, char* payload, char *id) {
  debug_info("QueryRequest from %d %d %d\n", peer, isFreeloader_, state_);
  double hitprob=0;

  rquery++; 
  if (isFreeloader_ || state_==PS_OFFLINE) 
    return;

  unsigned char nhit = (unsigned char)((float)rand()/(float)RAND_MAX *5.0);
  
  if (ac_) {
    int M = ac_->sys_->nFiles_;
    int fnum = atoi(payload);
    hitprob = (double)(M+1-fnum)/(double)(M+1);
  }

  double hit = (float)rand()/(float)RAND_MAX; //random probability of hit
  if (hitprob>0 && hit<hitprob || (hitprob==0 && hit<HIT_PROB))
    gagent_->QueryHit((int)peer, nhit, addr_, speed_, "", id); 
}

//=============================================================================
/* QueryHit received*/
void GnutellaApp::QueryHitReply(NodeAddr_t peer, int ttl, char *payload) {
  debug_info("QueryHit received by %d from %d\n", addr_, peer);

  if (ac_ && state_==PS_ACTIVE)
    ac_->expire(NULL);
}

//=============================================================================
/* available for more connections? */
bool GnutellaApp::avail(EPeerType) const
{
    return (lpeers_.size() < max_deg_);
}
//=============================================================================
/* configure a list of bootstrap servers from a file */
void GnutellaApp::setBootServer(FILE *file) {
  char line[20];
  NodeAddr_t servent;
  BootServerRec *brec;

  while(fgets((char *)&line, 20, file)) {
#ifdef PDNS
    Tcl &tcl = Tcl::instance();
    tcl.evalf("[Simulator instance] convert-ipaddr %s\n", line);
    servent = atoi(tcl.result()); 
#else
    servent = atoi(line);
#endif
    brec = new BootServerRec(servent, LISTEN_PORT);
    if (brec) 
      bserver_list_.insert(bserver_list_.end(), *brec);
  }
}

//=============================================================================
/* find a servent in the servent cache */
int GnutellaApp::find_servent(NodeAddr_t peer) {
  for(ServentMap_t::iterator si = servent_cache_.begin(); si != servent_cache_.end(); si++) {
    if (si->addr_==peer)
      return TRUE;
  }
  return FALSE;
}

//=============================================================================
/* remove a servent from the known servent cache */
void GnutellaApp::remove_servent(NodeAddr_t peer) {
  for(ServentMap_t::iterator si = servent_cache_.begin(); si != servent_cache_.end(); si++) {
    if (si->addr_==peer) {
      servent_cache_.erase(si);
      return;
    }
  }
}

//=============================================================================
/* send an update to the bootstrap server */ 
void GnutellaApp::update_bootcache() {
  if (smpBoot_)
	return;

  gagent_->UpdateBootcache();
}
//=============================================================================
void GnutellaApp::watchdogExpired(void)
{
    // TODO: AD
    printf("Watchdog expired for %d\n", addr_);

    bool reload_timer;

    connect(true, &reload_timer);
    update_bootcache();
    
    if (reload_timer)
    {
        printf("Will reload watchdog for %d\n", addr_);
        watchDog_.resched();
    }
    else
    {
        // TODO: AD
        printf("Disabling watchdog for %d\n", addr_);
    }
}
//=============================================================================
void GnutellaApp::pingExpired(void)
{
    printf("Ping timer expired for %d\n", addr_);
    ping();
    ping_timer_.resched();  
}
//=============================================================================
int GnutellaApp::command(int argc, const char*const* argv) {
  Tcl &tcl = Tcl::instance();
  const char* const cmd = argv[1];

  if (argc==2) {
    if (strcmp(cmd, "start")==0) {
      join();
      return TCL_OK;
    }
    else if (strcmp(cmd, "join")==0) {
      join();
      return TCL_OK;
    }
    else if (strcmp(cmd, "cleanup")==0) {
      return cleanup();
    }
    else if (strcmp(cmd, "leave")==0) {
      leave();
      return TCL_OK;
    }
    else if (strcmp(cmd, "stop")==0) {
      stop();
      return TCL_OK;
    }
    else if (strcmp(cmd, "statistics")==0) {
      gagent_->statistics();
      return TCL_OK;
    }
  }
  else if (argc==3) {
    if (strcmp(cmd, "set-agent")==0) {
      GnutellaAgent *tmp = (GnutellaAgent *)tcl.lookup(argv[2]);
      if (tmp) {
        gagent_ = tmp;
        agent_ = tmp; 
        return TCL_OK;
      }
      return TCL_ERROR;
    }
    else if (strcmp(cmd, "search")==0) {
      search((char *)argv[2]);
      return TCL_OK;
    }
    else if (strcmp(cmd, "use-bootserver")==0) {
      FILE *file = fopen(argv[2], "r");
      if (file) {
        setBootServer(file);
        return TCL_OK;
      }
      return TCL_ERROR;
    }
    else if (strcmp(cmd, "use-smpbootserver")==0) {
      SmpBootServer *tmp = (SmpBootServer *)tcl.lookup(argv[2]);
      if (tmp!=NULL) {
        bootSrv_ = tmp;
        smpBoot_ = TRUE;
        return TCL_OK;
      }
      return TCL_ERROR;
    }
  }
  else if (argc==4) {
    if (strcmp(cmd, "attach-peersys")==0) {
      PeerSys *tmp = (PeerSys *)tcl.lookup(argv[2]);
      if (tmp!=NULL) {
        printf("Adding activity controller for %d\n", addr_);
        ac_ = new ActivityController(tmp, this, atoi(argv[3]));
        return TCL_OK;
      }
      return TCL_ERROR;
    }
  }

  return TCL_ERROR;
}

//=============================================================================
static class GnutellaAppClass: public TclClass {
public:
  GnutellaAppClass(): TclClass("PeerApp/GnutellaApp") {}
  TclObject* create(int argc, const char*const* argv) {
    return (new GnutellaApp(atoi(argv[4])));
  }
}class_gnutellaapp;
//=============================================================================

