// File: ultrapeer.cc
// Original by Qi He
// Modified by Andre Dufour

#include "ultrapeer.h"
#include "gnut_types.h"
#include "gnutellaapp.h"
#include "gnutellaagent.h"
#include "activitycontroller.h"
#include "pingtimer.h"
#include "watchdog.h"
#include "smpbootserver.h"
#include "gc.h"


//=============================================================================
/**** Ultrapeer Application****/
Ultrapeer::
Ultrapeer(NodeAddr_t aAddr)
: GnutellaApp(aAddr),
  max_peer_(0),   // controlled in TCL
  max_leaves_(0), // controlled in TCL
  max_legacy_(0), // controlled in TCL
  npeers_(0),
  nleaves_(0),
  nlegacy_(0)
{
    bind("max_leaves_", &max_leaves_);
    bind("max_peer_", &max_peer_);
    bind("max_legacy_", &max_legacy_);
    
    printf("Creating ultrapeer %p for node %d\n", this, aAddr);
};

//=============================================================================
Ultrapeer::~Ultrapeer(void)
{ 
    // TODO: AD
    printf("Destroying ultrapeer %d\n", addr_);
}
//=============================================================================
void Ultrapeer::connect(bool aTryBootserver, bool* aReloadTimer /* = 0 */) 
{
    double prob;
    int cnt = 0;
    int limit = max_peer_ - npeers_;

    if(avail(EPeerUltra) == false && avail(EPeerLegacy) == false)
    {
        return;
    }

    for (ServentMap_t::const_iterator i = servent_cache_.begin(); 
         cnt < limit && i != servent_cache_.end(); 
         ++i ) 
    {
        NodeAddr_t peer = i->addr_;
        PeerMap_t::iterator pi = lpeers_.find(peer);
        if(pi==lpeers_.end()) 
        {
            if (gagent_->Connect(peer, CONN_TIMEOUT))
            {
                // Successfully connected
            	cnt++;
            }
        }
    }

    if (aTryBootserver && cnt < limit)
    {
        bootstrap();
    }
    if (aReloadTimer != 0)
    {
        if (cnt != limit)
        {
            *aReloadTimer = true;
        }
        else
        {
            *aReloadTimer = false;
        }
    }
//    if(trybootserver && !smpBoot_ && cnt < limit)
//      bootstrap();
}

//=============================================================================
/* periodically send PING to connected peers */
void Ultrapeer::ping() 
{
    double ctime = NOW;

    printf("%d (ultra) sending ping\n", addr_);

    for(PeerMap_t::iterator i = lpeers_.begin(); i != lpeers_.end(); ++i) 
    {
        if(ctime - i->second.lstamp_ > PingTimer::PING_INTERVAL) 
        {
            gagent_->Ping(i->second.peer_, INIT_TTL);
            i->second.lstamp_ = ctime;
        }
    }
//  ping_timer_.resched(ping_interval_);  
}

//=============================================================================
// upcalls from GnutellaAgent
/* Connection request to a peer succeeded */
void Ultrapeer::ConnectSucceeded(NodeAddr_t peer) {
  PeerMap_t::iterator pi = lpeers_.find(peer);
  if(pi==lpeers_.end()) {
    PeerEntry *newentry = new PeerEntry(peer, NOW, TRUE);
    lpeers_.insert(PeerMap_t::value_type(peer, *newentry));
    degree_.increment(lpeers_.size());

    if(lpeers_.size()==1) {
// 	ping_timer_.resched(ping_interval_);
	// watchDog_.resched(watch_interval_);
	if(gagent_->gc_ && gagent_->gc_->status()==TIMER_IDLE) 
	  gagent_->gc_->resched(DESC_TIMEOUT/2);
    }
  }

  if(state_ == PS_OUTSYS)
    state_ = PS_OFFLINE;

  if(state_ == PS_OFFLINE) {
    printf("%d ultra setState\n", __LINE__);
    setState(PS_IDLE);
    if(ac_)
      ac_->expire(NULL);
  }
}

//=============================================================================
/* connection request received*/
int Ultrapeer::ConnectionRequest(NodeAddr_t peer, Socket *sock) {

  // TODO: AD
  printf("%d recvd connection request from %d\n", addr_, peer);

  PeerMap_t::iterator i = lpeers_.find(peer);

  if(avail(EPeerLegacy) && i==lpeers_.end()) {
    gagent_->gnutella_ok(sock);

    PeerEntry *newentry = new PeerEntry(peer, NOW, TRUE);
    lpeers_.insert(PeerMap_t::value_type(peer, *newentry));
    degree_.increment(lpeers_.size());
    nlegacy_++;

    if(lpeers_.size()==1) {
        if(state_ == PS_OUTSYS)
            state_ = PS_OFFLINE;

        if(state_ == PS_OFFLINE) {
            printf("%d ultra setState\n", __LINE__);
            setState(PS_IDLE);
            if(ac_)
                ac_->expire(NULL);
        }
        
        // ping_timer_.resched(ping_interval_);
        // watchDog_.resched(watch_interval_);
    }

    if(!find_servent(peer)) {
      ServentRec *srec = new ServentRec(LISTEN_PORT, peer, 0, 0);
      servent_cache_.push_back(*srec);
    }

    printf("%d will accept conn req from %d\n", addr_, peer);
    return 1;
  } else {
    gagent_->gnutella_reject(sock);
    printf("%d will reject conn req from %d\n", addr_, peer);
    return 0;
  }
}

//=============================================================================
/* connection request received*/
int Ultrapeer::UltraConnRequest(NodeAddr_t peer, Socket *sock) {

  // TODO: AD
  printf("%d recvd ultra conn request from %d\n", addr_, peer);

  PeerMap_t::iterator i = lpeers_.find(peer);

  if(avail(EPeerUltra) && i==lpeers_.end()) {
    gagent_->gnutella_ok(sock);

    PeerEntry *newentry = new PeerEntry(peer, NOW, TRUE);
    lpeers_.insert(PeerMap_t::value_type(peer, *newentry));
    degree_.increment(lpeers_.size());
    npeers_++;

    if(lpeers_.size()==1) {
        if(state_ == PS_OUTSYS)
            state_ = PS_OFFLINE;

        if(state_ == PS_OFFLINE) {
            printf("%d ultra setState\n", __LINE__);
            setState(PS_IDLE);
            if(ac_)
                ac_->expire(NULL);
        }

        // ping_timer_.resched(ping_interval_);
        // watchDog_.resched(watch_interval_);
    }

    if(!find_servent(peer)) {
      ServentRec *srec = new ServentRec(LISTEN_PORT, peer, 0, 0);
      servent_cache_.push_back(*srec);
    }
    
    // TODO: AD
    printf("%d will accept ultra conn request from %d\n", addr_, peer);
    
    return 1;
  } else {
    gagent_->gnutella_reject(sock);
    
    // TODO: AD
    printf("%d will reject ultra conn request from %d\n", addr_, peer);
    
    return 0;
  }
}

//=============================================================================
/* connection request received*/
int Ultrapeer::LeafConnRequest(NodeAddr_t peer, Socket *sock) {

  // TODO: AD
  printf("%d recvd leaf conn request from %d\n", addr_, peer);

  if(state_ == PS_OUTSYS) {
    // TODO: AD
    printf("%d is OUTSYS and will reject leaf conn request from %d\n", addr_, peer);

    gagent_->gnutella_reject(sock);
    return FALSE;
  }

  PeerMap_t::iterator i = lpeers_.find(peer);
  if((avail(EPeerLeaf)) && i==lpeers_.end()) {
    gagent_->gnutella_ok(sock);

    PeerEntry *newentry = new PeerEntry(peer, NOW, TRUE);
    lpeers_.insert(PeerMap_t::value_type(peer, *newentry));
    degree_.increment(lpeers_.size());
    nleaves_++;

    if(!find_servent(peer)) {
      ServentRec *srec = new ServentRec(LISTEN_PORT, peer, 0, 0);
      servent_cache_.push_back(*srec);
    }

    // TODO: AD
    printf("%d will accpt leaf conn req from %d\n", addr_, peer);

    return 1;
  } else {
    gagent_->gnutella_reject(sock);

    // TODO: AD
    printf("%d will reject leaf conn req from %d\n", addr_, peer);

    return 0;
  }
}

//=============================================================================
/* PONG reply received */
void Ultrapeer::PongReply(NodeAddr_t peer, int ttl, char* payload) {
  tPong *pong = (tPong *)payload;

  if((avail(EPeerLegacy) || avail(EPeerUltra)) && pong->addr_!= addr_) {
    PeerMap_t::iterator pi = lpeers_.find(pong->addr_);
    if(pi==lpeers_.end()) {
      gagent_->Connect(pong->addr_, CONN_TIMEOUT);
    }
  }

  if(find_servent(pong->addr_))
    return;
  
  if(servent_cache_.size() < KNOWNCACHE_LIMIT && pong->addr_!= addr_) {
    ServentRec *newsrv = new ServentRec(pong->port_, pong->addr_, pong->byte_shared_, pong->file_shared_);
    servent_cache_.push_back(*newsrv);
  }
}

//=============================================================================
/* bootstrap */
void Ultrapeer::bootstrap() {

  if(smpBoot_) {
  BootstrapRes_t *res;

  res = bootSrv_->BootstrapRequest(addr_, this, EPeerUltra);
  if(res==NULL)
    return;

  BootstrapResult(res->cnt_, res->servents_);
  if(res->servents_)
    free(res->servents_);

  free(res);
  } else
    GnutellaApp::bootstrap();
}

//=============================================================================
/* Ultrapeer availability for different types of peer connections */
bool Ultrapeer::avail(EPeerType aType) const
{
    return (aType == EPeerLegacy && nlegacy_ < max_legacy_) || 
           (aType == EPeerUltra && npeers_ < max_peer_)     ||
           (aType == EPeerLeaf && nleaves_ < max_leaves_);
}
//=============================================================================
static class UltrapeerClass: public TclClass {
public:
  UltrapeerClass(): TclClass("PeerApp/GnutellaApp/Ultrapeer") {}
  TclObject* create(int argc, const char*const* argv) {
    return (new Ultrapeer(atoi(argv[4])));
  }
}class_ultrapeer;
//=============================================================================

