// File: gnutellaapp.h
// Original by Qi He
// Modified by Andre Dufour

#ifndef __GNUTELLAAPP_H__
#define __GNUTELLAAPP_H__

#include "gnut_types.h"
#include "peerapp.h"
#include "basicstat.h"
#include "ignutellaapp.h"
#include "pingtimer.h"
#include "watchdog.h"
#include "basicstat.h"
#include "gnuteventlistener.h"
#include "gnutstats.h"

class PingTimer;
class PongRes;
class GnutHierApp;
class GnutQueryHitMsg;
class IVivaldiManager;
class GnutCoord;

//=============================================================================
/* GnutellaApp: Gnutella application, such as LimeWire, Gnucleus */
class GnutellaApp: public PeerApp, public IGnutellaApp
{
public:
    // = FOUNDATION

    GnutellaApp(NodeAddr_t addr, const char* const parent, int aUsesVivaldi);
    virtual ~GnutellaApp();


    // = ACTION    

    /*user command interface */
    virtual void join();
    virtual void leave();
    virtual void stop();  //"stop" is forced "leave" but its current 
                          //implementation is identical to "leave"
    virtual void search(); 
    virtual void share();
    virtual void disconnect(NodeAddr_t node);
    virtual void update_bootcache();
    virtual void setState(PeerState_t newstate);
    virtual int cleanup(); // to be called before the object is destroyed.
                           // Can perform stuff that uses the TCL side of
                           // the split object. That's not possible in d'tor.

    /* upcalls provided to GnutellaAgent*/
    virtual void ConnectSucceeded(NodeAddr_t peer, EGnutApp type,
        const GnutCoord& remoteCoord, double remoteError);
    void ConnectionRejected(NodeAddr_t peer);
    void ConnectionFailed(NodeAddr_t peer);
    void ConnectionTimeout(NodeAddr_t peer);
    void ConnectionClosed(NodeAddr_t peer);
    virtual int  ConnectionRequest(NodeAddr_t peer, Socket* socket, const GnutCoord& remoteCoord, double remoteError, double timeSent);
    virtual int  UltraConnRequest(NodeAddr_t peer, Socket* socket, const GnutCoord&, double, double);
    virtual int  LeafConnRequest(NodeAddr_t peer, Socket* socket, const GnutCoord&, double, double);
    virtual void PingRequest(NodeAddr_t peer, int ttl, DescriptorID_t id);
    virtual void PongReply(NodeAddr_t peer, const PongRes& payload);
    virtual void QueryRequest(NodeAddr_t peer, char* payload, DescriptorID_t id,
                              const GnutQueryMsg& aQuery);
    virtual void QueryHitReply(NodeAddr_t peer, const GnutQueryHitMsg& hit);
    virtual void BootstrapRequest(NodeAddr_t peer);
    virtual void BootstrapResult(int, NodeAddr_t* );
    virtual void BootcacheUpdate(NodeAddr_t peer);

    // Watchdog callback
    void watchdogExpired(void);

    // Ping timer callback
    void pingExpired(void);

    /* internal operations related to peer connections*/
    virtual void bootstrap();
    virtual void maintenance();
    virtual void ping();
    virtual bool avail(EPeerType aType) const;
    virtual void smp_bootstrap();
    void setBootServer(FILE* );
    int find_servent(NodeAddr_t);
    void remove_servent(NodeAddr_t);
    void setHasContent(bool aHasContent);

    virtual void notifyRecv(NodeAddr_t src, const GnutCoord& remoteCoord, double timeDiff, double remoteError);

protected:
    virtual void connect(bool aTryBootserver, 
                         bool* aTimerReload = 0 // indicates whether the watchdog
                                                // should be rescheduled
                        );
public:

    virtual int command(int argc, const char*const* argv);

    // = DATA
// TODO: AD: ping_interval_, watch_interval_ and max_deg_ should be made
//           static, but the TCL binding is trickier. Later...

    int isBootserver_;
    int max_deg_;
    PingTimer ping_timer_; 
    WatchDog watchDog_;
    GnutellaAgent* gagent_;
    ServentMap_t servent_cache_;
    BasicStat degree_;
    BServerList_t bserver_list_;
    int smpBoot_;
    SmpBootServer* bootSrv_;
    GnutHierApp& parent_;
    IVivaldiManager* const mVivaldi;

protected:
    bool hasContent_; // Does this servent have the content requested by
                      // queries? If true, it will send a queryhit when
                      // it gets a query.

    // If the node with the given coordinates is closer than one of
    // the neighbours we are connected to, disconnect from our farthest
    // neighbour and return true. Else return false.
    // Side-effect: adds the new conn to the list if it is closer.
    // TODO: AD - that's a bit ugly 
    bool removeOldConnIfCloser(
        NodeAddr_t aPeer, EPeerType aPeerType, const GnutCoord& aRemoteCoord, 
        double aRemoteError);

    double estimateDistance(const GnutCoord& aRemoteCoord) const;

private:
    const GnutEventListener<GnutellaApp> watchdogListener_;
    const GnutEventListener<GnutellaApp> pingListener_;

protected:

    // Keep track of the estimated RTT to our neighbours
    // Key = node address
    // Value = {worst case distance to that node, 
    //          node type,
    //          time we established a connection to that node}
    struct SPeerDistance
    {
        SPeerDistance(EPeerType aType, double aDist, double aTime) 
        : mDistance(aDist),
          mPeerType(aType),
          mTimeFirstConnected(aTime)
        { }

        SPeerDistance() 
        : mDistance(0.0),
          mPeerType(EPeerAny),
          mTimeFirstConnected(0.0)
        { }

        double    mDistance;
        EPeerType mPeerType;
        double    mTimeFirstConnected;
    };

    typedef std::map<NodeAddr_t, SPeerDistance> PeerDistances;

    PeerDistances mPeerDistances;

    GnutStats& mGnutStats;


    // Minimum time we must be connected to a peer before considering
    // dumping its connection for a more favourable one.
    static const double GNUT_MIN_CONNECTION_TIME;

private:
    // = FOUNDATION

    GnutellaApp(const GnutellaApp&);
    GnutellaApp& operator=(const GnutellaApp&);
};


//=============================================================================

#endif /* __GNUTELLAAPP_H__ */

