// File: gnutmsgparser.cc
// Original by Qi He
// Modified by Andre Dufour


#include "gnutmsgparser.h"
#include <stdlib.h>
#include "gnut_types.h"
#include "gnut_util.h"
#include "ignutmsg.h"
#include "gnutpingmsg.h"
#include "gnutquerymsg.h"
#include "gnutqueryhitmsg.h"
#include "gnutpongmsg.h"
#include "gnutbootstrapmsg.h"
#include "gnutbootstrapresmsg.h"
#include "gnutpayload.h"
#include "gnutokmsg.h"

//============================================================================
// Private
GnutMsgParser::
GnutMsgParser(void) 
{ }
//============================================================================
// Private
GnutMsgParser::
~GnutMsgParser(void)
{ }
//============================================================================
GnutMsgParser&
GnutMsgParser::
instance(void)
{
    // WARNING: not thread safe. That's ok for ns-2 though.
    static GnutMsgParser parser;

    return parser;
}
//============================================================================
/* Gnutella Message Parsing */
GnutMsgParser::EGnutMsgType 
GnutMsgParser::
parse(const unsigned char* data, GnutDescHdr* header, GnutPayload* payload,
      GnutCoord* coord, double* error, double* timeSent) const
{
    GNUT_ASSERT(data != 0);
    EGnutMsgType msg_type = *(reinterpret_cast<const EGnutMsgType*>(data));

    switch (msg_type)
    {
    default:
        // Bad!
        GNUT_ASSERT(0);
        break;

    case GNUT_PING:
        {
            GnutPingMsg ping(data);
            if (coord != 0)
                *coord  = ping.getCoordinates();
            if (error != 0)
                *error  = ping.getError();
            if (timeSent != 0)
                *timeSent = ping.getTimeSent();
            if (header != 0)
                *header = ping.getHeader();
        }
        break;

    case GNUT_QUERY:
        {
            GnutQueryMsg query(data);
            if (coord != 0)
                *coord  = query.getCoordinates();
            if (error != 0)
                *error  = query.getError();
            if (timeSent != 0)
                *timeSent = query.getTimeSent();
            if (header != 0)
                *header = query.getHeader();
        }
        break;

    case GNUT_QUERYHIT:
        {
            GnutQueryHitMsg hit(data);
            if (coord != 0)
                *coord  = hit.getCoordinates();
            if (error != 0)
                *error  = hit.getError();
            if (timeSent != 0)
                *timeSent = hit.getTimeSent();
            if (header != 0)
                *header = hit.getHeader();
        }
        break;

    case GNUT_PONG:
        {
            GnutPongMsg pong(data);
            if (header != 0)
                *header = pong.getHeader();
            if (payload != 0)
                payload->setPongRes(pong.getResult());
            if (coord != 0)
                *coord  = pong.getCoordinates();
            if (error != 0)
                *error  = pong.getError();
            if (timeSent != 0)
                *timeSent = pong.getTimeSent();
        }
        break;


    case GNUT_BOOTSTRAP_RES:
        {
            GnutBootstrapResMsg bsr(data);
            if (coord != 0)
                *coord  = bsr.getCoordinates();
            if (error != 0)
                *error  = bsr.getError();
            if (timeSent != 0)
                *timeSent = bsr.getTimeSent();
            if (payload != 0)
                payload->setBootstrapRes(bsr.getResult());
        }
        break;

    case GNUT_BOOTSTRAP:
    case GNUT_LEGACY_CONNECT:
    case GNUT_OK:
    case GNUT_REJ:
    case GNUT_ULTRA_CONNECT:
    case GNUT_LEAF_CONNECT:
    case GNUT_ULTRA_OK:
    case GNUT_LEAF_OK:
    case GNUT_BOOTCACHE_UPDATE:
    case GNUT_ULTRA_LEAF_DIR:
        // This is a bit ugly, but they are all the same inside anyway...
        {
            GnutOkMsg ok(data);
            if (coord != 0)
                *coord = ok.getCoordinates();
            if (error != 0)
                *error  = ok.getError();
            if (timeSent != 0)
                *timeSent = ok.getTimeSent();
        }
        break;

    } // switch(msg_type)

    return msg_type;
}
//============================================================================
/* compose a PacketData with a Gnutella message as the payload         */
/* called with message type and length, as well as some optional parms */
PacketData* 
GnutMsgParser::newpacket(const IGnutMsg& aMsg) const
{
    PacketData* userdata = new PacketData(aMsg.getSize());

    aMsg.writeToPacket(userdata->data());

    return userdata;
}
//============================================================================
double 
GnutMsgParser::
getNewGuid(void)
{
    return ++mGuid;
}
//============================================================================

