// File: ultrapeer.cc
// Original by Qi He
// Modified by Andre Dufour

#include "ultrapeer.h"
#include "gnut_types.h"
#include "gnut_util.h"
#include "gnutellaapp.h"
#include "gnutellaagent.h"
#include "activitycontroller.h"
#include "pingtimer.h"
#include "watchdog.h"
#include "smpbootserver.h"
#include "gc.h"
#include "gnutpongmsg.h"
#include "gnutbootstrapresmsg.h"
#include "gnuthierapp.h"
#include "gnutstats.h"
#include "gnutcoord.h"
#include "vivaldimanager.h"


//=============================================================================
/**** Ultrapeer Application****/
Ultrapeer::
Ultrapeer(NodeAddr_t aAddr, const char* const parent, int aUsesVivaldi)
: GnutellaApp(aAddr, parent, aUsesVivaldi),
  max_peer_(0),   // controlled in TCL
  max_leaves_(0), // controlled in TCL
  max_legacy_(0), // controlled in TCL
  npeers_(0),
  nleaves_(0),
  nlegacy_(0)
{
    bind("max_leaves_", &max_leaves_);
    bind("max_peer_", &max_peer_);
    bind("max_legacy_", &max_legacy_);
    
    GnutUtil::gnutTrace("Creating ultrapeer %p for node %d\n", this, aAddr);
};

//=============================================================================
Ultrapeer::~Ultrapeer(void)
{ 
    // TODO: AD
    GnutUtil::gnutTrace("Destroying ultrapeer %d\n", addr_);
}
//=============================================================================
void Ultrapeer::connect(bool aTryBootserver, bool* aReloadTimer /* = 0 */) 
{
    double prob;
    int cnt = 0;
    int limit = max_peer_ - npeers_;

    if(avail(EPeerUltra) == false && avail(EPeerLegacy) == false)
    {
        return;
    }

    for (ServentMap_t::const_iterator i = servent_cache_.begin(); 
         cnt < limit && i != servent_cache_.end(); 
         ++i ) 
    {
        NodeAddr_t peer = i->addr_;
        PeerMap_t::iterator pi = lpeers_.find(peer);
        if(pi==lpeers_.end()) 
        {
            if (gagent_->Connect(peer, CONN_TIMEOUT))
            {
                // Successfully connected
            	cnt++;
            }
        }
    }

    if (aTryBootserver && cnt < limit)
    {
        bootstrap();
    }
    if (aReloadTimer != 0)
    {
        if (cnt != limit)
        {
            *aReloadTimer = true;
        }
        else
        {
            *aReloadTimer = false;
        }
    }
//    if(trybootserver && !smpBoot_ && cnt < limit)
//      bootstrap();
}

//=============================================================================
/* periodically send PING to connected peers */
void Ultrapeer::ping() 
{
    double ctime = NOW;

    for(PeerMap_t::iterator i = lpeers_.begin(); i != lpeers_.end(); ++i) 
    {
        if(ctime - i->second.lstamp_ > PingTimer::PING_INTERVAL) 
        {
            gagent_->Ping(i->second.peer_, INIT_TTL);
            i->second.lstamp_ = ctime;
        }
    }
//  ping_timer_.resched(ping_interval_);  
}

//=============================================================================
// upcalls from GnutellaAgent
/* Connection request to a peer succeeded */
void Ultrapeer::ConnectSucceeded(NodeAddr_t peer, EGnutApp type,
    const GnutCoord& remoteCoord, double remoteError) 
{
  PeerMap_t::iterator pi = lpeers_.find(peer);
  if(pi==lpeers_.end()) 
  {
    PeerEntry *newentry = new PeerEntry(peer, NOW, type);
    lpeers_.insert(PeerMap_t::value_type(peer, *newentry));
    mGnutStats.setPeerConns(addr_, lpeers_.size());
    degree_.increment(lpeers_.size());

    // Add this new neighbour's distance info
    const double combined_error = (1.0 + remoteError) * 
                                  (1.0 + mVivaldi->getLocalError());

    const double worst_case_distance = 
        combined_error * (remoteCoord - mVivaldi->getCoordinates()).abs();

    // Ugly mapping
    EPeerType peer_type = type == EGnutAppLeaf 
                          ?  EPeerLeaf
                          :  type == EGnutAppUltrapeer
                          ?  EPeerUltra
                          :  type == EGnutAppLegacy
                          ?  EPeerLegacy
                          :  EPeerAny;

    GNUT_ASSERT(peer_type != EPeerAny);

    SPeerDistance dist(peer_type, worst_case_distance, GnutUtil::now());
    mPeerDistances[peer] = dist;



    if(lpeers_.size()==1) 
    {
      if(gagent_->gc_ && gagent_->gc_->status()==TIMER_IDLE) 
        gagent_->gc_->resched(DESC_TIMEOUT/2);
    }
  }

  if(state_ == PS_OUTSYS)
    state_ = PS_OFFLINE;

  if(state_ == PS_OFFLINE) 
  {
    setState(PS_IDLE);
    if(ac_)
      ac_->expire(NULL);
  }
}
//=============================================================================
// Connected to a peer that offered leaf guidance
void Ultrapeer::UltraLeafGuidance(NodeAddr_t peer) 
{
    GnutUtil::gnutTrace("Ultrapeer %d got leaf guidance from ultrapeer %d\n", addr_, peer);

    // Per spec: leaf guidance ignored if we have leaves.
    if (nleaves_ == 0)
    {
        // TODO: AD
        GnutUtil::gnutTrace("%d will accept the leaf guidance\n", addr_);


        // Spec says to drop all gnutella 0.4 connections, but we must drop all
        // ultrapeer connections too, because at the time that spec was written,
        // leaves were only allowed to connect to one ultrapeer. Hence, we drop
        // all connections.
        disconnect(GnutellaAgent::ALL_NEIGHBOURS);

        
        // Now, we must become a leaf
        GnutUtil::gnutTrace("Ultrapeer %d becoming a leaf after leaf guidance from %d\n",
               addr_, peer);

        parent_.leafGuidance(peer);

        // TODO: AD actually become leaf!
        //       then send leaf conn request to peer
    }
}
//=============================================================================
// Recv'd connection request from legacy peer
int Ultrapeer::ConnectionRequest(NodeAddr_t peer, Socket *sock, const GnutCoord& coord, double remoteError, double) {

  PeerMap_t::iterator i = lpeers_.find(peer);

  if((avail(EPeerLegacy) && i==lpeers_.end()) || 
     removeOldConnIfCloser(peer, EPeerLegacy, coord, remoteError)) {

        const double worst_case_distance = estimateDistance(coord);
        SPeerDistance dist(EPeerLegacy, worst_case_distance, GnutUtil::now());
        mPeerDistances[peer] = dist;
        GnutUtil::gnutTrace("%d recording distance to %d as %f\n",
                            addr_, peer, worst_case_distance);

   
        gagent_->gnutella_ok(sock);

        PeerEntry *newentry = new PeerEntry(peer, NOW, EGnutAppLegacy);
        lpeers_.insert(PeerMap_t::value_type(peer, *newentry));
        mGnutStats.setPeerConns(addr_, lpeers_.size());
        degree_.increment(lpeers_.size());
        nlegacy_++;

        if(lpeers_.size()==1) {
            if(state_ == PS_OUTSYS)
                state_ = PS_OFFLINE;

            if(state_ == PS_OFFLINE) {
                setState(PS_IDLE);
                if(ac_)
                    ac_->expire(NULL);
            }
        }

        if(!find_servent(peer)) {
          ServentRec *srec = new ServentRec(LISTEN_PORT, peer);
          servent_cache_.push_back(*srec);
        }

        return 1;
  } else {
    gagent_->gnutella_reject(sock);
    return 0;
  }
}

//=============================================================================
/* connection request received*/
int Ultrapeer::UltraConnRequest(NodeAddr_t peer, Socket *sock, const GnutCoord& coord, double remoteError, double) 
{
    PeerMap_t::iterator i = lpeers_.find(peer);

    if (i == lpeers_.end())
    {
        if (nleaves_ != 0)
        {
            if(avail(EPeerUltra) || 
               removeOldConnIfCloser(peer, EPeerUltra, coord, remoteError)) 
            {
                const double worst_case_distance = 
                    estimateDistance(coord);
                SPeerDistance dist(EPeerUltra, worst_case_distance, GnutUtil::now());
                mPeerDistances[peer] = dist;
                GnutUtil::gnutTrace("%d recording distance to %d as %f\n",
                                    addr_, peer, worst_case_distance);

                gagent_->gnutella_ok(sock);

                PeerEntry *newentry = new PeerEntry(peer, NOW, 
                                                    EGnutAppUltrapeer);
                lpeers_.insert(PeerMap_t::value_type(peer, *newentry));
                mGnutStats.setPeerConns(addr_, lpeers_.size());
                degree_.increment(lpeers_.size());
                npeers_++;

                if(lpeers_.size()==1) 
                {
                    if(state_ == PS_OUTSYS)
                    {
                        state_ = PS_OFFLINE;
                    }

                    if(state_ == PS_OFFLINE) 
                    {
                        setState(PS_IDLE);
                        if(ac_)
                        {
                            ac_->expire(NULL);
                        }
                    }

                }

                if(!find_servent(peer)) 
                {
                    ServentRec *srec = new ServentRec(LISTEN_PORT, peer);
                    servent_cache_.push_back(*srec);
                }
              
                return 1;
            } 
            else 
            {
                gagent_->gnutella_reject(sock);
              
                return 0;
            }

        }
        else
        {
// Leaf guidance disabled for now            
#ifdef LEAF_GUIDANCE
            // We don't have enough leaves. Issue leaf guidance.
            gagent_->gnutella_leaf_guidance(sock);
#else
            GnutUtil::gnutTrace("%d would have issued leaf guidance, but"
                                " leaf guidance is disabled.\n", addr_);
#endif /* LEAF_GUIDANCE */

            return 0;
        }
    }
    else 
    {
        gagent_->gnutella_reject(sock);
      
        return 0;
    }

}

//=============================================================================
/* connection request received*/
int Ultrapeer::LeafConnRequest(NodeAddr_t peer, Socket *sock, const GnutCoord& coord, double remoteError, double) {
#if 0
  if(state_ == PS_OUTSYS) {
    // TODO: AD
    GnutUtil::gnutTrace("%d is OUTSYS and will reject leaf conn request from %d\n", addr_, peer);

    gagent_->gnutella_reject(sock);
    return FALSE;
  }
#endif /* 0 */

  PeerMap_t::iterator i = lpeers_.find(peer);
  if(((avail(EPeerLeaf)) && i==lpeers_.end()) ||
     removeOldConnIfCloser(peer, EPeerLeaf, coord, remoteError)) {
    // TODO: AD. Is this ok?
    if (state_ == PS_OUTSYS) {
        state_ = PS_OFFLINE;
    }

    if(state_ == PS_OFFLINE) 
    {
      setState(PS_IDLE);
      if(ac_)
        ac_->expire(NULL);
    }


    const double worst_case_distance = 
        estimateDistance(coord);
    SPeerDistance dist(EPeerLeaf, worst_case_distance, GnutUtil::now());
    mPeerDistances[peer] = dist;
    GnutUtil::gnutTrace("%d recording distance to %d as %f\n",
                        addr_, peer, worst_case_distance);

    gagent_->gnutella_ok(sock);

    PeerEntry *newentry = new PeerEntry(peer, NOW, EGnutAppLeaf);
    lpeers_.insert(PeerMap_t::value_type(peer, *newentry));
    mGnutStats.setPeerConns(addr_, lpeers_.size());
    degree_.increment(lpeers_.size());
    nleaves_++;

    if(!find_servent(peer)) {
      ServentRec *srec = new ServentRec(LISTEN_PORT, peer);
      servent_cache_.push_back(*srec);
    }

    return 1;
  } else {
    gagent_->gnutella_reject(sock);

    return 0;
  }
}

//=============================================================================
/* PONG reply received */
void Ultrapeer::PongReply(NodeAddr_t peer, const PongRes& pong) 
{
  if((avail(EPeerLegacy) || avail(EPeerUltra)) && pong.addr_!= addr_) 
  {
    PeerMap_t::iterator pi = lpeers_.find(pong.addr_);
    if(pi==lpeers_.end()) 
    {
      gagent_->Connect(pong.addr_, CONN_TIMEOUT);
    }
  }

  if(find_servent(pong.addr_))
    return;
  
  if(servent_cache_.size() < KNOWNCACHE_LIMIT && pong.addr_!= addr_) 
  {
    ServentRec *newsrv = new ServentRec(pong.port_, pong.addr_);
    servent_cache_.push_back(*newsrv);
  }
}

//=============================================================================
/* bootstrap */
void Ultrapeer::bootstrap() 
{
    if(smpBoot_) 
    {
        BootstrapRes* res = bootSrv_->BootstrapRequest(addr_, this, EPeerUltra);

        if(res != 0)
        {
            BootstrapResult(res->cnt_, res->servents_);
            delete res;
        }
    } 
    else
    {
        GnutellaApp::bootstrap();
    }
}

//=============================================================================
/* Ultrapeer availability for different types of peer connections */
bool Ultrapeer::avail(EPeerType aType) const
{
    return (aType == EPeerLegacy && nlegacy_ < max_legacy_) || 
           (aType == EPeerUltra && npeers_ < max_peer_)     ||
           (aType == EPeerLeaf && nleaves_ < max_leaves_);
}
//=============================================================================
void 
Ultrapeer::setState(PeerState_t newstate)
{
    if (newstate == PS_OFFLINE || newstate == PS_OUTSYS)
    {
        nlegacy_ = 0;
        npeers_  = 0;
        nleaves_ = 0;
    }

    GnutellaApp::setState(newstate);
}
//=============================================================================
static class UltrapeerClass: public TclClass {
public:
  UltrapeerClass(): TclClass("PeerApp/GnutellaApp/Ultrapeer") {}
  TclObject* create(int argc, const char*const* argv) {
    return (new Ultrapeer(atoi(argv[4]), argv[5], atoi(argv[6])));
  }
}class_ultrapeer;
//=============================================================================

