/*-------------------------------------------------------------------------*/
/* Packet-level peer-to-peer and Gnutella simulator                        */
/* Author: Qi He <http://www.cc.gatech.edu/~qhe> 01Aug2003                 */
/* $Revision:$ $Name:$ $Date:$                                             */
/*-------------------------------------------------------------------------*/
// Modified by Andre Dufour

#ifndef __GNUT_TYPES_H__
#define __GNUT_TYPES_H__

#include <list>
#include <map>

// Some useful ifdefs
#ifdef GTNETS
typedef IPAddr_t NodeAddr_t;
typedef PortId_t Port_t;
typedef PacketData Packet;
typedef TCP Socket;
#else
#include <stdarg.h>
#include "agent.h"
#include "node.h"
#include "packet.h"
#include "scheduler.h"
#include "tclcl.h"
#include "timer-handler.h"
#include "nssocket.h"

typedef unsigned short Word_t;
typedef unsigned char Byte_t;
#endif

/* gnutella protocol related macros */
#define LISTEN_PORT 6346 //default Gnutella listening port
// #define PING_INTERVAL  120//PING INTERVAL (seconds)
// #define CONN_INTERVAL 10 //Connection Watchdog Interval (seconds)
// #define CONN_INTERVAL 60 //Connection Watchdog Interval (seconds)
#define INIT_TTL 7  //default TTL
#define KNOWNCACHE_LIMIT 60 //known servents cache size limit
#define CONN_TIMEOUT 60 //connection request timeout
#define MAX_DEGREE 3 //maximum number of connections
#define BK_THRESH 60
#define DESC_TIMEOUT 300 //5 mins per 
#define MAX_DESC 1000 //max descriptor held by a peer
#define SMP_INTERVAL 1000 //report available member every 200 seconds
#define RATE_LIMIT 10 //incoming rate limit

/*boostrap related*/
//#define MAX_BOOTREPLY 20 //the maximum number of peers in a bootstrap reply 
#define MAX_BOOTREPLY 1 //the maximum number of peers in a bootstrap reply 
#define PUBLISH_PROB 100 //probabiltiy a node publishes its presence
#define BOOTSERVER 1 //nodeID of the default bootserver

#define QUERY_TIMEOUT 120 //outstanding query timeout in the UMASS model
#define HIT_PROB 0.2 //a random hit probability for query
#define QRP_HITPROB 0.2 //probability forwarding a query to a leaf

#define GNUTELLA_BOOTSTRAP "LIME BOOTSTRAP"
#define GNUTELLA_BOOTSTRAP_RES "LIME BOOTSTRAP RES"
#define GNUTELLA_CONN "GNUTELLA CONNECT/0.4\n\n"
#define GNUTELLA_OK "GNUTELLA OK\n\n"
#define GNUTELLA_REJ "GNUTELLA REJ\n\n"
#define ULTRA_CONN "GNUTELLA CONNECT/0.6\nX-Ultrapeer:True"
#define LEAF_CONN "GNUTELLA CONNECT/0.6\nX-Ultrapeer:False"
#define ULTRA_OK "GNUTELLA/0.6 200 OK\nX-Ultrapeer:True"
#define BOOTCACHE_UPDATE "BOOTSTRAP UPDATE"

/*lengths of the corresponding null-terminated string */
/*notice that the length includes the null terminator */
#define BOOTSTRAP_LEN 15
#define BOOTSTRAP_RESLEN 19 
#define CONNREJ_LEN 15
#define CONN_LEN 23
#define CONNOK_LEN 14
#define ULTRACONN_LEN 38
#define LEAFCONN_LEN 39
#define ULTRAOK_LEN 37
#define CACHEUPDATE_LEN 17

/* some fixed lengths of message */
#define DESC_HDRLEN 23 //descriptor header
#define PONG_LEN 14 //PONG
#define QUERYHIT_FIXLEN 11 //fixed part of a query

/* message ID returned by ParseMsg()*/
#define MSG_UNKNOWN -1
#define MSG_ILLFORMAT 0
#define MSG_CONNREQ 1
#define MSG_CONNOK 2
#define MSG_BOOTSTRAP 3
#define MSG_BOOTSTRAP_RES 4
#define MSG_CONNREJ 5

/* Ultrapeer-Leaf arch related */
#define MSG_ULTRACONNREQ 6
#define MSG_LEAFCONNREQ 7
/* prioritized messages */
#define MSG_QUERYHIT  8
#define MSG_QUERY 9
#define MSG_PONG 10
#define MSG_PING 11
#define MSG_BOOTCACHE_UPDATE 12
#define MSG_PUSH 13

#define REQ_PING 0
#define REQ_QUERY 1

/* Gnutella Msg type */
#define PING 0x00
#define PONG 0x01
#define QUERY 0x80
#define QUERYHIT 0x81
#define PUSH 0x40

enum EPeerType
{
    EPeerLegacy = 0,
    EPeerLeaf,
    EPeerUltra,
    EPeerAny
};

#define SOCK_LEGACY 0
#define SOCK_LEAF 1
#define SOCK_ULTRA 2
#define SOCK_BOOT 3
#define SOCK_BOOT_WAIT 4

#define DESCRIPTOR_SIZE 16

typedef enum {NOT_CONNECTED, TRANSPORT_CONNECTED, GNUTELLA_CONNECTED} ConnState_t; //pending conn req state

typedef enum {PS_OFFLINE, PS_ACTIVE, PS_IDLE, PS_OUTSYS} PeerState_t; //peer state in the UMASS model

typedef int (*Upcall_Proto) (va_list); //upcalls from PeerAgent to PeerApp
typedef char DescriptorID_t [DESCRIPTOR_SIZE];//Gnutella Descriptor ID
typedef char ServentID_t [DESCRIPTOR_SIZE]; //Gnutella Servent ID

/* Structures to represent Gnutella msgs */
typedef struct
{
    DescriptorID_t id_;
    Byte_t payload_desc_;
    Byte_t TTL_;
    Byte_t hops_;
    int length_;
} desc_hdr; //Descriptor header of a Gnutella Message

typedef struct
{
    Word_t port_; 
    NodeAddr_t addr_;
    int file_shared_;
    int byte_shared_;
} tPong;  //Pong
  
typedef struct
{
    Word_t minSpeed_;
    char *criteria_;
} tQuery; //Query

typedef struct
{
    Byte_t num_hits_;
    Word_t port_;
    NodeAddr_t addr_;
    int speed_;
} tQueryHit; //QueryHit

typedef struct
{
  ServentID_t srvid_;
  int idx_;
  NodeAddr_t addr_;
  Word_t port_;
} tPush; //Push	 

class PeerApp;
class PeerAgent;
class GnutellaApp;
class GnutellaAgent;
class GnutellaMsg;
class PendingConnEntry;
class PendingReqEntry;
class PeerEntry;
class DescEntry;
class SockEntry;
class ConnTimer;
class PingTimer;
class WatchDog;
class ServentRec;
class PeerSys;
class ActivityController;
class SmpBootServer;
class GServentRec;
class GC;
class BootServerRec;

#ifdef PDNS
NsObject* GetLocalIP(ipaddr_t ipaddr);
#endif

typedef std::map<NodeAddr_t, PendingConnEntry> PendingConns_t;
typedef std::list<DescEntry> DescMap_t;
typedef std::map<NodeAddr_t, SockEntry> SockMap_t;
typedef std::list<ServentRec> ServentMap_t;
typedef std::map<NodeAddr_t, PeerEntry> PeerMap_t;
typedef std::list<PendingReqEntry> ReqList_t;
typedef std::map<NodeAddr_t, GServentRec> GServentMap_t;
typedef std::list<BootServerRec> BServerList_t;

/**********************************************************************/
/****                Part I: GnutellaAgent related                     ****/

struct PendingConnEntry
{
  PendingConnEntry(NodeAddr_t aPeer, double aTimeout,
                   double aTstamp, ConnState_t aState)
  : peer_(aPeer),
    timeout_(aTimeout),
    start_(aTstamp),
    state_(aState)
  { }

  NodeAddr_t peer_;
  double timeout_;
  double start_;
  ConnState_t state_;
};

/* Descriptor Cache */
struct DescEntry 
{
  DescEntry(char *id, Socket *incoming, double tstamp)
  : sock_(incoming),
    tstamp_(tstamp)
  {
    memcpy((char *)&id_, id, DESCRIPTOR_SIZE);
  }

  DescriptorID_t id_;
  Socket *sock_;
  double tstamp_;
};

/* Sockets maintained by a GnutellaAgent */
struct SockEntry 
{
  SockEntry(Socket *sock, NodeAddr_t peer, Byte_t type) 
  : sock_(sock),
    peer_(peer),
    type_(type)
  { }

  Socket *sock_;
  NodeAddr_t peer_;
  Byte_t type_;
};

/* bootstrap server record */
struct BootServerRec 
{
  BootServerRec(NodeAddr_t addr, Word_t port) 
  : addr_(addr),
    port_(port)
  { }

  BootServerRec(NodeAddr_t addr) 
  : addr_(addr),
    port_(LISTEN_PORT)
  { }

  NodeAddr_t addr_;
  Word_t port_;
};

/* Servent cache */
struct ServentRec 
{
  ServentRec(Byte_t port, NodeAddr_t addr, int nbytes, int nfiles) 
  : port_(port),
    addr_(addr),
    file_shared_(nfiles),
    byte_shared_(nbytes)
  { }

  Word_t port_; 
  NodeAddr_t addr_;
  int file_shared_;
  int byte_shared_;
};

// TODO: UGLY!!
extern int nquery, nqueryhit, nthru, rquery, squeryhit;
extern long pongcnt, nforceremove; 
extern int Na1, Na2;
extern double Avg_Na1, Avg_Na2, Last_Chg1, Last_Chg2, TSysTime1, TSysTime2; 
extern double pAvg_Na1, pAvg_Na2, pTime1, pTime2, Avg_Rsp;

struct ClassSpec_t
{
  int count_;
  double loff_;
  double lidle_;
  double poff_;
  int isFreeloader_;
};

typedef std::map<int, ClassSpec_t> ClassMap_t;

struct BootstrapRes_t
{
  int cnt_;
  NodeAddr_t *servents_;
};



#endif /* __GNUT_TYPES_H__ */

