// File: gnutellamsg.cc
// Original by Qi He
// Modified by Andre Dufour


#include "gnutellamsg.h"
#include "gnut_types.h"

/* Gnutella Message Format Handler */
GnutellaMsg::GnutellaMsg(NodeAddr_t addr) {

  bzero((void *)&header_, sizeof(header_));
  payload_.pong_ = NULL;
  payload_.query_ = NULL;
  payload_.queryhit_ = NULL;
  payload_.push_ = NULL;
  addr_ = addr;

  seq1_ = seq2_ = seq3_ = 0;
  
}

GnutellaMsg::~GnutellaMsg(void)
{ }

/* clear after every use of GnutellaMsg::parse() */
int GnutellaMsg::clear() {

  if(payload_.pong_) {
    free(payload_.pong_);
    payload_.pong_ = NULL;
  }

  if(payload_.query_) {
    if(payload_.query_->criteria_)
      free(payload_.query_->criteria_);
    payload_.query_->criteria_ = NULL;

    free(payload_.query_);
    payload_.query_ = NULL;
  }

  if(payload_.queryhit_) {
    free(payload_.queryhit_);
    payload_.query_ = NULL;
  }

  if(payload_.push_){
    free(payload_.push_);
    payload_.push_ = NULL;
  }

  bzero((void *)&header_, sizeof(header_));
}

/* Gnutella Message Parsing */
int GnutellaMsg::parse(int len, unsigned char *data) {
  char *teststring = (char *)malloc(len+1);
  void *temp=NULL;

  memcpy((char *)teststring, (char *)data, len);
  if(strcmp(GNUTELLA_BOOTSTRAP_RES, (char *)teststring)==0) {
    free(teststring);
    return MSG_BOOTSTRAP_RES;
  }

  teststring[len] = 0;
  if(strcmp(GNUTELLA_BOOTSTRAP, (char *)teststring)==0) {
    free(teststring);
    return MSG_BOOTSTRAP;
  }

  if(strcmp(GNUTELLA_CONN, (char *)teststring)==0) {
    free(teststring);
    return MSG_CONNREQ;
  }

  if(strcmp(ULTRA_CONN, (char *)teststring)==0) {
    free(teststring);
    return MSG_ULTRACONNREQ;
  }

  if(strcmp(LEAF_CONN, (char *)teststring)==0) {
    free(teststring);
    return MSG_LEAFCONNREQ;
  }

  if(strcmp(GNUTELLA_OK, (char *)teststring)==0) {
    free(teststring);
    return MSG_CONNOK;
  }

  if(strcmp(ULTRA_OK, (char *)teststring)==0) {
    free(teststring);
    return MSG_CONNOK;
  }

  if(strcmp(GNUTELLA_REJ, (char *)teststring)==0) {
    free(teststring);
    return MSG_CONNREJ;
  }

  if(strcmp(BOOTCACHE_UPDATE, (char *)teststring)==0) {
    free(teststring);
    return MSG_BOOTCACHE_UPDATE;
  }

  free(teststring);

  if(len < DESC_HDRLEN) 
    return MSG_UNKNOWN;

  memcpy((void *)&header_, (void *)data, DESC_HDRLEN);
  data[18] = header_.hops_ + 1;
  switch(header_.payload_desc_) {
  case PING:
    return MSG_PING;

  case PONG:
    if(header_.length_ != PONG_LEN)
      return MSG_ILLFORMAT;

    temp = malloc(header_.length_);
    memcpy(temp, (void *)&data[DESC_HDRLEN], PONG_LEN);
    payload_.pong_ = (tPong *)temp;
    return MSG_PONG;

  case QUERY:
    if(header_.length_< 1) 
      return MSG_ILLFORMAT;
    temp = malloc(sizeof(tQuery));
    memcpy(temp, (void *)&data[DESC_HDRLEN], sizeof(Word_t));
    payload_.query_ = (tQuery *)temp;
    payload_.query_->criteria_ = (char *)malloc(header_.length_ - sizeof(Word_t));
    strcpy(payload_.query_->criteria_, (char *)&data[DESC_HDRLEN + sizeof(Word_t)]);
    return MSG_QUERY;
    
  case QUERYHIT:
    if(header_.length_ < QUERYHIT_FIXLEN)
      return MSG_ILLFORMAT;
    temp = malloc(sizeof(tQueryHit));
    memcpy((void *)temp, (void *)&data[DESC_HDRLEN], header_.length_);
    payload_.queryhit_ = (tQueryHit *)temp;
    return MSG_QUERYHIT;

  default:
    return MSG_UNKNOWN;
  };

  return MSG_UNKNOWN;
}

/* compose a PacketData with a Gnutella message as the payload         */
/* called with message type and length, as well as some optional parms */
PacketData *GnutellaMsg::newpacket(char *id, Byte_t type, int ttl, int hops, int len, char *data) {
  
  PacketData *userdata = new PacketData(DESC_HDRLEN + len);
  unsigned char *dataptr = userdata->data();

  if(dataptr == NULL)
    return NULL;
  header_.payload_desc_ = type;
  header_.TTL_ = ttl;
  header_.hops_ = hops;
  header_.length_ = len;
  if(id)
    memcpy((void *)&header_.id_, id, 16);
  else {
    id = new_descid();
    memcpy((void *)&header_.id_, id, 16);
    free(id);
  }
  
  memcpy((void *)dataptr, (void *)&header_, DESC_HDRLEN);
  if(data && len>0)
    memcpy((void *)&dataptr[DESC_HDRLEN], data, len);

  return userdata;
}

/* generate a new unique descriptor ID */
char *GnutellaMsg::new_descid() {
  char *newid = NULL;
  
  newid = (char*)malloc(16);
  memcpy((void *)newid, &addr_, 4);
  seq1_ ++;
  memcpy((void *)&newid[4], &seq1_, 4);
  memcpy((void *)&newid[8], &seq2_, 4);
  memcpy((void *)&newid[12], &seq3_, 4);
  if(seq1_ == INT_MAX -1) {
    seq1_ = 0;
    seq2_ ++; 
    if(seq2_ == INT_MAX -1) {
      seq2_ = 0;
      seq3_ ++;
    }
  }

  return newid;
}


