// File: peersys.cc
// Original by Qi He
// Modified by Andre Dufour

#include "peersys.h"
#include "gnut_types.h"

//=============================================================================
/* UMASS P2P system model */
PeerSys::PeerSys() {
  bind("nFiles_", &nFiles_);
  bind("nClasses_", &nClasses_);
  bind("alpha_", &alpha_);
  
  cdffile_ = NULL;
  init_file();
}

//=============================================================================
/* initialize the file popularity CDF structure */
void PeerSys::init_file() {
  int i;
  double K=0.0;

  cdffile_ = (double *)malloc(sizeof(double)*nFiles_);
  for(i=0; i< nFiles_; i++) {
    K = K + 1.0/pow(i+1, alpha_);
  }
  K = 1.0/K;

  for(i=0; i<nFiles_; i++) {
    if(i==0)
      cdffile_[i] = K/pow(i+1, alpha_);
    else
      cdffile_[i] = K/pow(i+1, alpha_) + cdffile_[i-1];
  }
}

//=============================================================================
/* initialize the class info based on a configuration file */
/* configuration file default: classinfo.txt               */
void PeerSys::init_class(FILE *fh) {
  char line[100];
  
  for(int i=0; i< nClasses_; i++) {
    ClassSpec_t *ncl =  (ClassSpec_t *)malloc(sizeof(ClassSpec_t));
    if(fgets((char *)&line, 100, fh)!=NULL) {
      sscanf(line, "%lf\t%lf\t%lf\t%d\n", &ncl->loff_, &ncl->lidle_, &ncl->poff_, &ncl->isFreeloader_);
      classes_.insert(ClassMap_t::value_type(i, *ncl));
    }
  }
}

//=============================================================================
int PeerSys::command(int argc, const char*const* argv) {
  FILE *fh;

  if(strcmp(argv[1], "init-class")==0) {
    if(argc==2) 
      fh = fopen("classinfo.txt", "r");
    else
      fh = fopen(argv[2], "r");
      
    if(fh==NULL) {
      fprintf(stderr, "PeerSys::init-class: unable to open class info file\n");
      return TCL_ERROR;
    }
      
    init_class(fh);
    fclose(fh);
    return TCL_OK;
  }
  return TclObject::command(argc, argv);
}

/* Peer state:
   OUTSYS->INSYS->OUTSYS
   while INSYS, a peer could be OFFLINE, ONLINE and IDLE
*/

//=============================================================================
static class PeerSysClass: public TclClass {
public:
  PeerSysClass(): TclClass("PeerSys") {}
  TclObject* create(int, const char*const* argv) {
    return (new PeerSys());
  }
}class_peersys;
//=============================================================================


