//=============================================================================
// File: gnutcoord.cc
// Author: Andre Dufour
//=============================================================================

#include "gnutcoord.h"
#include "gnut_util.h"
#include <cstdlib>
#include <cmath>

const double GnutCoord::INITIAL_HEIGHT = 1.0;

//=============================================================================
GnutCoord::GnutCoord(void)
: mX(0.0),
  mY(0.0),
  mHeight(INITIAL_HEIGHT)
{ }
//=============================================================================
GnutCoord::GnutCoord(double aX, double aY, double aHeight)
: mX(aX),
  mY(aY),
  mHeight(aHeight)
{ }
//=============================================================================
GnutCoord::GnutCoord(double aLength)
{
    double x      = rand();
    double y      = rand();
    double height = rand();

//    double length = sqrt( pow(x,2) + pow(y,2) ) + fabs(height);
    double length = sqrt( x * x + y * y + height * height );
    double scale  = aLength / length;

    mX      = x * scale;
    mY      = y * scale;
    mHeight = height * scale;
}
//=============================================================================
GnutCoord::GnutCoord(double aLength, const GnutCoord& aDirection)
{
    double x, y, height;

    if (aDirection.abs() != 2.0 && aDirection.abs() > 0.1)
    {
        x      = aDirection.mX;
        y      = aDirection.mY;
        height = aDirection.mHeight;
    }
    else
    {
        // If the direction is (0,0,0), we pick an arbitrary direction.
        // This is like GnutCoord(double aLength)
        x      = rand();
        y      = rand();
        height = rand();
    }

//    double length = sqrt( pow(x,2) + pow(y,2) ) + fabs(height);
    double length = sqrt( x * x + y * y + height * height );
    double scale  = aLength / length;

    mX      = x * scale;
    mY      = y * scale;
    mHeight = height * scale;
}
//=============================================================================
GnutCoord::~GnutCoord(void)
{ }
//=============================================================================
double
GnutCoord::abs(void) const
{
//    return sqrt( pow(mX,2) + pow(mY,2) ) + fabs(mHeight);
    return sqrt( mX * mX + mY * mY + mHeight * mHeight);
}
//=============================================================================
double
GnutCoord::funkyAbs(void) const
{
    return sqrt( mX * mX + mY * mY ) + fabs(mHeight);
}

//=============================================================================
GnutCoord& 
GnutCoord::operator+=(const GnutCoord& aRhs)
{
    this->mX      += aRhs.mX;
    this->mY      += aRhs.mY;
    this->mHeight += aRhs.mHeight;

    if (this->mHeight < 0.0)
        this->mHeight = INITIAL_HEIGHT;

    return *this;
}
//=============================================================================
GnutCoord& 
GnutCoord::operator-=(const GnutCoord& aRhs)
{
    this->mX      -= aRhs.mX;
    this->mY      -= aRhs.mY;
    this->mHeight += aRhs.mHeight;

    return *this;
}
//=============================================================================
GnutCoord&
GnutCoord::operator*=(double aFactor)
{
    mX      *= aFactor;
    mY      *= aFactor;
    mHeight *= aFactor;

    return *this;
}
//=============================================================================
GnutCoord 
GnutCoord::
towards(const GnutCoord& aCoord) const
{
    return GnutCoord(aCoord.mX - this->mX,
                     aCoord.mY - this->mY,
                     -1 * (this->mHeight + aCoord.mHeight));
}
//=============================================================================
GnutCoord 
GnutCoord::
awayFrom(const GnutCoord& aCoord) const
{
//    return GnutCoord(this->mX - aCoord.mX,
//                     this->mY - aCoord.mY,
//                     (aCoord.mHeight + this->mHeight));
    return GnutCoord(this->mX - aCoord.mX,
                     this->mY - aCoord.mY,
                     aCoord.mHeight);
}

//=============================================================================
// Non class-member operators
//=============================================================================
GnutCoord operator+(const GnutCoord& aLhs, const GnutCoord& aRhs)
{
    if (aLhs.mHeight + aRhs.mHeight > 0.0)
    {   
        return GnutCoord(aLhs.mX + aRhs.mX,
                         aLhs.mY + aRhs.mY,
                         aLhs.mHeight + aRhs.mHeight);
    }
    else
    {
        return GnutCoord(aLhs.mX + aRhs.mX,
                         aLhs.mY + aRhs.mY,
                         GnutCoord::INITIAL_HEIGHT);
    }
        
}
//=============================================================================
GnutCoord operator-(const GnutCoord& aLhs, const GnutCoord& aRhs)
{
    return GnutCoord(aLhs.mX - aRhs.mX,
                     aLhs.mY - aRhs.mY,
                     aLhs.mHeight + aRhs.mHeight);
}
//=============================================================================
GnutCoord operator*(const GnutCoord& aCoord, double aFactor)
{
    return GnutCoord(aCoord.mX * aFactor, 
                     aCoord.mY * aFactor,  
                     aCoord.mHeight * aFactor);
}
//=============================================================================
GnutCoord operator*(double aFactor, const GnutCoord& aCoord)
{
    return GnutCoord(aCoord.mX * aFactor, 
                     aCoord.mY * aFactor,  
                     aCoord.mHeight * aFactor);
}
//=============================================================================
bool operator==(const GnutCoord& aLhs, const GnutCoord& aRhs)
{
    return aLhs.mX == aRhs.mX &&
           aLhs.mY == aRhs.mY &&
           aLhs.mHeight == aRhs.mHeight;
}
//=============================================================================
bool operator!=(const GnutCoord& aLhs, const GnutCoord& aRhs)
{
    return aLhs.mX != aRhs.mX ||
           aLhs.mY != aRhs.mY ||
           aLhs.mHeight != aRhs.mHeight;
}
//=============================================================================
std::ostream& operator<<(std::ostream& aOs, const GnutCoord& aCoord) 
{
    return aOs << "(" << aCoord.mX << ", " << aCoord.mY << ", " 
               << aCoord.mHeight << ")";
}
//=============================================================================

