// File: gnutellaapp.cc
// Original by Qi He
// Modified by Andre Dufour

#include "gnutellaapp.h"
#include "gnut_types.h"
#include "gnut_util.h"
#include "pingtimer.h"
#include "watchdog.h"
#include "activitycontroller.h"
#include "gnutellaagent.h"
#include "smpbootserver.h"
#include "gnut_util.h"
#include "peersys.h"
#include "gc.h"
#include "gnutpongmsg.h"
#include "gnutbootstrapresmsg.h"
#include "gnuthierapp.h"
#include "gnutstats.h"
#include "gnutqueryhitmsg.h"
#include "gnutquerymsg.h"
#include "vivaldimanagerbc.h"
#include "vivaldimanager.h"
#include "gnutcoord.h"

//=============================================================================
const double GnutellaApp::GNUT_MIN_CONNECTION_TIME = 20.0; // Seconds
//const double GnutellaApp::GNUT_MIN_CONNECTION_TIME = 0.0; // Seconds
//=============================================================================
GnutellaApp::
GnutellaApp(NodeAddr_t aAddr, const char* const aParent, int aUsesVivaldi,
            double aMinDisconnectTime)
: PeerApp(aAddr),
  isBootserver_(FALSE),
  max_deg_(MAX_DEGREE),
  ping_timer_(pingListener_),
  watchDog_(watchdogListener_),
  gagent_(0),
  degree_(aAddr),
  smpBoot_(FALSE),
  bootSrv_(0),
  watchdogListener_(*this, &GnutellaApp::watchdogExpired),
  pingListener_(*this, &GnutellaApp::pingExpired),
  parent_(*(reinterpret_cast<GnutHierApp*>(TclObject::lookup(aParent)))),
  mVivaldi(aUsesVivaldi ? new VivaldiManager() : new VivaldiManagerBC()),
  hasContent_(false),
  mGnutStats(GnutStats::instance()),
  mMinDisconnectTime(aMinDisconnectTime)
{
  bind("isBootserver_", &isBootserver_);
  bind("max_deg_", &max_deg_);
  
  GnutUtil::gnutError("Creating gnutellaapp %p for node %d. MinDisconnect at %f\n", this, aAddr, mMinDisconnectTime);
}
//=============================================================================
GnutellaApp::
~GnutellaApp(void)
{
    (void)cleanup();
    delete gagent_;
    delete mVivaldi;
    GnutUtil::gnutTrace("Destroying gnutellaapp %d (%p) (in C++)\n", 
                        addr_, this);
}
//=============================================================================
/* user command interface */
/* join: outsys->insys */
void GnutellaApp::join() {
  GnutUtil::gnutTrace("%d told to join\n", addr_);
  
  watchDog_.resched();
  ping_timer_.resched();

  bootstrap();
}
//=============================================================================
int
GnutellaApp::
cleanup()
{
    setState(PS_OFFLINE); // Required for integrity of system.

    Tcl::instance().evalf("%s cleanup_app", name());

    return TCL_OK;
}
//=============================================================================
/* leave: insys->outsys */
void GnutellaApp::leave() {
  if (state_!= PS_OUTSYS)
	setState(PS_OFFLINE);
  if (ping_timer_.status()==TIMER_PENDING)
    ping_timer_.cancel();

  if (watchDog_.status()==TIMER_PENDING)
  {
    watchDog_.cancel();
  }

  if (ac_ && ac_->status_==TIMER_PENDING)
  {
    ac_->cancel();
  }

  mVivaldi->resetCoordinates();
}

//=============================================================================
void GnutellaApp::stop() {
  if (state_ != PS_OUTSYS)
	setState(PS_OFFLINE); 

  if (ping_timer_.status()==TIMER_PENDING)
    ping_timer_.cancel();

  if (watchDog_.status()==TIMER_PENDING)
  {
    watchDog_.cancel();
  }

  if (ac_ && ac_->status_==TIMER_PENDING)
    ac_->cancel();
}

//=============================================================================
void GnutellaApp::disconnect(NodeAddr_t node) 
{
    GnutUtil::gnutTrace("GnutellaApp::disconnect(%d)\n", node);

    gagent_->Disconnect(node);
    if (node == GnutellaAgent::ALL_NEIGHBOURS) 
    {
        GnutUtil::gnutError("Node %d disconnecting all nodes\n",
                            addr_);
        lpeers_.clear();
        mPeerDistances.clear();
        if (smpBoot_) 
        {
            if (bootSrv_)
            {
                bootSrv_->RemovePeer(addr_, this);
            }
        }
    }
    else 
    {
        PeerMap_t::iterator pi = lpeers_.find(node);
        GNUT_ASSERT(pi != lpeers_.end());
        if (pi!=lpeers_.end()) 
        {
            GnutUtil::gnutError("Node %d disconnecting from node %d\n",
                                addr_, node);
            lpeers_.erase(pi);

            PeerDistances::iterator diter = mPeerDistances.find(node);
            GNUT_ASSERT(diter != mPeerDistances.end());
            mPeerDistances.erase(diter);
        }

        if (watchDog_.status() == TIMER_IDLE)
        {
            // If the watchdog had been turned off because we had all the 
            // connections we needed, turn it back on.
            // TODO: AD
            GnutUtil::gnutTrace("%d can handle more connections. Re-enabling "
                                "watchdog\n", addr_);
            watchDog_.resched();
        }
    }
    mGnutStats.setPeerConns(addr_, lpeers_.size());
    degree_.increment(lpeers_.size());
}

//=============================================================================
/* share files */
void GnutellaApp::share() {
}

//=============================================================================
/* maintenance of peer relationship */
void GnutellaApp::maintenance() {
}

//=============================================================================
/* initiate a query */
void GnutellaApp::search() 
{
    if (lpeers_.empty() == false)
    {
        gagent_->Query(GnutellaAgent::ALL_NEIGHBOURS); 
    }
}
//=============================================================================
/* GnutellaApp internal operations */
/* proactively change its own status */
void GnutellaApp::setState(PeerState_t state) 
{
    GnutUtil::gnutError("Setting state in %d to %d at time %f\n",
                        addr_, (int)state, GnutUtil::now());
    PeerState_t old_state = state_;
    double tmp, ctime = NOW;

    state_ = state;

    if (state == PS_OFFLINE || state == PS_OUTSYS)
    {
        mGnutStats.setPeerOnline(addr_, false);
    }
    else
    {
        mGnutStats.setPeerOnline(addr_, true);
    }

    // TODO: AD
    char* statestr = 0;
    switch (state)
    {
    case PS_OFFLINE:
      statestr = "offline";
      break;
    case PS_ACTIVE:
      statestr = "active";
      break;
    case PS_IDLE:
      statestr = "idle";
      break;
    case PS_OUTSYS:
      statestr = "out_sys";
      break;
    default:
      statestr = "???";
      GNUT_ASSERT(false);
      break;
    }

    GnutUtil::gnutTrace("%d going %s\n", addr_, statestr);
      

    //the following code is for membership accounting purpose
    if (state_ == PS_OFFLINE && old_state != PS_OFFLINE) 
    {
        if (!isFreeloader_) 
        {
            if (Last_Chg1 !=-1) 
            {
                tmp = (ctime - Last_Chg1) * (double)Na1 + Avg_Na1 * TSysTime1;
                TSysTime1 += (ctime - Last_Chg1);
                Avg_Na1 = tmp/TSysTime1;

                /* periodic averaging */
                tmp = (ctime - Last_Chg1) * (double)Na1 + pAvg_Na1 * pTime1;
                pTime1 += (ctime - Last_Chg1);
                pAvg_Na1 = tmp/pTime1;

                Na1--;
            }
            else 
            {
                debug_warning("WARNING: shouldn't get here (setState:1)\n");
            }

            Last_Chg1 = ctime;

        } 
        else 
        {
            if (Last_Chg2 !=-1) 
            {
                tmp = (ctime - Last_Chg2) * (double)Na2 + Avg_Na2 * TSysTime2;
                TSysTime2 += (ctime - Last_Chg2);
                Avg_Na2 = tmp/TSysTime2;

                /* periodic averaging */
                tmp = (ctime - Last_Chg2) * (double)Na2 + pAvg_Na1 * pTime2;
                pTime2 += (ctime - Last_Chg2);
                pAvg_Na2 = tmp/pTime2;

                Na2 --;
            }
            else 
            {
                debug_warning("WARNING: shouldn't get here (setState:2)\n");
            }  

            Last_Chg2 = ctime;
        }

        if (ping_timer_.status()==TIMER_PENDING)
            ping_timer_.cancel();

        if (watchDog_.status()==TIMER_PENDING)
        {
            watchDog_.cancel();
        }

        //disconnect() is required for correct functioning though
        disconnect(GnutellaAgent::ALL_NEIGHBOURS);

        mVivaldi->resetCoordinates();

        GnutUtil::gnutTrace("Membership %d\t%d\n",  Na1, Na2);
        gagent_->bkblock_.offline();
        gagent_->fwblock_.offline();
        gagent_->rcvRate_.offline();
        degree_.offline();

        fflush(NULL);
    }
    else if (state_ != PS_OFFLINE && old_state == PS_OFFLINE) 
    {
        if (!isFreeloader_) 
        {
            if (Last_Chg1 !=-1) 
            {
                tmp = (ctime - Last_Chg1 ) * (double)Na1 + Avg_Na1 * TSysTime1;
                TSysTime1 += (ctime - Last_Chg1);
                Avg_Na1 = tmp/TSysTime1;

                tmp = (ctime - Last_Chg1) * (double)Na1 + pAvg_Na1 * pTime1;
                pTime1 += (ctime - Last_Chg1);
                pAvg_Na1 = tmp/pTime1;

            }
            Last_Chg1 = ctime; 
        }
        else 
        {
            if (Last_Chg2 !=-1) 
            {
                tmp = (ctime - Last_Chg2 ) * (double)Na2 + Avg_Na2 * TSysTime2;
                TSysTime2 += (ctime - Last_Chg2);
                Avg_Na2 = tmp/TSysTime2;

                tmp = (ctime - Last_Chg2) * (double)Na2 + pAvg_Na1 * pTime2;
                pTime2 += (ctime - Last_Chg2);
                pAvg_Na2 = tmp/pTime2;

            } 

            Last_Chg2 = ctime; 
        }

        if (!isFreeloader_)
            Na1 ++;
        else
            Na2 ++;

        ping_timer_.resched();
        watchDog_.resched();

        GnutUtil::gnutTrace("Membership %d\t%d\n",  Na1, Na2);
        gagent_->bkblock_.online();
        gagent_->fwblock_.online();
        gagent_->rcvRate_.online();
        degree_.online();
        fflush(NULL);
    }
    
    fflush(NULL);

    if (pTime1 > SMP_INTERVAL) 
    {
        debug_info("Membership non-freeloader %f\t%d\t%d\n", 
                   pAvg_Na1, Na1, Na2);
        pAvg_Na1 = 0;
        pTime1 = 0;
    }

    if (pTime2 > SMP_INTERVAL) 
    {
        debug_info("Membership freeloader %f\t%d\t%d\n", pAvg_Na2, Na1, Na2);
        pAvg_Na2 = 0;
        pTime2 = 0;
    }
}

//=============================================================================
/* bootstrap */
void GnutellaApp::bootstrap() 
{
    if (smpBoot_) 
    {
        if (bootSrv_)
        {
            smp_bootstrap();
        }
    }
    else if (bserver_list_.size() > 0) 
    {
        BootServerRec bi = bserver_list_.front();
        gagent_->Bootstrap(bi.addr_);
        bserver_list_.push_back(bi);
        bserver_list_.pop_front();
    }
}
//=============================================================================
/* use SmpBootServer for bootstrapping */
void GnutellaApp::smp_bootstrap() 
{
    BootstrapRes* res = bootSrv_->BootstrapRequest(addr_, this, EPeerLegacy);

    if (res != 0)
    {
        BootstrapResult(res->cnt_, res->servents_);
        delete res;
    }
}
//=============================================================================
/* tries to connect to some known servents */
void GnutellaApp::connect(bool aTryBootserver, bool* aReloadTimer /* = 0 */) 
{
    int cnt = 0; 

    //GNUT_ASSERT(gagent_->conn_pending_.size() <= max_deg_ / 2);

//    int limit = (max_deg_ - lpeers_.size() - gagent_->conn_pending_.size());

    int limit = 0;

    if ((lpeers_.size() + gagent_->conn_pending_.size()) < max_deg_)
    {
        limit = 1;
    }


    GnutUtil::gnutError("Peer %d Max deg: %d lpeers: %d pending %d limit: %d\n",
                        addr_,
                        max_deg_, 
                        lpeers_.size(), 
                        gagent_->conn_pending_.size(),
                        limit);

    GNUT_ASSERT(lpeers_.size() + gagent_->conn_pending_.size() <= max_deg_);

    if (limit < 0)
    {
        limit = 0;
    }

    for (ServentMap_t::const_iterator i = servent_cache_.begin(); 
         cnt < limit && i != servent_cache_.end(); ++i ) 
    {
        NodeAddr_t peer = i->addr_;

        if (isAssociated(peer) == false)
        {
            GnutUtil::gnutTrace("%d trying to connect to peer %d\n", 
                                addr_, peer);
            if (gagent_->Connect(peer, CONN_TIMEOUT))
            {
                // successfully connected
            	cnt++;
                fflush(NULL);
            }
        }
    }

    if (aTryBootserver && cnt < limit)
    {
        bootstrap();
    }
    if (aReloadTimer != 0)
    {
#if 0
        if (cnt == limit)
        {
            *aReloadTimer = false;
        }
        else
        {
            *aReloadTimer = true;
        }
#else
        *aReloadTimer = true;
#endif

    }
}
//=============================================================================
/* periodically send PING to connected peers */
void GnutellaApp::ping() {
  double ctime; 

  GnutUtil::gnutTrace("%d sending ping\n", addr_);

  ctime = NOW;
  for(PeerMap_t::iterator i = lpeers_.begin(); i != lpeers_.end(); ++i) {
    if (ctime - i->second.lstamp_ > PingTimer::PING_INTERVAL) {
      gagent_->Ping(i->second.peer_, INIT_TTL);
      i->second.lstamp_ = ctime;
    }
  }
}

//=============================================================================
// upcalls from GnutellaAgent
/* Connection request to a peer succeeded */
void GnutellaApp::ConnectSucceeded(
    NodeAddr_t peer, EGnutApp type, const GnutCoord& remoteCoord,
    double remoteError) 
{
    PeerMap_t::iterator pi = lpeers_.find(peer);
    if (pi==lpeers_.end())
    {
        if (lpeers_.size() >= max_deg_) 
        {
            // Between the time we requested this connection and the time
            // we got a reply, all of our connections filled up.
            // We must therefore disconnect the socket to advise the
            // other peer.
            GnutUtil::gnutError("%d must disconnect from %d because it "
                                "oversubscribed\n", addr_, peer);
            gagent_->Disconnect(peer);
            return;
        }

        GnutUtil::gnutError("Node %d succeeded in connecting to node %d\n",
                            addr_, peer);
        PeerEntry *newentry = new PeerEntry(peer, NOW, type);
        lpeers_.insert(PeerMap_t::value_type(peer, *newentry));
        mGnutStats.setPeerConns(addr_, lpeers_.size());

        // Add this new neighbour's distance info
        const double worst_case_distance = estimateDistance(remoteCoord);

//        const double worst_case_distance =   
//             distance * remoteError + distance * mVivaldi->getLocalError();
//

        GnutUtil::gnutTrace("%d recording distance to %d as %f\n",
                            addr_, peer, worst_case_distance);

        // Ugly mapping
        EPeerType peer_type = type == EGnutAppLeaf 
                              ?  EPeerLeaf
                              :  type == EGnutAppUltrapeer
                              ?  EPeerUltra
                              :  type == EGnutAppLegacy
                              ?  EPeerLegacy
                              :  EPeerAny;

        GNUT_ASSERT(peer_type != EPeerAny);
         
        SPeerDistance dist(peer_type, worst_case_distance, GnutUtil::now());
        mPeerDistances[peer] = dist;
                             

        
        degree_.increment(lpeers_.size());
        if (lpeers_.size()==1) 
        {
            if (gagent_->gc_ && gagent_->gc_->status()==TIMER_IDLE) 
                gagent_->gc_->resched(DESC_TIMEOUT/2);
        }
    }

    if (state_ == PS_OUTSYS)
    {
        GnutUtil::gnutTrace("%d connect succeeded to %d. %d going OUTSYS->IDLE\n",
               addr_, peer, addr_);
        // TODO: AD - check!
        state_ = PS_OFFLINE;
    }

    if (state_ == PS_OFFLINE) 
    {
        setState(PS_IDLE);


        // TODO: Check
        if (GnutUtil::now() > mMinDisconnectTime)
        {
            if (ac_)
                ac_->expire(NULL);
        }
    }
}
//
//=============================================================================
/* Bootstrap request received*/
void GnutellaApp::BootstrapRequest(NodeAddr_t peer) {

  debug_info("bootstrap request received from %d\n", peer);

  if (isBootserver_) {

    GnutUtil::gnutTrace("bootstrap request received from %d\n", peer);

    int i=0, size = servent_cache_.size();
    int cnt = MAX_BOOTREPLY>size ? size: MAX_BOOTREPLY;

    NodeAddr_t* addrs = new NodeAddr_t[cnt];
    for(ServentMap_t::iterator fi = servent_cache_.begin(); 
        fi != servent_cache_.end(); ++fi) {
      addrs[i] = fi->addr_;
      i++;
      if (i>=MAX_BOOTREPLY)
        break;
    }

    gagent_->Bootstrap_Reply(peer, cnt, addrs);
    if (find_servent(peer))
      remove_servent(peer);
    
    int prob = (int)((double)rand()/(double)RAND_MAX * 100.0);
     
    if (prob < PUBLISH_PROB) {
      ServentRec *newsrv;
      newsrv = new ServentRec(LISTEN_PORT, peer);
      servent_cache_.push_back(*newsrv);
    }

    while(servent_cache_.size() > KNOWNCACHE_LIMIT)
       servent_cache_.pop_front();

    delete[] addrs;
  }
}

//=============================================================================
/* Bootstrap results received */
void GnutellaApp::BootstrapResult(int cnt, NodeAddr_t *res) {
  int i;
  
  for(i=0; i<cnt; i++) {
    NodeAddr_t cur = res[i];

    if (!find_servent(cur) && cur != addr_) {
      ServentRec* newsrv = new ServentRec(LISTEN_PORT, cur);
      servent_cache_.push_back(*newsrv);
    }
  }

  connect(false);
}

//=============================================================================
/* Bootstrap results received */
void GnutellaApp::BootcacheUpdate(NodeAddr_t peer) {
   if (!isBootserver_)
	return;

   if (!find_servent(peer))
	return;

   remove_servent(peer);
   ServentRec *newsrv;
   newsrv = new ServentRec(LISTEN_PORT, peer);
   servent_cache_.push_back(*newsrv);
}

//=============================================================================
/* connection request received*/
int GnutellaApp::ConnectionRequest(NodeAddr_t peer, Socket *sock, const GnutCoord& remoteCoord, double remoteError, double) 
{
  GnutUtil::gnutTrace("%d recvd connection request from %d\n", addr_, peer);
  PeerMap_t::iterator i = lpeers_.find(peer);

  if ((avail(EPeerAny) && i==lpeers_.end()) || 
      removeOldConnIfCloser(peer, EPeerAny, remoteCoord, remoteError)) {
    
    const double worst_case_distance = estimateDistance(remoteCoord);
    SPeerDistance dist(EPeerAny, worst_case_distance, GnutUtil::now());
    mPeerDistances[peer] = dist;
    GnutUtil::gnutTrace("%d recording distance to %d as %f\n",
                        addr_, peer, worst_case_distance);


    GnutUtil::gnutError("Node %d ACCEPTING Connection request from node %d\n",
                        addr_, peer);
    gagent_->gnutella_ok(sock);

    PeerEntry *newentry = new PeerEntry(peer, NOW, EGnutAppLegacy);
    lpeers_.insert(PeerMap_t::value_type(peer, *newentry));
    mGnutStats.setPeerConns(addr_, lpeers_.size());
    degree_.increment(lpeers_.size());
    if (lpeers_.size()==1) {
      if (state_ == PS_OUTSYS)
      {
        state_ = PS_OFFLINE;
      }

      if (state_ == PS_OFFLINE) {
        setState(PS_IDLE);
        // TODO: check
        if (GnutUtil::now() > mMinDisconnectTime)
        {
            if (ac_)
              ac_->expire(NULL);
        }
      }
    }

    if (!find_servent(peer)) {
      ServentRec *srec = new ServentRec(LISTEN_PORT, peer);
      servent_cache_.push_back(*srec);
    }

    GnutUtil::gnutTrace("%d will accept conn req from %d\n", addr_, peer);
    return 1;
  } else {
    GnutUtil::gnutError("Node %d REJECTING connection request from node %d\n",
                        addr_, peer);
    gagent_->gnutella_reject(sock);
    return 0;
  }
}

//=============================================================================
bool GnutellaApp::removeOldConnIfCloser(
    NodeAddr_t aPeer, EPeerType aPeerType, const GnutCoord& aRemoteCoord, 
    double aRemoteError) 
{
//    const double worst_case_distance =   distance * aRemoteError 
//                                       + distance * mVivaldi->getLocalError();
//

    GNUT_ASSERT(lpeers_.size() == mPeerDistances.size());
    
    const double worst_case_distance = estimateDistance(aRemoteCoord);

    GnutUtil::gnutTrace("%d estimating distance from %d as %f\n",
                        addr_, aPeer, worst_case_distance);

    double worst_distance = 0.0;
    bool retval = false;

    NodeAddr_t neighbourToReplace = -1;

    const double currentTime = GnutUtil::now();

    for (PeerDistances::const_iterator iter = mPeerDistances.begin();
         iter != mPeerDistances.end(); iter++)
    {
        if (aPeerType == EPeerAny || aPeerType == iter->second.mPeerType)
        {
            if (currentTime - iter->second.mTimeFirstConnected >
                GNUT_MIN_CONNECTION_TIME)
            {
                if (worst_case_distance < iter->second.mDistance)
                {
                    if (iter->second.mDistance > worst_distance)
                    {
                        worst_distance = iter->second.mDistance;
                        neighbourToReplace = iter->first;
                    }
                    retval = true;
                }
            }
        }
    }

    // We do the disconnect here. Caller must check return value and do the
    // connect as desired.
    if (retval)
    {
        // Removes node from peer distance map
        disconnect(neighbourToReplace);
        mGnutStats.newConnDumpTime(GnutUtil::now());
    }

    return retval;
}
//=============================================================================
int GnutellaApp::UltraConnRequest(NodeAddr_t peer, Socket *socket, const GnutCoord&, double, double) {
  return FALSE;
}
//=============================================================================
int GnutellaApp::LeafConnRequest(NodeAddr_t peer, Socket *socket, const GnutCoord&, double, double) {
  return FALSE;
}
//=============================================================================
/* connection request rejected */
void GnutellaApp::ConnectionRejected(NodeAddr_t peer) {
  debug_info("connection request from %d to %d rejected\n", addr_, peer);

  if (find_servent(peer)) 
    remove_servent(peer);
}

//=============================================================================
/* connection failed, probably at network level */
void GnutellaApp::ConnectionFailed(NodeAddr_t peer) {
  debug_info("connection to %d failed\n", peer);

  if (find_servent(peer))
    remove_servent(peer);
}

//=============================================================================
/* connection request timed out */
void GnutellaApp::ConnectionTimeout(NodeAddr_t peer) {
  debug_info("connection from %d to %d timed out\n", addr_, peer);

  if (find_servent(peer))
  {
    remove_servent(peer);
  }
}

//=============================================================================
/* connection closed by peer */
void GnutellaApp::ConnectionClosed(NodeAddr_t peer) {

  GnutUtil::gnutError("connection to %d closed by peer %d\n", addr_, peer);

  PeerMap_t::iterator it = lpeers_.find(peer);
  GNUT_ASSERT(it != lpeers_.end());
  lpeers_.erase(it);
  mGnutStats.setPeerConns(addr_, lpeers_.size());
  degree_.increment(lpeers_.size());

  PeerDistances::iterator iter = mPeerDistances.find(peer);
  GNUT_ASSERT(iter != mPeerDistances.end());
  mPeerDistances.erase(iter);

  if (watchDog_.status() == TIMER_IDLE)
  {
      // If the watchdog had been turned off because we had all the 
      // connections we needed, turn it back on.
      // TODO: AD
      GnutUtil::gnutTrace("%d can handle more connections. Re-enabling "
                        "watchdog\n", addr_);
      watchDog_.resched();
  }
}

//=============================================================================
/* PING request received */
void GnutellaApp::PingRequest(NodeAddr_t peer, int ttl, DescriptorID_t id) {
   int i=0;
   int cnt = servent_cache_.size();

   GnutUtil::gnutTrace("PingRequest from %d to %d\n", peer, addr_);

   NodeAddr_t *addrs = (NodeAddr_t *)malloc(sizeof(NodeAddr_t)*cnt);
   for(ServentMap_t::iterator si = servent_cache_.begin(); si!=servent_cache_.end(); si++) {
     addrs[i] = si->addr_;
     i++;   
   }

   // Need to limit the number of pongs a bit because we don't implement
   // pong caching in this simulation.
   if (lpeers_.size() * 2 < max_deg_)
   {
       GnutUtil::gnutTrace("%d sending pong to %d\n", addr_, peer);
       gagent_->Pong(peer, cnt, addrs, id);
   }
}

//=============================================================================
/* PONG reply received */
void 
GnutellaApp::
PongReply(NodeAddr_t peer, const PongRes& pong) {

  if (((lpeers_.size() + gagent_->conn_pending_.size()) < max_deg_) && 
      (pong.addr_!= addr_)) 
  {
    if (isAssociated(pong.addr_) == false) 
    {
      gagent_->Connect(pong.addr_, CONN_TIMEOUT);
    }
  }

  if (find_servent(pong.addr_))
    return;

  if (pong.addr_!= addr_) {
    ServentRec *newsrv = new ServentRec(pong.port_, pong.addr_);
    servent_cache_.push_back(*newsrv);
    while(servent_cache_.size() >= KNOWNCACHE_LIMIT)
        servent_cache_.pop_front();
  }
}
//=============================================================================
/* Query Request received */
void GnutellaApp::QueryRequest(NodeAddr_t peer, char* /* payload */, 
                               DescriptorID_t id, 
                               const GnutQueryMsg& aQuery) {
    debug_info("QueryRequest from %d %d %d\n", peer, isFreeloader_, state_);

    mGnutStats.newNodeTime(addr_, aQuery.getHeader().id_, 
                                      GnutUtil::now());

    if (hasContent_)
    {
        GnutUtil::gnutError("Peer %d sending query hit in response to query with id %f through peer %d\n", addr_, id, peer);

        gagent_->QueryHit(peer, id, aQuery);    
    }

// Initial UMASS model used
#if 0
  double hitprob=0;

  rquery++; 
  if (isFreeloader_ || state_==PS_OFFLINE) 
    return;

  unsigned char nhit = (unsigned char)((float)rand()/(float)RAND_MAX *5.0);
  
  if (ac_) {
    int M = ac_->sys_->nFiles_;
//    int fnum = atoi(payload);
    // TODO: AD - fix!
//    int fnum = 0;
//    hitprob = (double)(M+1-fnum)/(double)(M+1);
    hitprob = 0.05; // TODO: AD - fix!
  }

  double hit = (float)rand()/(float)RAND_MAX; //random probability of hit
  if (hitprob>0 && hit<hitprob || (hitprob==0 && hit<HIT_PROB))
    gagent_->QueryHit(peer, id); 
#endif /* 0 */
}
//=============================================================================
/* QueryHit received*/
void GnutellaApp::QueryHitReply(NodeAddr_t peer, const GnutQueryHitMsg& hit) 
{
    debug_info("QueryHit received by %d from %d\n", addr_, peer);

    if (std::find(mQueryHitList.begin(), mQueryHitList.end(),
                  createQueryHitId(hit.getNeighbourTimeSent(), 
                                   hit.getHeader().id_))
        == mQueryHitList.end())
    {
        mQueryHitList.push_back(createQueryHitId(hit.getNeighbourTimeSent(),
                                                 hit.getHeader().id_));
        
        mGnutStats.newHitTime(hit.getHeader().id_, GnutUtil::now());
        if (GnutUtil::now() > mMinDisconnectTime)
        {
            GnutUtil::gnutError("Time: %f. Min disc time: %f\n",
                                GnutUtil::now(), mMinDisconnectTime);
            if (ac_ && state_==PS_ACTIVE)
            {
                ac_->expire(NULL);
            } 
        }
    }
    else
    {
        GnutUtil::gnutError("Query hit was a duplicate\n");
    }
}

//=============================================================================
/* available for more connections? */
bool GnutellaApp::avail(EPeerType) const
{
    return (lpeers_.size() + gagent_->conn_pending_.size() < max_deg_);
}
//=============================================================================
/* configure a list of bootstrap servers from a file */
void GnutellaApp::setBootServer(FILE *file) {
  char line[20];
  NodeAddr_t servent;
  BootServerRec *brec;

  while(fgets((char *)&line, 20, file)) {
#ifdef PDNS
    Tcl &tcl = Tcl::instance();
    tcl.evalf("[Simulator instance] convert-ipaddr %s\n", line);
    servent = atoi(tcl.result()); 
#else
    servent = atoi(line);
#endif
    brec = new BootServerRec(servent, LISTEN_PORT);
    if (brec) 
      bserver_list_.insert(bserver_list_.end(), *brec);
  }
}

//=============================================================================
/* find a servent in the servent cache */
int GnutellaApp::find_servent(NodeAddr_t peer) {
  for(ServentMap_t::iterator si = servent_cache_.begin(); si != servent_cache_.end(); si++) {
    if (si->addr_==peer)
      return TRUE;
  }
  return FALSE;
}

//=============================================================================
/* remove a servent from the known servent cache */
void GnutellaApp::remove_servent(NodeAddr_t peer) 
{
    for(ServentMap_t::iterator si = servent_cache_.begin(); 
        si != servent_cache_.end(); si++) 
    {
        if (si->addr_==peer) 
        {
            servent_cache_.erase(si);
            return;
        }
    }
}

//=============================================================================
/* send an update to the bootstrap server */ 
void GnutellaApp::update_bootcache() {
  if (smpBoot_)
	return;

  gagent_->UpdateBootcache();
}
//=============================================================================
void GnutellaApp::watchdogExpired(void)
{
    bool reload_timer;

    connect(true, &reload_timer);
    update_bootcache();
    
    if (reload_timer)
    {
        watchDog_.resched();
    }
}
//=============================================================================
void GnutellaApp::pingExpired(void)
{
    ping();
    ping_timer_.resched();  
}
//=============================================================================
int GnutellaApp::command(int argc, const char*const* argv) {
  Tcl &tcl = Tcl::instance();
  const char* const cmd = argv[1];

  if (argc==2) {
    if (strcmp(cmd, "start")==0) {
      join();
      return TCL_OK;
    }
    else if (strcmp(cmd, "join")==0) {
      join();
      return TCL_OK;
    }
    else if (strcmp(cmd, "cleanup")==0) {
      return cleanup();
    }
    else if (strcmp(cmd, "leave")==0) {
      leave();
      return TCL_OK;
    }
    else if (strcmp(cmd, "stop")==0) {
      stop();
      return TCL_OK;
    }
    else if (strcmp(cmd, "statistics")==0) {
      gagent_->statistics();
      return TCL_OK;
    }
    else if (strcmp(cmd, "search")==0) {
      search();
      return TCL_OK;
    }

  }
  else if (argc==3) {
    if (strcmp(cmd, "set-agent")==0) {
      GnutellaAgent *tmp = (GnutellaAgent *)tcl.lookup(argv[2]);
      if (tmp) {
        gagent_ = tmp;
        agent_ = tmp; 
        return TCL_OK;
      }
      return TCL_ERROR;
    }
    else if (strcmp(cmd, "use-bootserver")==0) {
      FILE *file = fopen(argv[2], "r");
      if (file) {
        setBootServer(file);
        return TCL_OK;
      }
      return TCL_ERROR;
    }
    else if (strcmp(cmd, "use-smpbootserver")==0) {
      SmpBootServer *tmp = (SmpBootServer *)tcl.lookup(argv[2]);
      if (tmp!=NULL) {
        bootSrv_ = tmp;
        smpBoot_ = TRUE;
        return TCL_OK;
      }
      return TCL_ERROR;
    }
    else if (strcmp(cmd, "set-hascontent") == 0) {
      if (strcmp(argv[2], "0") == 0) {
        setHasContent(false);
      }
      else
      {
        setHasContent(true);
      }
    }
  }
  else if (argc==4) {
    if (strcmp(cmd, "attach-peersys")==0) {
      PeerSys *tmp = (PeerSys *)tcl.lookup(argv[2]);
      if (tmp!=NULL) {
        ac_ = new ActivityController(tmp, this, atoi(argv[3]));
        return TCL_OK;
      }
      return TCL_ERROR;
    }
  }

  return TCL_ERROR;
}
//=============================================================================
void 
GnutellaApp::
setHasContent(bool aHasContent)
{
    hasContent_ = aHasContent;
    mGnutStats.setPeerHasContent(addr_, aHasContent);
}
//=============================================================================
bool 
GnutellaApp::
isAssociated(NodeAddr_t aPeer) const
{
    if ((lpeers_.find(aPeer) != lpeers_.end()) ||
        (gagent_->lsocks_.find(aPeer) != gagent_->lsocks_.end()) ||
        (gagent_->conn_pending_.find(aPeer) != gagent_->conn_pending_.end()))
    {
        return true;
    }
    else
    {
        return false;
    }
}
//=============================================================================
void 
GnutellaApp::
notifyRecv(NodeAddr_t src, const GnutCoord& remoteCoord, double timeDiff, double remoteError)
{
    GNUT_ASSERT(timeDiff > 0.0);
    
    // Update our coordinates
    mVivaldi->updateCoordinates(remoteCoord, timeDiff, remoteError);

    PeerDistances::iterator iter = mPeerDistances.find(src);
    if (iter != mPeerDistances.end())
    {
        // Update the estimated distance to the remote peer
        double new_distance = estimateDistance(remoteCoord);
        GnutUtil::gnutTrace("%d updating distance to %d to be %f\n",
                            addr_, src, new_distance);
        iter->second.mDistance = new_distance;
    }
}
//=============================================================================
double
GnutellaApp:: estimateDistance(const GnutCoord& aRemoteCoord) const
{
    return (mVivaldi->getCoordinates() - aRemoteCoord).funkyAbs();
}
//=============================================================================
static class GnutellaAppClass: public TclClass {
public:
  GnutellaAppClass(): TclClass("PeerApp/GnutellaApp") {}
  TclObject* create(int argc, const char*const* argv) {
    return (new GnutellaApp(atoi(argv[4]), argv[5], atoi(argv[6]), atof(argv[7])));
  }
}class_gnutellaapp;
//=============================================================================
