//=============================================================================
// File: gnutstats.h
// Author: Andre Dufour
//
// Class type: singleton
// Description: Keeps track of global simulation statistics and timestamps.
//=============================================================================

#ifndef __GNUTSTATS_H__
#define __GNUTSTATS_H__

#include <map>
#include "gnut_types.h"
#include "gnutqueryrecord.h"

class GnutStats : public TclObject
{
public:
    // = ACCESS
    
    // Initialize will only be called from TCL when the object is created.
    // It must only be called once and before any calls to instance.
    // Usage notes:
    // - aRootDir should be an absolute unix-style (i.e. forward slashes)
    //   path. It must not end with a /
    //   e.g. /home/username/gnut_stats
    // - aTopologyId and aRandomSeed are used to uniquely name directories
    //   and result files, nothing more.
    static GnutStats* initialize(
        unsigned int         aNumPeers,     // number of peers in simulation
        const char* const    aTopologyId,   // a unique id for the topology 
        const char* const    aRootDir,      // root dir where to output results
        long                 aRandomSeed);  // the random seed used for this sim

    // All access to this singleton class starts by getting a reference with
    // this method.
    static GnutStats& instance(void);


    // = ACTION

    // Report the sending of a new query
    void newQuery(DescriptorID_t aId, double aTimeSent);

    // Report that a query has reached a node
    void newNodeTime(NodeAddr_t aNode, DescriptorID_t aId, double aTime);

    // Report that a query hit has been recv'd
    void newHitTime(DescriptorID_t aId, double aTime);

    // Record a time when a node dumped an existing connection in favour
    // of a new one based on the predicted RTT
    void newConnDumpTime(double aTime);

    // Report a change in the number of connections a peer has
    void setPeerConns(NodeAddr_t aNode, unsigned int aNumConns);

    // Report a change in the error on a peer's coordinates
    void setPeerCoordError(NodeAddr_t aNode, double aError);

    // Report a change in role of a peer
    void setPeerState(NodeAddr_t aNode, EGnutApp aRole);

    // Report a change in the online status of a peer
    void setPeerOnline(NodeAddr_t aNode, bool aOnline);

    // Report a change in the content status of a peer
    void setPeerHasContent(NodeAddr_t aNode, bool aHasContent);

    // Take a snapshot of the peer situation and record it for output to the
    // log file. Normally this is done periodically.
    void takeSnapshot(double aTime);

    // Deal with TCL commands
    int command(int argc, const char* const* argv);

private:
    // = ACTION

    int dumpStats(void) const;
    int dumpPeerFile(void) const;
    int dumpTimeFiles(void) const;

    // = TYPES

    // Info on a peer
    struct PeerRecord
    {
        PeerRecord(void)
        : mRole(EGnutAppLeaf),
          mNumConns(0),
          mError(1.0),
          mOnline(false),
          mHasContent(false)
        { }
            
        EGnutApp      mRole;
        unsigned long mNumConns;
        double        mError;
        bool          mOnline;
        bool          mHasContent;
    };

    // A summary of the peer situation taken as a snapshot at a particular time.
    struct PeerSummary
    {
        PeerSummary(double aTime)
        : mTime(aTime),
          mNumLegacy(0),
          mNumLeaf(0),
          mNumUltra(0),
          mMedianError(1.0),
          mNumHasContent(0)
        { }

        double                      mTime;        // Time snapshot was taken.
        long                        mNumLegacy;   // Number of legacy peers
        long                        mNumLeaf;     // Number of leaves
        long                        mNumUltra;    // Number of ultrapeers 
        double                      mMedianError; // Median coordinate error
        std::vector<unsigned long>  mConnections; // Number of connections
                                                  //  for all peers.
        long                        mNumHasContent;// Number of peers with
                                                   // content that are online
    };

    typedef std::map<DescriptorID_t, GnutQueryRecord> QueryMap;

    typedef std::map<NodeAddr_t, PeerRecord> PeerRecMap;

    typedef std::vector<PeerSummary> PeerSummaryVector;

    typedef std::list<double> ConnDumpTimeList;

    typedef std::list<double> OnlineTimeList;


    // = DATA

    static GnutStats*   mInstance;
    QueryMap            mQueries;
    PeerRecMap          mPeerRecs;
    PeerSummaryVector   mPeerSummaries;
    char*               mTopologyId;
    char*               mRootDir;
    const long          mRandomSeed;
    ConnDumpTimeList    mDumpTimeList;
    OnlineTimeList      mOnlineTimes;


    // = FOUNDATION

    GnutStats(unsigned int         aNumPeers, 
              const char* const    aTopologyId, 
              const char* const    aRootDir,
              long                 aRandomSeed);
    virtual ~GnutStats(void);

    // Disable copy constructor and assignment operator
    GnutStats(const GnutStats&);
    GnutStats& operator=(const GnutStats&);
};
//=============================================================================
#endif /* __GNUTSTATS_H__ */

