// File: leafagent.cc
// Original by Qi He
// Modified by Andre Dufour

#include "leafagent.h"
#include "gnut_types.h"
#include "gnutellaapp.h"
#include "peersys.h"
#include "gnut_util.h"
#include "gnutstats.h"
#include "pendingreqentry.h"
#include "conntimer.h"
#include "gnutmsgparser.h"
#include "gnutpayload.h"
#include "gnutleafconnmsg.h"
#include "gnutleafokmsg.h"
#include "gnutquerymsg.h"
#include "gnutqueryhitmsg.h"
#include "ivivaldimanager.h"

//=============================================================================

/* PeerAgent for Leaf */
LeafAgent::LeafAgent(NodeAddr_t addr): GnutellaAgent(addr) {
}
//=============================================================================
LeafAgent::LeafAgent(GnutellaApp *app): GnutellaAgent(app) {
}
//=============================================================================
LeafAgent::~LeafAgent(void)
{ 
    GnutUtil::gnutTrace("LeafAgent::~LeafAgent()\n");
}
//=============================================================================
int LeafAgent::upcall_recv(Socket *sock, PacketData *pktdata, Handler *) 
{
    //continue only if peer is currently online
    if(app_->state_==PS_OFFLINE) 
    {
        return pktdata->size();  
    }

    GnutMsgParser& parser = GnutMsgParser::instance();


    NodeAddr_t src;
    int res, cnt, block=FALSE, nhit;
    SockMap_t::iterator si;
    PendingConns_t::iterator pi;

    src = sock->peer_.addr_;

    GnutPayload payload;
    GnutDescHdr header;
    GnutCoord remote_coord;
    double remote_error;
    double time_sent;
    const double current_time = GnutUtil::now();

    GnutMsgParser::EGnutMsgType msg = parser.parse(pktdata->data(), &header,
                                                   &payload, &remote_coord,
                                                   &remote_error, &time_sent);

    GNUT_ASSERT(current_time > time_sent);

    // Update the vivaldi coordinates!
    GnutUtil::gnutTrace("Updating coordinates for node %d\n", gapp_->addr_);
    gapp_->notifyRecv(src, remote_coord, current_time - time_sent, remote_error);

    GnutStats::instance().setPeerCoordError(
        gapp_->addr_, gapp_->mVivaldi->getLocalError());

    switch (msg) 
    {
    case GnutMsgParser::GNUT_BOOTSTRAP:
    case GnutMsgParser::GNUT_LEGACY_CONNECT:
    case GnutMsgParser::GNUT_ULTRA_CONNECT:
    case GnutMsgParser::GNUT_LEAF_CONNECT:
    case GnutMsgParser::GNUT_PING:  
        break;

    case GnutMsgParser::GNUT_BOOTSTRAP_RES:
        {
            const BootstrapRes& bsr = payload.getBootstrapRes();
            gapp_->BootstrapResult(bsr.cnt_, bsr.servents_);
            payload.freeBootstrapRes();
        }
        break;

    case GnutMsgParser::GNUT_LEAF_OK:
    case GnutMsgParser::GNUT_OK:
        GNUT_ASSERT(0);
        break;

    case GnutMsgParser::GNUT_ULTRA_OK:
        for(PendingConns_t::iterator i = conn_pending_.begin(); 
            i != conn_pending_.end(); 
            i++) 
        {
            if(i->second.peer_==src) 
            {
                conn_pending_.erase(i);
                break; // for
            }
        }

        si = lsocks_.find(src);
        if(si==lsocks_.end()) 
        {
            SockEntry *newentry = new SockEntry(sock, src, SOCK_ULTRA);
            lsocks_.insert(SockMap_t::value_type(src, *newentry));
        }
        gapp_->ConnectSucceeded(src, EGnutAppUltrapeer, remote_coord,
                                remote_error); 
        
        break;

    case GnutMsgParser::GNUT_REJ:
        pi = conn_pending_.find(src);
        if(pi!=conn_pending_.end()) 
        {
            gapp_->ConnectionRejected(src);

            for(pi=conn_pending_.begin(); pi!=conn_pending_.end(); pi++) 
            {
                if(pi->second.peer_==src) 
                {
                    conn_pending_.erase(pi);
          
                    sock->close();
                    break;
                }
            }
        }
        break;

    case GnutMsgParser::GNUT_PONG:
        if(find_desc(header.id_)==0 && find_ping(header.id_, 0)==-1)
        {
            payload.freePongRes();
            break;
        }

        gapp_->PongReply(src, payload.getPongRes());
        if(find_desc(header.id_)==0) 
        {
            pongcnt++;
        }

        payload.freePongRes();

        break;

    case GnutMsgParser::GNUT_QUERY:
        if(find_desc(header.id_)!=0)
            break;

        if(find_query(header.id_, 1)!=-1)
            break;

        // TODO: AD - fix payload parameter.
        {
            GnutQueryMsg query(pktdata->data());
            gapp_->QueryRequest(src, 0, header.id_, query);
        }
        break;

    case GnutMsgParser::GNUT_QUERYHIT:
        if(find_desc(header.id_)==0 && find_query(header.id_, 0)==-1) 
        {
            break;
        }

        {
            GnutQueryHitMsg hit(pktdata->data());
            gapp_->QueryHitReply(src, hit);
        }

        if(find_desc(header.id_)==0) 
        {
            nqueryhit++;
        }

        break;

    default: 
        debug_warning("WARNING: unknown Message Type\n");
        GNUT_ASSERT(0);
        break;

    };

    return pktdata->size();  
}
//=============================================================================
/* compose and send Gnutella Connection OK msg */
void LeafAgent::gnutella_ok(Socket *sock) 
{
    GnutLeafOkMsg ok(gapp_->mVivaldi->getCoordinates(),
                     gapp_->mVivaldi->getLocalError());
    PacketData* pkt = GnutMsgParser::instance().newpacket(ok);
    sock->send(ok.getSize(), pkt);
}
//=============================================================================
void LeafAgent::gnutella_req(Socket *sock) 
{
    GnutLeafConnMsg con(gapp_->mVivaldi->getCoordinates(),
                        gapp_->mVivaldi->getLocalError());
    PacketData* pkt = GnutMsgParser::instance().newpacket(con);
    sock->send(con.getSize(), pkt);
  
    if(conn_pending_.size()==1)
        conn_timer_.resched(CONN_TIMEOUT);
}
//=============================================================================
static class LeafAgentClass: public TclClass {
public:
  LeafAgentClass(): TclClass("SocketApp/PeerAgent/GnutellaAgent/LeafAgent") {}
  TclObject* create(int argc, const char*const* argv) {
    return (new LeafAgent(atoi(argv[4])));
  }
}class_leafagent;
//=============================================================================


