// File: pdnsbootserver.cc
// Original by Qi He
// Modified by Andre Dufour

#include "pdnsbootserver.h"
#include "gnutellaapp.h"
#include "gnut_types.h"
#include "gnut_util.h"
#include "gserventrec.h"
#include "gnutbootstrapresmsg.h"

//=============================================================================
/* simple bootstrap server in PDNS */
PDNSBootServer::PDNSBootServer(NodeAddr_t addr): SmpBootServer() {
  addr_ = addr;
}

//=============================================================================
BootstrapRes* 
PDNSBootServer::
BootstrapRequest(NodeAddr_t peer, GnutellaApp *app, EPeerType peertype) 
{
  int i = 0;
  int cnt = 0, asize = 0; 
  double thresh, prob;
  cnt = pservent_cache_.size();
  asize = cnt > MAX_BOOTREPLY? MAX_BOOTREPLY: cnt;

  if (cnt <= MAX_BOOTREPLY)
    thresh = 1;
  else
    thresh = (float)MAX_BOOTREPLY/(float)cnt;

  NodeAddr_t* addrs = new NodeAddr_t[asize];

  for (PDNSServentMap_t::iterator fi = pservent_cache_.begin(); 
       fi != pservent_cache_.end(); 
       ++fi) 
  {
    prob = (double)rand()/(double)RAND_MAX;
    if(prob <= thresh) 
    {
      addrs[i] = fi->second.addr_;
      i++;
      if(i>=asize)
        break;
    }
  }

  if(peertype!=EPeerLeaf) 
  {
    PDNSServentMap_t::iterator fi = pservent_cache_.find(peer);
    if(fi==pservent_cache_.end()) 
    {
      PDNSServentEntry *newsrv;
      newsrv = new PDNSServentEntry(LISTEN_PORT, peer);
      pservent_cache_.insert(PDNSServentMap_t::value_type(peer, *newsrv));
      broadcast(ADD_PEER, peer);
    }
  }

  BootstrapRes* res = new BootstrapRes(i, addrs);
  delete[] addrs;
  return res;
}

//=============================================================================
void PDNSBootServer::RemovePeer(NodeAddr_t peer) {
  PDNSServentMap_t::iterator fi = pservent_cache_.find(peer);
  if(fi!=pservent_cache_.end()) {
    pservent_cache_.erase(fi);
    broadcast(REM_PEER, peer);
  }
}

//=============================================================================
/* PDNSBootServer receives updates from boot servers in other physical nodes*/
int PDNSBootServer::upcall_recv(Socket *sock, PacketData *pktdata, Handler *){
  PDNSBootMsg *msg = NULL;
  NodeAddr_t peer;
  PDNSServentMap_t::iterator pi;
  PDNSServentEntry *newentry = NULL;

  if(pktdata==NULL)
    return 0;
  unsigned char *data = pktdata->data();
  msg = parse(pktdata->size(), data);

  if(msg==NULL) 
    return pktdata->size();

  peer = msg->peer_;
  switch(msg->type_) {

  case ADD_PEER:
    pi=pservent_cache_.find(peer);
    if(pi==pservent_cache_.end()) {
      PDNSServentEntry *newentry = new PDNSServentEntry(BOOTSERVER_PORT, peer);
      pservent_cache_.insert(PDNSServentMap_t::value_type(peer, *newentry));
    } else {
      //warning
    }
    
    break;

  case REM_PEER:
    pi=pservent_cache_.find(peer);
    if(pi!=pservent_cache_.end()) {
      pservent_cache_.erase(pi);
    } else {
      //warning
    }
    break;

    default:
      debug_warning("WARNING: unknown msg type %d\n", msg->type_);
  }
  
  free(msg);
  return pktdata->size();
}

//=============================================================================
/* parse the messages received from other boot servers*/
PDNSBootMsg *PDNSBootServer::parse(int len, unsigned char *data) {
  PDNSBootMsg *msg = NULL;

  if(len!=sizeof(PDNSBootMsg)) 
    return NULL;

  msg = (PDNSBootMsg *)malloc(sizeof(PDNSBootMsg));
  memcpy((void *)msg, (void *)data, len);
  if(msg->type_!=ADD_PEER && msg->type_!=REM_PEER) {
    free(msg);
    return NULL;
  }

  return msg;
}

//=============================================================================
/* broadcast an update (about available peers) to other boot servers */
void PDNSBootServer::broadcast(int type, NodeAddr_t peer) {
  PDNSBootMsg msg;
  
  if(bservers_.size()==0)
    return;

  msg.type_ = type;
  msg.peer_ = peer;

  for(BootServerMap_t::iterator pi=bservers_.begin(); pi!=bservers_.end(); pi++) {
    PacketData *userdata = new PacketData(sizeof(PDNSBootMsg));
    unsigned char *dataptr = userdata->data();
    memcpy((void *)dataptr, (void *)&msg, sizeof(PDNSBootMsg));  
    pi->second.sock_->send(sizeof(PDNSBootMsg), userdata);
  }
}

//=============================================================================
int PDNSBootServer::command(int argc, const char*const* argv) {
  Tcl &tcl = Tcl::instance();

  if(argc==4) {
    if(strcmp(argv[1], "add-bootserver")==0) {
      NodeAddr_t saddr = atoi(argv[2]);
      Socket* sock = (Socket *)tcl.lookup(argv[3]);
      if(sock!=NULL) {
	BootServerEntry *newentry = new BootServerEntry(saddr, sock);
	bservers_.insert(BootServerMap_t::value_type(saddr, *newentry));
	debug_info("boot server entry added %d\n", saddr);
	fflush(NULL);
	sock->connect();
	return TCL_OK;
      } else
	return TCL_ERROR;    
    }  
  }

  return SmpBootServer::command(argc, argv);
}

//=============================================================================
static class PDNSBootServerClass: public TclClass {
public:
  PDNSBootServerClass(): TclClass("SocketApp/SmpBootServer/PDNSBootServer") {}
  TclObject* create(int, const char*const* argv) {
    return (new PDNSBootServer(atoi(argv[4])));
  }
}class_pdnsbootserver;
//=============================================================================


