function [yp, bad] = deval(J, y, ypold)

% DEVAL Finds derivatives of unknowns from the Jacobian of the homotopy function
%
% function [yp, bad] = deval(J, y, ypold)
%
% INPUTS:
% J        - String containing name of user-supplied homotopy Jacobian. 
% y        - Current value of homotopy trajectory.
% ypold    - Previous calculated derivative from homotopy Jacobian.
%
% OUTPUTS:
% yp       - Derivative calculated from homotopy Jacobian.
% bad      - Boolean, determines if there were problems during derivative
%            calculation.
% 
% Originally written by Heath Hofmann
% Deptartment of Electrical Engineering and Computer Science 
% The University of California at Berkeley
% 
% Modified by Archit Singhal
% Research Intern at Simon Fraser University
% Home University
% Deptartment of Electrical Engineering
% Indian Institute of Technology Roorkee

n = length(y)-1;
lambda = y(1);
X = y(2:n+1);
z = zeros(n+1,1);

% Get Jacobian of the homotopy function

DH = feval(J,lambda,X);

% Checks to see if Jacobian contains NAN's or infinite #'s

bad = norm(isnan(DH) + isinf(DH),1);

% Checks to see if Jacobian is rank deficient

if ~bad
    bad = (rank(DH) < n);
end

if ~bad
    
    % Take QR factorization of DH
    
    [~,R] = qr(DH);

    % Calculate unnormalized derivative vector

    z(1:n) = -R(:,1:n)\R(:,(n+1));
    z(n+1) = 1;

    % Normalize derivative vector
    
    zn = z/norm(z);

    % Get appropriate direction (angle between current and previous vector < 90 deg.)

    yp = sign(zn'*ypold)*zn;

else
    
    yp = zeros(n+1,1);

end

end