function [Xsolout, sout, lambdaout, Xout, NJaceval] = pchomotopy(J, X0, sfinal, tol, k, trace)

% PCHOMOTOPY Solves a nonlinear system of equations using ODE-based homotopy,
% with a variable-order variable-step predictor-corrector integration technique
%
% [Xsolout, sout, lambdaout, Xout, NJaceval] = pchomotopy(J, X0, sfinal, tol, k, trace)
%
% INPUTS:
% J      - String containing name of user-supplied homotopy Jacobian.
% X0     - Solution of homotopy when embedded parameter lambda = 0.
% sfinal - Final value of arc length s (Default: sfinal = 30).
% tol    - Two-dimension vector that determines desired accuracy. 
%          tol(1) provides relative bound to local truncation error
%          of numerical integration (Default: tol(1) = 1.e-6).
%          tol(2) provides bound for lambda being close to 1 before
%          accepting solution (Default: tol(2) = 1.e-6).
% k      - order of predictor-corrector method (Default: k = 4).
% trace  - If nonzero, each step is printed. (Default: trace = 0).
%
% OUTPUTS:
% Xsolout   - Returned solutions of f(X)=0 (i.e., solutions of homotopy 
%             when lambda = 1), one column-vector per solution
% sout      - Returned integration arc length points (column-vector).
% lambdaout - Returned values of embedded parameter along trajectory (row
%             vector).
% Xout      - Returned solutions along trajectory, one solution column-vector
%             per tout-value.
% NJaceval  - Number of evaluations of the homotopy Jacobian J.
%
% SUBROUTINE:
% deval.m   - Evaluates derivative from homotopy Jacobian
%
% The trajectory can be displayed by: plot(lambdaout, Xout).
% 
% Originally written by Heath Hofmann
% Deptartment of Electrical Engineering and Computer Science 
% The University of California at Berkeley
% 
% Modified by Archit Singhal
% Research Intern at Simon Fraser University
% Home University
% Deptartment of Electrical Engineering
% Indian Institute of Technology Roorkee

if nargin < 3, sfinal = 30; end                     % nargin - number of input arguments
if nargin < 4, tol = [1e-6 1e-6]; end                
if nargin < 5, k = 4; end
if nargin < 6, trace = 0; end

% Initialization

n = length(X0);
m = n+1;
gamma = zeros(k+1,1);
gamma(1) = 1;
gammas = zeros(k+1,1);

for i = 2:k+1
    for j = 1:i-1
        gamma(i) = gamma(i) - gamma(j)/(i+1-j);     
    end
end
        
for i = 1:k+1
    for j = 1:i
        gammas(i) = gammas(i) + gamma(j);           
    end
end

beta = (pascal(k,1)'*gamma(1:k))';                  % pascal(N,1) is the lower triangular Cholesky factor (up to signs
betak = beta(1);                                    % of columns) of the Pascal matriX.  It is involutory (is its own
beta(1:k-1) = beta(2:k);                            % inverse).
beta(k) = 0;
betas = (pascal(k,1)'*gammas(1:k))';
deltas = (betas - beta)/betak;
cpp1 = gamma(k+1);
cspp1 = gammas(k+1);
W = cpp1/(cspp1-cpp1);
pow = 1/(k+1);         
PI = kron(abs(pascal(k+1,1))',eye(m));              % kron(X,Y) is the Kronecker tensor product of X and Y.
Qb = zeros(k+1);                                    % The result is a large matriX formed by taking all possible
Qb(1,1) = 1;                                        % products between the elements of X and those of Y
Qb(2,2) = 1;

for i = 2:k+1
    for j = 3:k+1
        Qb(j,i) = (i-1)*(2-j)^(i-2);
    end
end

Qb = inv(Qb);
Q = kron(Qb, eye(m));
G = kron([betak 1 zeros(1,k-1)]',eye(m));
Gt = Q*G;
f = [];
s = 0;
numsol = 0;
NJaceval = 0;
hmax = sfinal;
hmin = 1e-12;
h = 0.1;
lambda = 0;
y0 = [0; X0];
lambdaout = 0;
fc = 0*y0;
fc(1,1) = 1;
sout = 0;
Xout = X0;
Xsol = zeros(n,1);
Xsolout=[];
endgame = 0;

% First-order Predictor-Corrector Startup Procedure

count = 0;
y = y0;
fail = 0;

while (count < k) && (h >= hmin) 
    
    % First-order Adams Bashforth (Euler's method), Predictor

    [fp, bad] = deval(J,y,fc);
    if bad, fail = 1; end
    yp = y + h*fp;   

    % First-order Adams-Moulton, Corrector

    [fc, bad] = deval(J,yp,fp);
    NJaceval = NJaceval+2;
    if bad, fail = 1; end
    yc = y + h*fc;

    % Determine error using "Milne's trick"

    error = -.5*(yp-yc);

    % Estimate the error and the acceptable error

    delta = norm(error,'inf');

    % Error in function evaluation

    tau = tol(1)*max(norm(y,'inf'),1.0);

    % Update the solution only if the error is acceptable, otherwise
    % start over from scratch
    
    if (delta <= tau) && ~fail && (yc(1) < 1)
        y = yc;
        lambda = y(1);
        s = s + h;
        y = yc;
        sout = [sout s];
        lambdaout = [lambdaout lambda];
        Xout = [Xout y(2:m)];
        count = count + 1;
        f = [fc f];
    else
        f = [];
        count = 0;
        sout = 0;
        if fail || (yc(1) >= 1)
            h = h/4;
            fail = 0;
        else
            h = min(hmax, 0.8*h*(tau/delta)^pow);
        end
    end
end

% Set up  problem

d = f*deltas';
Y = [y; h*d];

for i = 1:k-1
    Y = [Y; h*f(:,i)];
end

Zc = Q*Y;
fnpk = fc;

% Start Variable-Step Predictor-Corrector Method

while (s < sfinal) 
    
    % Predict

    Zp = PI*Zc;

    % Correct

    [fnpk,fail] = deval(J,Zp(1:m),fnpk);
    NJaceval = NJaceval + 1;
    
    if ~fail
        
        F = (h*fnpk - Zp(m+1:2*m));
        Zctemp = Zp + Gt*F;
   
        % Determine error using Milne's trick

        error = W*(Zctemp(1:m) - Zp(1:m));

        % Estimate the error and the acceptable error

        delta = norm(error,'inf');

        % Error in function evaluation

        tau = tol(1)*max(norm(y,'inf'),1.0);

        % Update the solution only if the error is acceptable
        % or if lambda = 1 hasn't been crossed
        
        if (delta < tau) 
            
            lambdatemp = Zctemp(1);
            ytemp = Zctemp(1:m);
            yptemp = fnpk;
            Xtemp = Zctemp(2:m);   
            stemp = s + h;

            % Determine if trajectory is close to lambda = 1 or crosses lambda = 1

            close = ( abs( lambdatemp - 1 ) < tol(2) | abs( lambda - 1 ) < tol(2));
            cross = ((lambda < 1) & (lambdatemp > 1)) | ((lambdatemp < 1) & (lambda > 1));

            % Determine if we are in endgame

            if ~endgame
                endgame = (cross & ~close);
                if endgame
                    if (lambda < 1)
                        lambda1 = lambda;
                        s1 = s;
                        lambdap1 = yp(1);
                        lambda2 = lambdatemp;
                        s2 = stemp;
                        lambdap2 = yptemp(1);
                    else
                        lambda2 = lambda;
                        s2 = s;
                        lambdap2 = yptemp(1);
                        lambda1 = lambdatemp;
                        s1 = stemp;
                        lambdap1 = yp(1);
                    end
                end
            end
            
            % If lambda is within tol(2) of 1, write solution
            
            if close
                endgame = 0;
                if (numsol == 0) || (norm(Xtemp - Xsol) > 100*tol(2)*norm(Xtemp))
                    Xsol = Xtemp;
                    numsol = numsol + 1;
                    Xsolout = [Xsolout Xsol];
                    h = hmax;
                end
            end
            
            if ~endgame
                 Zc = Zctemp;
                 y = ytemp;
                 yp = yptemp;
                 s = stemp;
                 lambda = lambdatemp;
                 sout = [sout stemp];
                 lambdaout = [lambdaout, lambdatemp];
                 Xout = [Xout Xtemp];
                 if trace
                     clc, s, lambda, numsol
                     pause
                 end
                 
            else
                
                % Create spline lambda(s) from previous data points to determine
                % best guess for neXt step size
                         
                if (lambdatemp < 1)
                    lambda1 = lambdatemp;
                    s1 = stemp;
                    lambdap1 = yptemp(1);
                else
                    lambda2 = lambdatemp;
                    s2 = stemp;
                    lambdap2 = yptemp(1);
                end
                
                r = roots(hermite(s1,s2,lambda1,lambda2,lambdap1,lambdap2)-[zeros(1,3) 1]);
                hi = hmax;
                
                for i = 1:3
                    htemp = r(i)-s;
                    if ( isreal(htemp) && (htemp > 0) && (htemp < hi) )
                        hi = htemp;
                    end
                end
            end
        end
        
        % Determine appropriate step size
      
        hold = h;
        
        if endgame
            h = hi;
        else
            h = min([hmax 0.8*h*(tau(1)/(delta+1e-7))^pow]);
        end
        
        alpha = h/hold;
    
    else
        hold = h;
        h = h/4;
        alpha = h/hold;
    end
    
    % Adjust step size

    for i = 1:k, Zc((i*m+1):(i+1)*m) = alpha^i*Zc((i*m+1):(i+1)*m); end
    
end

if (numsol == 0)
    disp('Solution not found')
end