
/************************************************************************************/
/* CONDITIONAL LOGIT (McFADDEN) MLE ESTIMATOR                                       */
/*----------------------------------------------------------------------------------*/
/* This macro computes MLEs of the McFadden conditional logit model.                */
/*                                                                                  */
/* SYNTAX:                                                                          */
/*                                                                                  */
/* %clogit(indat,betaout,covbout,x,chooser,choice,d,pout=0,maxit=10,tol=1e-8);      */
/*                                                                                  */
/* Arguments are:                                                                   */
/*     indat      Name of input data file                                           */
/*     betaout    Name of output file for estimates of beta (posterior mode)        */
/*     covbout    Name of output file for estimates of covariance of beta (negative */
/*                inverse of hessian at posterior mode)                             */
/*     x          List of covariates                                                */
/*     chooser    Name of ID variable for the chooser (unit of observation)         */
/*     choice     Name of ID variable for the choice (outcome)                      */
/*     d          Name of dummy variable which indicates whether this choice was    */
/*                chosen by the chooser (1) or not (0).                             */
/*     pout       Name of file in which to store predicted probabilities (default is*/
/*                POUT=0 -- do not save predicted probabilities)                    */
/*     maxit      Maximum number of iterations (default is 10)                      */
/*     tol        Convergence criterion on g`*inv(H)*g (default is 1e-8)            */
/*----------------------------------------------------------------------------------*/
/* Notes:                                                                           */
/*     - INDAT should have one observation per CHOOSER per CHOICE that the chooser  */
/*       was at rsk to choose.                                                      */
/*     - POUT contains predicted probabilites as computed under the full model and  */
/*       and under the restricted (beta=0) model                                    */
/*----------------------------------------------------------------------------------*/
/* Author: Simon D. Woodcock, Cornell University, sdw9@cornell.edu                  */
/* Last Revision: March 14, 2002                                                    */
/************************************************************************************/

%MACRO clogit(indat,betaout,covbout,x,chooser,choice,d,pout=0,maxit=10,tol=1e-8);

	/*---- preliminaries ----*/

	%put ***************************************************;
	%put **** Invoked macro CLOGIT with arguments: ;
	%put **** INDAT: &indat.;
	%put **** BETAOUT: &betaout.;
	%put **** COVBOUT: &covbout.;
	%put **** X: &x.;
	%put **** CHOOSER: &chooser.;
	%put **** CHOICE: &choice.;
        %put **** D: &d.;
	%put **** POUT: &pout.;
	%put **** MAXIT: &maxit. ;
	%put **** TOL: &tol. ;
	%put ***************************************************;

	proc sort data=&indat. out=indat;
		by &chooser. &choice.;
	run;

	/*----- FIND POSTERIOR MODE OF p(beta|alpha,x,d) -----*/

	proc iml;

		/*---- define modules ----*/
		* module to compute values of probabilities (p) and log value of log posterior density;
		start logden(chooser,beta,x,d,nobs) global(p);
			xb=x*beta;
			e_xb=exp(xb);
			s_e_xb=e_xb;                                                /* correct start values for sum    */
			m=1;
			do n=1 to nobs-1;
				/* interior obs on chooser */
				if (chooser[n+1,]=chooser[n,]) then do;
					s_e_xb[n+1,]=s_e_xb[n,]+e_xb[n+1,];         /* accumulate sum for interior obs */
				end;
				/* last obs on chooser */
				else do;
					s_e_xb[m:n-1,]=s_e_xb[n,];                  /* distribute sum to chooser's rows */
					m=n+1;
				end;
			end;
			s_e_xb[m:nobs,]=s_e_xb[nobs,];                              /* fix last chooser                */

			p=e_xb/s_e_xb;                                              /* compute probabilities           */
			lnl=d`*(xb-log(s_e_xb));                                    /* evaluate log posterior          */
			return (lnl);
		finish;

		* module to compute gradient and hessian;
		start derivs(chooser,p,x,d,nobs,nvar) global(g,h);
			xbar=p#x;                                                   /* start values for xbar */
			m=1;
			do n=1 to nobs-1;
				if (chooser[n+1,]=chooser[n,]) then do;
					xbar[n+1,]=xbar[n,]+xbar[n+1,];             /* accumulate sum for interior obs */
				end;
				else do;
					do k=1 to nvar;
					    xbar[m:n-1,k]=xbar[n,k];                /* distribute sum to chooser's rows */
					end;
					m=n+1;
				end;
			end;
			do k=1 to nvar;
				xbar[m:nobs,k]=xbar[nobs,k];                        /* fix last chooser                */
			end;

			g=d`*(x-xbar);                                              /* evaluate gradient               */
			h=-(p#(x-xbar))`*(x-xbar);                                  /* evaluate hessian                */
		finish;
		/*---- end modules ----*/


		/*---- read in data and do preliminary calculations ----*/

		* read in data;
		use indat;
		read all var{&chooser. &choice. &d.};
		read all var{&x.} into x;		

		nobs=nrow(&d.);
		print '**** Number of obervations ****: ' nobs;

		* starting values for beta;
		nvar=ncol(x);
		beta=j(nvar,1,0);

		* compute value of log posterior and derivatives at starting values;
		lnl=logden(&chooser.,beta,x,&d.,nobs); 
		run derivs(&chooser.,p,x,&d.,nobs,nvar);

		iter=0;
		print 'Iteration: ' iter '    Value of log posterior density: ' lnl;

		/*---- do newton-raphson update ----*/
		do while (abs(g*inv(h)*g`)>&tol. & iter<=&maxit.);
			iter=iter+1;
			betaold=beta;
			beta=betaold-inv(h)*g`;
			lnl=logden(&chooser.,beta,x,&d.,nobs); 
			run derivs(&chooser.,p,x,&d.,nobs,nvar);
			print 'Iteration: ' iter '    Value of log posterior density: ' lnl;
			print 'Estimate of Beta: ' beta;
			print 'Gradient: ' g;
			conv=g*inv(h)*g`;
		end;

		/*---- output results ----*/
		* betas;
		xnames={&x.}`;
		print '**** Estimates of beta at posterior mode ****';
		print xnames beta;
		create &betaout. var{xnames beta};
		append;

		* covariance matrix;
		covb=-inv(h);
		print '**** Estimates of covariance matrix at posterior mode ****';
		print xnames covb;
		call symput("nvar",trim(left(char(nvar))));
		%macro splitcov;
			%do k=1 %to &nvar.;
				covb&k.=covb[,&k.];
			%end;
		%mend;
		%macro covnames;
			%do k=1 %to &nvar.;
				covb&k.
			%end;
		%mend;
		%splitcov;
		create &covbout. var{xnames %covnames};
		append;

		/*---- some basic specification testing ----*/
		* LR test -- beta=0;
		beta_r=j(nvar,1,0);
		lnl_r=logden(&chooser.,beta_r,x,&d.,nobs);
		%if &pout. ne 0 %then %do;
			p_beta0=p;
		%end;
		LR_beta0=-2*(lnl_r-lnl);
		pval_beta0=1-probchi(LR_beta0,nvar);
		print '**** LR test statistic for H0:beta=0 ****: ' LR_beta0;
		print '**** LR_beta0 is distributed Chi-Square with ' nvar ' degrees of freedom ****';
		print '**** P-value for LR_beta0 ****: ' pval_beta0;
		print '';

		/*---- store predicted probabilities if requested ----*/
		%if &pout. ne 0 %then %do;
			create &pout. var{&chooser. &choice. p p_beta0};
			append;
		%end;
	quit;
%MEND;
