#define _GNU_SOURCE
#include <stdio.h>
#include <unistd.h>
#include <sys/syscall.h>
#include <sys/types.h>
#include <sched.h>
#include <pthread.h>
#include <time.h>
#include <stdlib.h>
#include <strings.h>

#define NUM_CORES 60
#define NUM_THREADS_PER_CORE 4
#define NUM_THREADS (NUM_CORES * NUM_THREADS_PER_CORE)
 
#define NUM_BARRIERS 2
pthread_barrier_t barrier[NUM_BARRIERS];

//pin thread to core
void set_cpu(int cpu_no)
{
	cpu_set_t mask;
	CPU_ZERO(&mask);
	CPU_SET(cpu_no, &mask);
	sched_setaffinity(syscall(SYS_gettid), sizeof(cpu_set_t), &mask);
}

//get elapsed time
long time_diff(struct timespec start, struct timespec end)
{
  struct timespec temp;
  if ((end.tv_nsec-start.tv_nsec)<0) {
    temp.tv_sec = end.tv_sec-start.tv_sec-1;
    temp.tv_nsec = 1000000000+end.tv_nsec-start.tv_nsec;
  } else {
    temp.tv_sec = end.tv_sec-start.tv_sec;
    temp.tv_nsec = end.tv_nsec-start.tv_nsec;
  }
  return temp.tv_sec * 1000000000 + temp.tv_nsec;
}

struct thread_arg_t {
	int tid;
};

/* 2nd-level TLB can work for 128MB data, we guarantee 2nd-level
** TLB hits. And since the total memory on KNC is limited, 4GB in
** total, usually 2.2GB free memory available. So we limited the
** array size per core to be 4*6=24MB. Thus it doesn't exceed 128MB,
** always 2nd-level TLB hit; and total size is 24*60=1440MB, i.e.,
** 1.4GB, not exceeding free memory size*/
#define ARRAY_SIZE 6*1024*1024/NUM_THREADS_PER_CORE
volatile unsigned int *A;

//using system-wide timer micetc. time_begin, time_end struct
struct timespec time_begin[NUM_THREADS], time_end[NUM_THREADS];
long long elapsed_time[NUM_THREADS];

//test bandwidth using L2 prefetch, random read, streaming read,
//random write, and streaming write
//this is the routine for each thread
void test_bandwidth_p(int tid)
{
    int i;  
    //each thread starts from A[tid*ARRAY_SIZE]
    volatile unsigned int *A_ti = &A[tid*ARRAY_SIZE];
    for(i = 0; i < ARRAY_SIZE; i=i+16){
      _mm_clevict((void *)&A_ti[i], 1);
      _mm_clevict((void *)&A_ti[i], 2);
    }
    __asm cpuid

    pthread_barrier_wait(&barrier[0]);

    //get system-wide begin time
    clock_gettime(CLOCK_REALTIME, &time_begin[tid]);

    for(i = 0; i < ARRAY_SIZE/2048; i++){
        //each thread starts from A+tid*ARRAY_SIZE
        A_ti = &(A[tid*ARRAY_SIZE+i*2048]);

	//unroll two pages to minimize control overhead
	_mm_prefetch((void *)A_ti, 2);
	_mm_prefetch((void *)(A_ti+16), 2);
	_mm_prefetch((void *)(A_ti+16*2), 2);
	_mm_prefetch((void *)(A_ti+16*3), 2);
	_mm_prefetch((void *)(A_ti+16*4), 2);
	_mm_prefetch((void *)(A_ti+16*5), 2);
	_mm_prefetch((void *)(A_ti+16*6), 2);
	_mm_prefetch((void *)(A_ti+16*7), 2);

	_mm_prefetch((void *)(A_ti+16*8), 2);
	_mm_prefetch((void *)(A_ti+16*9), 2);
	_mm_prefetch((void *)(A_ti+16*10), 2);
	_mm_prefetch((void *)(A_ti+16*11), 2);
	_mm_prefetch((void *)(A_ti+16*12), 2);
	_mm_prefetch((void *)(A_ti+16*13), 2);
	_mm_prefetch((void *)(A_ti+16*14), 2);
	_mm_prefetch((void *)(A_ti+16*15), 2);

	_mm_prefetch((void *)(A_ti+16*16), 2);
	_mm_prefetch((void *)(A_ti+16*17), 2);
	_mm_prefetch((void *)(A_ti+16*18), 2);
	_mm_prefetch((void *)(A_ti+16*19), 2);
	_mm_prefetch((void *)(A_ti+16*20), 2);
	_mm_prefetch((void *)(A_ti+16*21), 2);
	_mm_prefetch((void *)(A_ti+16*22), 2);
	_mm_prefetch((void *)(A_ti+16*23), 2);

	_mm_prefetch((void *)(A_ti+16*24), 2);
	_mm_prefetch((void *)(A_ti+16*25), 2);
	_mm_prefetch((void *)(A_ti+16*26), 2);
	_mm_prefetch((void *)(A_ti+16*27), 2);
	_mm_prefetch((void *)(A_ti+16*28), 2);
	_mm_prefetch((void *)(A_ti+16*29), 2);
	_mm_prefetch((void *)(A_ti+16*30), 2);
	_mm_prefetch((void *)(A_ti+16*31), 2);

	_mm_prefetch((void *)(A_ti+16*32), 2);
	_mm_prefetch((void *)(A_ti+16*33), 2);
	_mm_prefetch((void *)(A_ti+16*34), 2);
	_mm_prefetch((void *)(A_ti+16*35), 2);
	_mm_prefetch((void *)(A_ti+16*36), 2);
	_mm_prefetch((void *)(A_ti+16*37), 2);
	_mm_prefetch((void *)(A_ti+16*38), 2);
	_mm_prefetch((void *)(A_ti+16*39), 2);

	_mm_prefetch((void *)(A_ti+16*40), 2);
	_mm_prefetch((void *)(A_ti+16*41), 2);
	_mm_prefetch((void *)(A_ti+16*42), 2);
	_mm_prefetch((void *)(A_ti+16*43), 2);
	_mm_prefetch((void *)(A_ti+16*44), 2);
	_mm_prefetch((void *)(A_ti+16*45), 2);
	_mm_prefetch((void *)(A_ti+16*46), 2);
	_mm_prefetch((void *)(A_ti+16*47), 2);

	_mm_prefetch((void *)(A_ti+16*48), 2);
	_mm_prefetch((void *)(A_ti+16*49), 2);
	_mm_prefetch((void *)(A_ti+16*50), 2);
	_mm_prefetch((void *)(A_ti+16*51), 2);
	_mm_prefetch((void *)(A_ti+16*52), 2);
	_mm_prefetch((void *)(A_ti+16*53), 2);
	_mm_prefetch((void *)(A_ti+16*54), 2);
	_mm_prefetch((void *)(A_ti+16*55), 2);

	_mm_prefetch((void *)(A_ti+16*56), 2);
	_mm_prefetch((void *)(A_ti+16*57), 2);
	_mm_prefetch((void *)(A_ti+16*58), 2);
	_mm_prefetch((void *)(A_ti+16*59), 2);
	_mm_prefetch((void *)(A_ti+16*60), 2);
	_mm_prefetch((void *)(A_ti+16*61), 2);
	_mm_prefetch((void *)(A_ti+16*62), 2);
	_mm_prefetch((void *)(A_ti+16*63), 2);

	_mm_prefetch((void *)(A_ti+16*64), 2);
	_mm_prefetch((void *)(A_ti+16*65), 2);
	_mm_prefetch((void *)(A_ti+16*66), 2);
	_mm_prefetch((void *)(A_ti+16*67), 2);
	_mm_prefetch((void *)(A_ti+16*68), 2);
	_mm_prefetch((void *)(A_ti+16*69), 2);
	_mm_prefetch((void *)(A_ti+16*70), 2);
	_mm_prefetch((void *)(A_ti+16*71), 2);

	_mm_prefetch((void *)(A_ti+16*72), 2);
	_mm_prefetch((void *)(A_ti+16*73), 2);
	_mm_prefetch((void *)(A_ti+16*74), 2);
	_mm_prefetch((void *)(A_ti+16*75), 2);
	_mm_prefetch((void *)(A_ti+16*76), 2);
	_mm_prefetch((void *)(A_ti+16*77), 2);
	_mm_prefetch((void *)(A_ti+16*78), 2);
	_mm_prefetch((void *)(A_ti+16*79), 2);

	_mm_prefetch((void *)(A_ti+16*80), 2);
	_mm_prefetch((void *)(A_ti+16*81), 2);
	_mm_prefetch((void *)(A_ti+16*82), 2);
	_mm_prefetch((void *)(A_ti+16*83), 2);
	_mm_prefetch((void *)(A_ti+16*84), 2);
	_mm_prefetch((void *)(A_ti+16*85), 2);
	_mm_prefetch((void *)(A_ti+16*86), 2);
	_mm_prefetch((void *)(A_ti+16*87), 2);

	_mm_prefetch((void *)(A_ti+16*88), 2);
	_mm_prefetch((void *)(A_ti+16*89), 2);
	_mm_prefetch((void *)(A_ti+16*90), 2);
	_mm_prefetch((void *)(A_ti+16*91), 2);
	_mm_prefetch((void *)(A_ti+16*92), 2);
	_mm_prefetch((void *)(A_ti+16*93), 2);
	_mm_prefetch((void *)(A_ti+16*94), 2);
	_mm_prefetch((void *)(A_ti+16*95), 2);

	_mm_prefetch((void *)(A_ti+16*96), 2);
	_mm_prefetch((void *)(A_ti+16*97), 2);
	_mm_prefetch((void *)(A_ti+16*98), 2);
	_mm_prefetch((void *)(A_ti+16*99), 2);
	_mm_prefetch((void *)(A_ti+16*100), 2);
	_mm_prefetch((void *)(A_ti+16*101), 2);
	_mm_prefetch((void *)(A_ti+16*102), 2);
	_mm_prefetch((void *)(A_ti+16*103), 2);

	_mm_prefetch((void *)(A_ti+16*104), 2);
	_mm_prefetch((void *)(A_ti+16*105), 2);
	_mm_prefetch((void *)(A_ti+16*106), 2);
	_mm_prefetch((void *)(A_ti+16*107), 2);
	_mm_prefetch((void *)(A_ti+16*108), 2);
	_mm_prefetch((void *)(A_ti+16*109), 2);
	_mm_prefetch((void *)(A_ti+16*110), 2);
	_mm_prefetch((void *)(A_ti+16*111), 2);

	_mm_prefetch((void *)(A_ti+16*112), 2);
	_mm_prefetch((void *)(A_ti+16*113), 2);
	_mm_prefetch((void *)(A_ti+16*114), 2);
	_mm_prefetch((void *)(A_ti+16*115), 2);
	_mm_prefetch((void *)(A_ti+16*116), 2);
	_mm_prefetch((void *)(A_ti+16*117), 2);
	_mm_prefetch((void *)(A_ti+16*118), 2);
	_mm_prefetch((void *)(A_ti+16*119), 2);

	_mm_prefetch((void *)(A_ti+16*120), 2);
	_mm_prefetch((void *)(A_ti+16*121), 2);
	_mm_prefetch((void *)(A_ti+16*122), 2);
	_mm_prefetch((void *)(A_ti+16*123), 2);
	_mm_prefetch((void *)(A_ti+16*124), 2);
	_mm_prefetch((void *)(A_ti+16*125), 2);
	_mm_prefetch((void *)(A_ti+16*126), 2);
	_mm_prefetch((void *)(A_ti+16*127), 2);
    }

    //get system-wide end time
    clock_gettime(CLOCK_REALTIME, &time_end[tid]);

    pthread_barrier_wait(&barrier[1]);

    elapsed_time[tid] = time_diff(time_begin[tid], time_end[tid]);

    //printf("thread %d time_begin: %llu\t time_end: %llu\t elapsed_time: %llu\n", tid, time_begin[tid].tv_sec * 1000000000 + time_begin[tid].tv_nsec, time_end[tid].tv_sec * 1000000000 + time_end[tid].tv_nsec, elapsed_time[tid]);

}

void* bandwidth_thread(void* thread_arg)
{
	struct thread_arg_t *my_thread_arg = (struct thread_arg_t *)thread_arg;
	int tid = my_thread_arg->tid;

	//pin the thread to the core, KNC starts from core 1
	set_cpu(tid*4/NUM_THREADS_PER_CORE+1);

	//initialize the page first, so that no 2nd-level TLB miss and all physical pages allocated
	volatile unsigned int *A_ti = &A[tid*ARRAY_SIZE];
	register unsigned int temp = 0;
	int i = 0;
	for(i = 0; i < ARRAY_SIZE; i=i+1024)
	  //A_ti[i]=i;
	  temp += A_ti[i];

	//test it 5 times to get stable results
	//test_bandwidth_p(tid);
	//test_bandwidth_p(tid);
	//test_bandwidth_p(tid);
	//test_bandwidth_p(tid);
	test_bandwidth_p(tid);
}

int main()
{
        //allocate the array space for the thread
	A = (volatile unsigned int *)malloc(ARRAY_SIZE*NUM_THREADS*sizeof(volatile unsigned int));
	bzero((void *)A, ARRAY_SIZE*NUM_THREADS*sizeof(volatile unsigned int));

	//initialize the page first, so that all physical pages allocated
	int i = 0;
	for(i = 0; i < ARRAY_SIZE*NUM_THREADS; i=i+1024)
	  A[i]=i;
	
	pthread_t thread_id[NUM_THREADS];
	struct thread_arg_t thread_arg[NUM_THREADS];
	int ti;
	for(ti = 0; ti < NUM_BARRIERS; ++ti)
	{
		pthread_barrier_init(&barrier[ti], NULL, NUM_THREADS);
	}

	for(ti = 0; ti < NUM_THREADS; ++ti)
	{
		thread_arg[ti].tid = ti;
		pthread_create( &thread_id[ti], NULL, bandwidth_thread, (void *)&thread_arg[ti]);
	}

	for(ti = 0; ti < NUM_THREADS; ++ti)   
	{      
		pthread_join(thread_id[ti], NULL);    
	}

	for(ti = 0; ti < NUM_BARRIERS; ++ti)   
	{      
		pthread_barrier_destroy(&barrier[ti]);
	}

	//free the memory
	free((void *)A);

	//get earliest begin time and latest end time of whole system
	long long min_time_begin = time_begin[0].tv_sec * 1000000000 + time_begin[0].tv_nsec;
	long long max_time_end = time_end[0].tv_sec * 1000000000 + time_end[0].tv_nsec;

	for(ti = 1; ti < NUM_THREADS; ++ti)   
	{      
	  if( (min_time_begin - (time_begin[ti].tv_sec * 1000000000 + time_begin[ti].tv_nsec)) > 0 )
	    min_time_begin = time_begin[ti].tv_sec * 1000000000 + time_begin[ti].tv_nsec;

	  if( (max_time_end - (time_end[ti].tv_sec * 1000000000 + time_end[ti].tv_nsec)) < 0  )
	    max_time_end = time_end[ti].tv_sec * 1000000000 + time_end[ti].tv_nsec;

	}

	//calculate total elapsed time and bandwidth
	/*long long total_elapsed_time = max_time_end - min_time_begin;
	double bandwidth = (double)sizeof(unsigned int)*ARRAY_SIZE*NUM_THREADS/total_elapsed_time;
	printf("%.3lf\t", bandwidth);
	//printf("%llu\n", total_elapsed_time);//*/

	//print the beign and end time
	for(ti = 0; ti < NUM_THREADS; ++ti){
	  printf("%llu\t", time_begin[ti].tv_sec * 1000000000 + time_begin[ti].tv_nsec);
	}
	printf("\n");

	/*for(ti = 0; ti < NUM_THREADS; ++ti){
	  printf("%llu\t", time_end[ti].tv_sec * 1000000000 + time_end[ti].tv_nsec);
	}
	printf("\n");//*/

	return 0;
}	
