%  -----------------------------------------------------------
%        Simulate data from Jolicoeur model for males.
%  -----------------------------------------------------------

Windows:

addpath('c:\matlab\fdaM.dir')
addpath('c:\matlab\growth.dir')

Unix

addpath('..')

load FelsmJUL00   %  load previous results from .mat file

%  -----------------------------------------------------------
%               Input coefficients.
%  -----------------------------------------------------------

%  157 males

fid      = fopen('Felsjpam.par','rt');
tempvec  = fscanf(fid,'%f');
ncaseJ   = length(tempvec)/21;
Felsjpam = reshape(tempvec, [21, ncaseJ]);
clear tempvec;

parmatm = Felsjpam(7:14,:)';

% ------------ histograms of parameter values  -----------------

hist(parmatm(:,1))
title('Parameter a')

hist(parmatm(:,8))
title('Parameter e')

for j=1:3
   subplot(1,3,j)
   hist(parmatm(:,1+j))
   axis('square')
   title(['Parameter b',num2str(j)])
end

for j=1:3
   subplot(1,3,j)
   hist(parmatm(:,4+j))
   axis('square')
   title(['Parameter c',num2str(j)])
end

%  -----------------------------------------------------------
%      set up a standard error functional data object   
%  -----------------------------------------------------------

%  Replace bad estimates by reasonable values

Varstore( 1) = 0.75^2;
Varstore(21) = 0.5^2;

serng    = [0,22];
sebasis  = create_bspline_basis(serng, 14, 4);

index = 1:length(Varstore);
sefd     = data2fd(sqrt(Varstore(index)), [0.5:1:21.5]', sebasis);
sesmthfd = smooth(sefd, 1e0, 2);

%  plot se object and actual square roots of variances
plot(sesmthfd);
axis([0,22,0,1.2])
hold on
plot(0.5:1:21.5, sqrt(Varstore(index)), 'o')
hold off

save sesmthfd sesmthfd
load sesmthfd

%  -----------------------------------------------------------
%  ------  Analyze and display data for a single case  -------
%  -----------------------------------------------------------

%  ---------  set parameters for function monotone  ----------------

norder  = 5;               %  order of spline
Lfd     = 3;               %  penalize curvature of acceleration
conv    = 1e-3;            %  convergence criterion
iterlim = 10;              %  max. no. iterations
dbglev  = 2;               %  level of output per iteration (full here)
lambda  = 10^(-1);         %  smoothing parameter

%  case number
icaseJ = floor(rand(1)*ncaseJ)+1;
icase  = floor(rand(1)*ncase)+1;
[icase, icaseJ]
nobs = FelsmJUL00(5,           icase);
age  = FelsmJUL00(6:(  5+nobs),icase);
[hgt0, vel0, acc0] = jolifn(age, parmatm(icaseJ,:)');
stderr = eval(sesmthfd, age);
hgt = hgt0 + randn(nobs,1).*stderr;
zmat = ones(nobs,1);   
wt   = 1./stderr.^2;
wt = wt/mean(wt);

%  ----------  set up B-spline basis for W  -------------------

%  put knots at data points
knots = age';
if age(1) > 0, knots = [0,knots]; end
nknots = length(knots);
knot0 = knots(1);
knotQ = knots(nknots);

nbasis = nknots + norder - 2;
rng    = [knot0,knotQ];
wbasis = create_bspline_basis(rng, nbasis, norder, knots);
cvec0 = zeros(nbasis,1);  
Wfd0  = fd(cvec0, wbasis);
active    = ones(nbasis,1);  %  specify active parameteres
%active(1) = 0;
%lambda = 1;
%  -------------  smooth the data  ------------------------

Wfd0 = Wfd;
%lambda = lambda/10;

[Wfd, beta, Fstr, iternum, iterhist] = ...
     monotone(age, hgt, wt, Wfd0, zmat, Lfd, lambda,  ...
              conv, iterlim, active, dbglev);
  
hgthat = beta(1) + beta(2).*monfn(age, Wfd);
velhat =           beta(2)*eval_mon(age, Wfd, 1);
acchat =           beta(2)*eval_mon(age, Wfd, 2);
RMSE   = sqrt(mean((hgthat - hgt).^2));
index = age >= 4 & age <= 18;
hgtRMSE = sqrt(mean((hgthat(index)-hgt0(index)).^2));
velRMSE = sqrt(mean((velhat(index)-vel0(index)).^2));
accRMSE = sqrt(mean((acchat(index)-acc0(index)).^2));

%  -------------  display the results  ------------------------

[log10(lambda), RMSE, hgtRMSE, velRMSE, accRMSE]

agefine = linspace(rng(1),rng(2),151)';
%  display velocity

Dhgt    = beta(2)*eval_mon(agefine, Wfd, 1);
subplot(1,2,1)
plot(agefine, Dhgt, '-', age, vel0, 'o')
axis([rng(1), rng(2), 0, 20])
axis square
xlabel('\fontsize{12} Years')
ylabel('\fontsize{12} Velocity (cm/year)')
title(['\fontsize{12} Records ',num2str(icaseJ),', ',num2str(icase)])
for i=1:length(knots)
   line([knots(i),knots(i)], [0,1])
end
   
%  display acceleration

D2hgt = beta(2)*eval_mon(agefine, Wfd, 2);
subplot(1,2,2)
plot(agefine, D2hgt, '-', age, acc0, 'o', agefine, 0, '--')
axis([rng(1), rng(2), -15, 15])
axis square
xlabel('\fontsize{12} Years')
ylabel('\fontsize{12} Acceleration (cm/year/year)')

print -dps2 velacc75.ps

%  display data and fit 

hgthat     = beta(1) + beta(2).*monfn(age, Wfd);

subplot(1,1,1)
plot(age, hgt, 'o', age, hgthat, '-')
axis([rng(1), rng(2), 50, 200])
xlabel('\fontsize{16} Years')
ylabel('\fontsize{16} Height (cm)')
title(['\fontsize{16} Height for Record ',num2str(icase)])
legend('Observed height', 'Estimated height')

% display residuals

res = hgt - hgthat;
RMSE = sqrt(mean(res.^2));

plot(age, res, 'o', ...
     age, res, 'b-', ...
     [knot0,knotQ], [0,0], 'g--',  ...
     [knot0,knotQ], [-2*RMSE,-2*RMSE], 'r--', ...
     [knot0,knotQ], [ 2*RMSE, 2*RMSE], 'r--')
xlabel('\fontsize{16} Years')
ylabel('\fontsize{16} Height (cm)')
title(['\fontsize{16} Residuals for Record ',num2str(icase)])

%  --------------------------------------------------------------------
%                Now run through a set of simulations
%  --------------------------------------------------------------------

%  ---------  set parameters for function monotone  ----------------

lambdavec = 10.^(0:-0.5:-2);
lambdavec = 10^(-.5);
nlambda   = length(lambdavec);
dbglev    = 0;
nsim      = 100;

RMSEstore = zeros( 4, nlambda, nsim);
indxstore = zeros( 2, nsim);
iterstore = zeros(nlambda, nsim);
nobsstore = zeros(nsim);
resstore  = zeros( 3, 100, nlambda, nsim);

for isim=1:nsim
  icaseJ = floor(rand(1)*ncaseJ)+1;
  icase  = floor(rand(1)*ncase)+1;
  indxstore(:,isim) = [icaseJ; icase];
  nobs   = FelsmJUL00(  5,           icase);
  age    = FelsmJUL00(  6:(  5+nobs),icase);
  [hgt0, vel0, acc0] = jolifn(age, parmatm(icaseJ,:)');
  stderr = eval(sesmthfd, age);
  hgt    = hgt0 + randn(nobs,1).*stderr;
  knots  = age';
  if age(1) > 0, knots = [0, knots]; end
  nknots = length(knots);
  zmat   = ones(nobs,1);   %  covariate matrix
  rng    = [knot0,knotQ];
  nbasis = nknots + norder - 2;
  active    = ones(nbasis,1);  
  wbasis = create_bspline_basis([0,knots(nknots)], nbasis, norder, knots);
  wgt    = 1./stderr.^2;
  ageind = age >= 4 & age <= 18;
  for ilambda=1:nlambda 
    lambda = lambdavec(ilambda);
    if ilambda == 1
       %conv = 1e-2;
       Wfd0 = fd(zeros(nbasis,1), wbasis);
    else
       Wfd0 = Wfd;
       if conv > 1e-3, conv = 1e-3; end
    end
    [Wfd, beta, Fstr, iternum, iterhist] = ...
       monotone(age, hgt, wgt, Wfd0, zmat, Lfd, lambda, ...
                conv, iterlim, active, dbglev);
    hgthat = beta(1) + beta(2).*eval_mon(age, Wfd);
    velhat =           beta(2).*eval_mon(age, Wfd, 1);
    acchat =           beta(2).*eval_mon(age, Wfd, 2);
    resstore(1,1:nobs,ilambda,isim) = hgthat - hgt0;
    resstore(2,1:nobs,ilambda,isim) = velhat - vel0;
    resstore(3,1:nobs,ilambda,isim) = acchat - acc0;
    RMSEstore(1,ilambda,isim) = sqrt(mean((hgthat - hgt).^2));
    RMSEstore(2,ilambda,isim) = sqrt(mean((hgthat(ageind)-hgt0(ageind)).^2));
    RMSEstore(3,ilambda,isim) = sqrt(mean((velhat(ageind)-vel0(ageind)).^2));
    RMSEstore(4,ilambda,isim) = sqrt(mean((acchat(ageind)-acc0(ageind)).^2));
    iterstore(ilambda,isim)   = iternum;
    nobsstore(isim) = nobs;
    fprintf('%4.f %3.f %5.f %12.5f %10.4f %10.4f %10.4f %10.4f\n', ...
       [isim, ilambda, iternum, Fstr.f, RMSEstore(:,ilambda,isim)'])
  end
end


stemean = mean(squeeze(RMSEstore(1,:,:))');
hgtmean = mean(squeeze(RMSEstore(2,:,:))');
velmean = mean(squeeze(RMSEstore(3,:,:))');
accmean = mean(squeeze(RMSEstore(4,:,:))');

%  -------------------------------------------------------------
%  Display age, height, and velocity of acceleration crossings  ---
%  -------------------------------------------------------------

index = 1:ncase;

norder    = 5;
lambdavec = 10.^(-3:-1:-6);
nlambda   = length(lambdavec);
nfine     = 201;
firstind  = 1:(nfine-1);
lastind   = 2:nfine;

%  display data and fit

subplot(1,1,1)
for icase = index
  ID     = FelsmJUL00(2,             icase);
  birth  = FelsmJUL00(3,             icase);
  nobs   = FelsmJUL00(5,             icase);
  age    = FelsmJUL00(  6:(  5+nobs),icase);
  hgt    = FelsmJUL00( 76:( 75+nobs),icase);
  rec    = FelsmJUL00(146:(145+nobs),icase); 
  nbasis = nbasstore(icase);
  nknots = nbasis - norder + 2;
  knots  = knotstore(1:nknots,icase)';
  rng    = [min(knots),max(knots)];
  wbasis = create_bspline_basis(rng, nbasis, norder, knots);
  cvec   =  cvecstore(1:nbasis,ilambda,icase);
  Wfd    = fd(cvec, wbasis);
  beta   = betastore(:,ilambda,icase);
  hgthat = beta(1) + beta(2).*rec + beta(3).*monfn(age, Wfd);
  plot(age, hgt, 'o', age, hgthat, '-')
  axis([rng(1), rng(2), 50, 200])
  xlabel('\fontsize{16} Years')
  ylabel('\fontsize{16} Height (cm)')
  title(['\fontsize{16} Male ',num2str(icase)])
  pause
end

% display residuals

subplot(1,1,1)
for icase = index
  ID     = FelsmJUL00(2,             icase);
  birth  = FelsmJUL00(3,             icase);
  nobs   = FelsmJUL00(5,             icase);
  age    = FelsmJUL00(  6:(  5+nobs),icase);
  hgt    = FelsmJUL00( 76:( 75+nobs),icase);
  rec    = FelsmJUL00(146:(145+nobs),icase); 
  nbasis = nbasstore(icase);
  nknots = nbasis - norder + 2;
  knots  = knotstore(1:nknots,icase)';
  rng    = [min(knots),max(knots)];
  wbasis = create_bspline_basis(rng, nbasis, norder, knots);
  cvec   =  cvecstore(1:nbasis,ilambda,icase);
  Wfd    = fd(cvec, wbasis);
  beta   = betastore(:,ilambda,icase);
  hgthat = beta(1) + beta(2).*rec + beta(3).*monfn(age, Wfd);
  res    = hgt - hgthat;
  RMSE   = RMSEstore(ilambda,icase);
  plot(age, res, 'o', age, res, 'b-', ...
       [knot0,knotQ], [0,0], 'g--', ...
       [knot0,knotQ], [-2*RMSE,-2*RMSE], 'r--', ...
       [knot0,knotQ], [ 2*RMSE, 2*RMSE], 'r--')
  xlabel('\fontsize{16} Years')
  ylabel('\fontsize{16} Height (cm)')
  title(['\fontsize{16} Residuals for Male ',num2str(icase)])
  pause
end

%  display velocities with peaks

for icase = index
  ID      = FelsmJUL00(2,           icase);
  birth   = FelsmJUL00(3,           icase);
  nobs    = FelsmJUL00(5           ,icase);
  age     = FelsmJUL00(6:(  5+nobs),icase);
  nbasis  = nbasstore(icase);
  nknots  = nbasis - norder + 2;
  knots   = knotstore(1:nknots,icase)';
  rng     = [min(knots),max(knots)];
  agefine = linspace(max([0,rng(1)]),min([rng(2),18]),nfine)';
  deltage = (agefine(2) - agefine(1))/2;
  wbasis  = create_bspline_basis(rng, nbasis, norder, knots);
  cvec    =  cvecstore(1:nbasis,ilambda,icase);
  Wfd     = fd(cvec, wbasis);
  beta    = betastore(:,ilambda,icase);
  Dhgthat = beta(3)*eval_mon(agefine, Wfd, 1);
  Dhgt    = interp1q(agefine,Dhgthat,age);
  plot(agefine, Dhgthat, '-', [2,18], [0,0], '--')
  axis([2, 18, 0, 20])
  xlabel('\fontsize{16} Years')
  ylabel('\fontsize{16} Velocity (cm/year)')
  title(['\fontsize{16} Male Record ',num2str(icase), ...
         '  ID = ', num2str(ID), ...
         '  Birth = ' num2str(birth)])
  hold on
  D2hgthat = beta(3)*eval_mon(agefine, Wfd, 2);
  crossval = D2hgthat(lastind).*D2hgthat(firstind);
  crossind = crossval < 0;
  crossage = agefine(crossind);
  ncross   = length(crossage);
  for i=1:ncross
    plot((crossage(i)+deltage).*ones(2,1),[0,20], 'g--')
  end
  plot(age, Dhgt, 'b+')
  hold off
  pause
end

%  display accelerations with crossings

for icase = index
  ID     = FelsmJUL00(2,           icase);
  birth  = FelsmJUL00(3,           icase);
  nobs   = FelsmJUL00(5           ,icase);
  age    = FelsmJUL00(6:(  5+nobs),icase);
  nbasis = nbasstore(icase);
  nknots = nbasis - norder + 2;
  knots  = knotstore(1:nknots,icase)';
  rng    = [min(knots),max(knots)];
  agefine = linspace(max([0,rng(1)]),min([rng(2),18]),nfine)';
  deltage = (agefine(2) - agefine(1))/2;
  wbasis = create_bspline_basis(rng, nbasis, norder, knots);
  cvec   =  cvecstore(1:nbasis,ilambda,icase);
  Wfd   = fd(cvec, wbasis);
  beta  = betastore(:,ilambda,icase);
  D2hgthat = beta(3)*eval_mon(agefine, Wfd, 2);
  D2hgt = interp1q(agefine,D2hgthat,age);
  plot(agefine, D2hgthat, '-', [2,18], [0,0], '--')
  axis([2, 18, -15, 15])
  xlabel('\fontsize{16} Years')
  ylabel('\fontsize{16} Acceleration (cm/year/year)')
  title(['\fontsize{16} Male Record ',num2str(icase), ...
         '  ID = ',    num2str(ID), ...
         '  Birth = ', num2str(birth)])
  hold on
  crossval = D2hgthat(lastind).*D2hgthat(firstind);
  crossind = crossval < 0;
  crossage = agefine(crossind);
  ncross = length(crossage);
  for i=1:ncross
    plot((crossage(i)+deltage).*ones(2,1),[-15,15], 'g--')
  end
  plot(age, D2hgt, 'b+')
  hold off
  pause
end

%  output crossings to a file

fid = fopen('felsmJUL00.txt', 'wt');

for icase = index
  ID     = FelsmJUL00(2,           icase);
  birth  = FelsmJUL00(3,           icase);
  nobs   = FelsmJUL00(5           ,icase);
  age    = FelsmJUL00(6:(  5+nobs),icase);
  nbasis = nbasstore(icase);
  nknots = nbasis - norder + 2;
  knots  = knotstore(1:nknots,icase)';
  rng    = [min(knots),max(knots)];
  agefine = linspace(max([0,rng(1)]),min([rng(2),18]),nfine)';
  deltage = (agefine(2) - agefine(1))/2;
  wbasis = create_bspline_basis(rng, nbasis, norder, knots);
  cvec   =  cvecstore(1:nbasis,nlambda,icase);
  Wfd   = fd(cvec, wbasis);
  beta  = betastore(:,ilambda,icase);
  hgthat   = beta(1) + beta(3)*eval_mon(agefine, Wfd);
  Dhgthat  =           beta(3)*eval_mon(agefine, Wfd, 1);
  D2hgthat =           beta(3)*eval_mon(agefine, Wfd, 2);
  crossval = D2hgthat(lastind).*D2hgthat(firstind);
  crossind = crossval < 0;
  crossage = agefine(crossind);
  crosshgt = hgthat(crossind);
  crossvel = Dhgthat(crossind);
  ncross = length(crossage);
  fprintf(fid, '%4.0f %4.0f\n', ID, 1);
  fprintf(fid, '%6.2f', crossage);
  fprintf(fid, '\n');
  fprintf(fid, '%6.1f', crosshgt);
  fprintf(fid, '\n');
  fprintf(fid, '%6.1f', crossvel);
  fprintf(fid, '\n');
end

%  output residuals to a file

%  open file for output  
fid = fopen('felsmJUL00.res', 'wt');

index = 1:ncase;
for icase = index
  ID     = FelsmJUL00(  2,           icase);
  nobs   = FelsmJUL00(  5,           icase);
  age    = FelsmJUL00(  6:(  5+nobs),icase);
  hgt    = FelsmJUL00( 76:( 75+nobs),icase);
  rec    = FelsmJUL00(146:(145+nobs),icase);
  recind = (rec == 1);
  nbasis = nbasstore(icase);
  nknots = nbasis - norder + 2;
  knots  = knotstore(1:nknots,icase)';
  rng    = [min(knots),max(knots)];
  wbasis = create_bspline_basis(rng, nbasis, norder, knots);
  cvec   = cvecstore(1:nbasis,nlambda,icase);
  Wfd    = fd(cvec, wbasis);
  beta   = betastore(:,nlambda,icase);
  hgthat = beta(1) - beta(2).*rec + beta(3).*eval_mon(age, Wfd);
  res = hgt - hgthat;
  fprintf(fid, '%4.0f %4.0f\n', ID,  1);
  fprintf(fid, '%6.2f', age);
  fprintf(fid, '\n');
  fprintf(fid, '%6.2f', res);
  fprintf(fid, '\n');
end
