%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%
%%% plotinvr.m:
%%%
%%% Compute the Radon transform of an image, and 
%%% then invert the trasform and display the linear
%%% features on top of the original image.
%%%
%%% Date:   June 24, 1998.
%%% Author: John Stockie.
%%% 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function ret = plotinvr( I, R, th, xp, thresh, invopt, maxlines )

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% MAIN OPTIONS:
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Maximum lines to plot:
% maxlines = 50;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Scaling of inverse Radon transform:
%%%   invopt = 0 : 1.0 (no scaling)
%%%          = 1 : nx * ny
%%%          = 2 : mean(R)
%%%          = 3 : max(R)
%%%          = 4 : nx * ny * mean(R)
%%%          = 5 : nx
% invopt = 4;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

fprintf( 1, '\nPerforming inverse transform...\n' );

%%% First place the image in the center of a square.
%%% This is needed because the inverse transform 
%%% plots are easiest to do on a square image.

[nx,ny] = size(I);
tI = zeros( ny, ny );
nx0 = round(ny/2) - round(nx/2) + 1;
tI(nx0:nx0+nx-1,1:ny) = I;

%%% Rescale the Radon transform based on 'invopt'.

% threshold of 4.5 works for original 512x512 image
nfac = nx*ny; meanfac = mean(R(:)); maxfac = max(R(:));
if     invopt == 0 fac = 1.0;
elseif invopt == 1 fac = nfac;
elseif invopt == 2 fac = meanfac;
elseif invopt == 3 fac = maxfac;
elseif invopt == 4 fac = nfac * meanfac;
else               fac = ny;
end;
R2 = R / fac;

fprintf( 1, '  threshold = %e\n', thresh );
fprintf( 1, '     nx*ny   = %e\n     mean|R| = %e\n     max|R|  = %e\n', ...
    nfac, meanfac, maxfac );
fprintf( 1, '  scaling R by = %e (invopt=%1d)\n', fac, invopt );
fprintf( 1, '  maximum rescaled R = %e\n', max(R2(:)) );

%%% Threshold the scaled Radon transform, but only plot
%%% a maximum of 'maxlines' linear features.

sortR = sort(R2(:));
[sizeR,dummy] = size(sortR);
midR = sortR( sizeR-maxlines );
if midR > thresh 
  thresh2 = midR;
  fprintf( 1, '  WARNING: too many lines found (max=%d)!!\n', maxlines );
  fprintf( 1, '           threshold reset to %e\n', thresh2 ); 
elseif maxfac/fac < thresh
  thresh2 = maxfac/fac;
  fprintf( 1, '  no lines found, but plotting the maximum point anyway...\n' );
else  
  thresh2 = thresh;
end;

[ii,jj,R2find] = find( R2 >= thresh2 );
[iinx,iiny] = size(ii);
fprintf( 1, '  plotting %d linear features...\n', iinx );

%%% Plot the original image, and then add the linear
%%% features in red.

imagesc(tI), colormap('gray'), colormap(1-colormap)
title( 'Original image + linear features' );

hold on
ymid = round(ny/2.0);
uu = [-ny:ny];

for i = 1:iinx
  pt = th( jj(i) ) / 180.0*pi;
  px = xp( ii(i) );
  xx = px * cos(pt) - uu * sin(pt) + ymid;
  yy = px * sin(pt) + uu * cos(pt) + ymid;
  yy = ny - yy;
  plot( xx, yy, 'r.' ); 
end
hold off

fprintf( 1, 'Done.\n\n' );
ret = 0;

