/**
 * @file permgrp.h
 * @brief Public interface for permutation group class.
 *
 * Public interface for permutation group class. This class implements a minimal
 * set of low-level operations on a permutation group as well as the action of a
 * permutation group on the point set and subsets of the point set.
 *
 * @author Thomas Pender
 * @date 2025-07
 * @copyright GNU Public License
 */
# ifndef INCLUDED_PERMGRP_H
# define INCLUDED_PERMGRP_H

# define _GNU_SOURCE
# include <stdint.h>
# include <stddef.h>
# include <stdlib.h>
# include <errno.h>
# include <error.h>

# ifdef HAVE_GMP_H
# include <stdio.h>
# include <gmp.h>
# endif

# include <containers/containers.h>

typedef struct permgrp_t* permgrp_t;

/**
 * @brief Instantiates a <tt>permgrp_t</tt> instance.
 *
 * A permutation group generated by <tt>gens</tt> is returned. If specified, the
 * initial segment of the groups base will be given by the parameter <tt>base</tt>.
 *
 * <dl>
 * <dt><strong>Unchecked Runtime Errors</strong></dt>
 * <dd><tt>gens</tt> does not have dimensions <tt>ngens</tt> x <tt>n</tt></dd>.
 * <dd><tt>base</tt>, if non<tt>NULL</tt>, does not have length <tt>b</tt></dd>.
 * </dl>
 *
 * @param[in] gens Array of generators for permutation group.
 * @param[in] ngens Number of generators.
 * @param[in] n Degree of generators.
 * @param[in] base Prefix of base if non<tt>NULL</tt>.
 * @param[in] b Length of base prefix <tt>base</tt>.
 * @return AN opaque pointer to <tt>permgrp_t</tt> structure.
 */
extern permgrp_t permgrp_new(const uint32_t *restrict gens[static 1], size_t ngens,
                             size_t n, const uint32_t *restrict base, size_t b);

/**
 * @brief Returns degree of permutation group.
 *
 * Returns degree of permutation group.
 *
 * <dl>
 * <dt><strong>Unchecked Runtime Errors</strong></dt>
 * <dd>Calling <tt>permgrp_degree</tt> on a <tt>NULL</tt> group.</dd>
 * </dl>
 *
 * @param[in] G Permutation group.
 * @return Permutation group degree.
 */
extern size_t permgrp_degree(permgrp_t G);

/**
 * @brief Return length of base of permutation group.
 *
 * Return length of base of permutation group.
 *
 * <dl>
 * <dt><strong>Unchecked Runtime Errors</strong></dt>
 * <dd>Calling <tt>permgrp_baselen</tt> on a <tt>NULL</tt> group.</dd>
 * </dl>
 *
 * @param[in] G Permutation group.
 * @return Length of base.
 */
extern size_t permgrp_baselen(permgrp_t G);

/**
 * @brief Free memory allocated for <tt>permgrp_t</tt> instance.
 *
 * Free memory allocated for <tt>permgrp_t</tt> instance.
 *
 * @param[in] G Pointer to opaque pointer type.
 */
extern void permgrp_free(permgrp_t *G);

/**
 * @brief Returns order of permutation group.
 *
 * Returns order of permutation group.
 *
 * @param[in] G Permutation group.
 * @return Order of permutation group.
 * @warning If the group is large, there is a danger of integer overflow. If there
 * is a danger of this, then you may consider using <tt>permgrp_order</tt> which
 * interfaces with the GNU GMP library.
 */
extern size_t permgrp_orderfast(permgrp_t G);

/**
 * @brief Returns order of ith stabilizer subgroup in stabilizer chain.
 *
 * Returns order of ith stabilizer subgroup in stabilizer chain.
 *
 * <dl>
 * <dt><strong>Unchecked Runtime Errors</strong></dt>
 * <dd>Calling <tt>permgrp_sbgrporderfast</tt> on a <tt>NULL</tt> group.</dd>
 * <dd>Index parameter is larger than the base's length.</dd>
 * </dl>
 *
 * @param[in] G Permutation group.
 * @param[in] i Index of stabilizer subgroup.
 * @return Order of permutation group.
 * @warning If the group is large, there is a danger of integer overflow. If there
 * is a danger of this, then you may consider using <tt>permgrp_sbgrporder</tt>
 * which interfaces with the GNU GMP library.
 */
extern size_t permgrp_sbgrporderfast(permgrp_t G, size_t i);

# ifdef HAVE_GMP_H
/**
 * @brief Returns order of permutation group.
 *
 * Returns order of permutation group.
 *
 * <dl>
 * <dt><strong>Unchecked Runtime Errors</strong></dt>
 * <dd>Multiprecision integer parameter is not properly allocated.</dd>
 * </dl>
 *
 * @param[in] n Multiprecision integer to hold the order of the group.
 * @param[in] G Permutation group.
 * @return Order of permutation group.
 */
extern void permgrp_order(mpz_t n, permgrp_t G);

/**
 * @brief Returns order of ith stabilizer subgroup in stabilizer chain.
 *
 * Returns order of ith stabilizer subgroup in stabilizer chain.
 *
 * <dl>
 * <dt><strong>Unchecked Runtime Errors</strong></dt>
 * <dd>Calling <tt>permgrp_sbgrporderfast</tt> on a <tt>NULL</tt> group.</dd>
 * <dd>Index parameter is larger than the base's length.</dd>
 * <dd>Multiprecision integer parameter is not properly allocated.</dd>
 * </dl>
 *
 * @param[in] G Permutation group.
 * @param[in] i Index of stabilizer subgroup.
 * @return Order of permutation group.
 */
extern void permgrp_sbgrporder(mpz_t n, permgrp_t G, size_t i);
# endif

/**
 * @brief Enlarge group by adding a generator.
 *
 * Enlarge group by adding a generator. In effect, we assign G <-- <G, g>.
 *
 * <dl>
 * <dt><strong>Unchecked Runtime Errors</strong></dt>
 * <dd>Calling <tt>permgrp_update</tt> on a <tt>NULL</tt> group.</dd>
 * <dd>The degree of <tt>g</tt> is different than that of <tt>G</tt>.</dd>
 * </dl>
 *
 * @param[in] G Permutation group being added to.
 * @param[in] g Generator being added to permutation group.
 */
extern void permgrp_update(permgrp_t G, const uint32_t g[static 1]);

/**
 * @brief Change base of permutation group.
 *
 * Change base of permutation group. The base is changed to one with prefix given
 * by the parameter <tt>base</tt>.
 *
 * <dl>
 * <dt><strong>Unchecked Runtime Errors</strong></dt>
 * <dd><tt>G</tt> or <tt>*G</tt> is <tt>NULL</tt>.</dd>
 * <dd><tt>base</tt>, if non<tt>NULL</tt>, does not have length <tt>b</tt></dd>.
 * </dl>
 *
 * @param[in,out] G Pointer to opaque pointer type.
 * @param[in] base Base prefix.
 * @param[in] b Length of prefix <tt>base</tt>.
 */
extern void permgrp_new_base(permgrp_t G[static 1],const uint32_t *base,size_t b);

/**
 * @brief Return symmetric group of degree <tt>n</tt>.
 *
 * Return symmetric group of degree <tt>n</tt>.
 *
 * @param[in] n Degree of symmetric group.
 * @return Symmetric group of degree <tt>n</tt>.
 */
extern permgrp_t permgrp_sym(size_t n);

/**
 * @brief Return <tt>i</tt>th fundamental orbit of permutation group.
 *
 * Return <tt>i</tt>th fundamental orbit of permutation group.
 *
 * <dl>
 * <dt><strong>Unchecked Runtime Errors</strong></dt>
 * <dd>Calling <tt>permgrp_fundorb</tt> on <tt>NULL</tt> permutation group.</dd>
 * </dl>
 *
 * @param[in] G Permutation group.
 * @param[in] i Index of fundamental orbit.
 * @return Array containing points of <tt>i</tt>th fundamental orbit.
 */
extern uint32_t *permgrp_fundorb(permgrp_t G, size_t i);

/**
 * @brief Update binary array representation of orbit decomposition of <tt>i</tt>th
 * stabilizer subgroup.
 *
 * Update binary array representation of orbit decomposition of <tt>i</tt>th
 * stabilizer subgroup. The <tt>j</tt>th entry is 1 if <tt>j</tt> is the minimal
 * representative of its orbit under the given action. The <tt>j</tt>th entry is 0
 * otherwise.
 *
 * <dl>
 * <dt><strong>Unchecked Runtime Errors</strong></dt>
 * <dd>Calling <tt>permgrp_orbrep_update</tt> on <tt>NULL</tt> permutation
 * group.</dd>
 * <dd>The length of <tt>orbrep</tt> is not equal to the degree of the
 * permuation group.</dd>
 * <dd>The parameter <tt>i</tt> is larger than the length of the base.</dd>
 * </dl>
 *
 * @param[in] G Permutation group.
 * @param[in,out] orbrep Array to hold orbit decomposition representation.
 * @param[in] i Index of acting stabilizer subgroup.
 */
extern void permgrp_orbrep_update(permgrp_t G,uint32_t orbrep[static 1],size_t i);

/**
 * @brief Return the length of the ith fundamental orbit.
 *
 * Return the length of the ith fundamental orbit.
 *
 * <dl>
 * <dt><strong>Unchecked Runtime Errors</strong></dt>
 * <dd>Calling <tt>permgrp_fundorb_size</tt> on a <tt>NULL</tt> group.</dd>
 * <dd><tt>i</tt> is larger than the length of the group's base.</dd>
 * </dl>
 *
 * @param[in] G Permutation group.
 * @param[in] i Index of fundamental orbit.
 *
 * @return The length of the ith fundamental orbit.
 */
extern size_t permgrp_fundorb_size(permgrp_t G, size_t i);

/**
 * @brief Return array containing ordering of points induced by the group's base.
 *
 * Return array containing ordering of points induced by the group's base.
 *
 *<dl>
 * <dt><strong>Unchecked Runtime Errors</strong></dt>
 * <dd>Calling <tt>permgrp_ordering</tt> on a <tt>NULL</tt> group.</dd>
 * </dl>
 *
 * @param[in] G Permutation group.
 *
 * @return Array containing ordering of points induced by the group's base.
 */
extern uint32_t *permgrp_ordering(permgrp_t G);

/**
 * @brief Return transversal element.
 *
 * Return transversal element mapping.
 *
 * @param[in] G Permutation group.
 * @param[in] i Index of fundamental orbit.
 * @param[in] gamma Element of ith fundamental orbit.
 *
 * @return Return transversal element mapping the ith base point to <tt>gamma</tt>
 * or <tt>NULL</tt> if <tt>gamma</tt> is not in the ith fundamental orbit.
 */
extern uint32_t *permgrp_rep(permgrp_t G, size_t i, uint32_t gamma);

/**
 * @brief Return inverse of transversal element.
 *
 * Return inverse of transversal element mapping.
 *
 * @param[in] G Permutation group.
 * @param[in] i Index of fundamental orbit.
 * @param[in] gamma Element of ith fundamental orbit.
 *
 * @return Return inverse transversal element mapping the ith base point to
 * <tt>gamma</tt> or <tt>NULL</tt> if <tt>gamma</tt> is not in the ith fundamental
 * orbit.
 */
extern uint32_t *permgrp_repinv(permgrp_t G, size_t i, uint32_t gamma);

/**
 * @brief Return base of permutation group.
 *
 * Return base of permutation group.
 *
 * <dl>
 * <dt><strong>Unchecked Runtime Errors</strong></dt>
 * <dd>Calling <tt>permgrp_base</tt> on a <tt>NULL</tt> group.</dd>
 * </dl>
 *
 * @param[in] G Permutation group.
 *
 * @return Array containing the base of the group.
 */
extern uint32_t *permgrp_base(permgrp_t G);

/**
 * @brief Determines longest prefix of base fixed by permutation.
 *
 * Determines longest prefix of base fixed by permutation.
 *
 * <dl>
 * <dt><strong>Unchecked Runtime Errors</strong></dt>
 * <dd>Calling <tt>permgrp_fixed_base</tt> on a <tt>NULL</tt> group.</dd>
 * <dd>Degree of <tt>p</tt> is different than the degree of the group.</dd>
 * </dl>
 *
 * @param[in] G Permutation Group.
 * @param[in] p Permutation.
 *
 * @return Index of first base element such that all previous base elements are
 * fixed by the permutation but this one is not. If the entire base is fixed by
 * the permutation, then the length of the base is returned.
 */
extern size_t permgrp_fixed_base(permgrp_t G, const uint32_t p[static 1]);

/**
 * @brief Return array of generators of group.
 *
 * Return array of generators of group. Likely not a strong generating set of the
 * group.
 *
 * @param[in] G Permutation Group.
 *
 * @return Array object containing generators of group. If the group is trivial or
 * <tt>NULL</tt>, a <tt>NULL</tt> pointer is returned.
 */
extern arrays_t permgrp_generators(permgrp_t G);

/**
 * @brief Return array of generators of subgroup in stabilizer chain.
 *
 * Return array of generators of subgroup in stabilizer chain. Likely not a strong
 * generating set of the group.
 *
 * <dl>
 * <dt><strong>Unchecked Runtime Errors</strong></dt>
 * <dd>Index is out of range.</dd>
 * </dl>
 *
 * @param[in] G Permutation Group.
 * @param[in] i Index of subgroup in stabilizer chain.
 *
 * @return Array object containing generators of subgroup in stabilizer chain. If
 * the group is trivial or <tt>NULL</tt>, a <tt>NULL</tt> pointer is returned.
 */
extern arrays_t permgrp_generatorssbgrp(permgrp_t G, size_t i);

/**
 * @brief Return shortest nonredundant prefix of base.
 *
 * Return shortest nonredundant prefix of base.
 *
 * <dl>
 * <dt><strong>Unchecked Runtime Errors</strong></dt>
 * <dd>Calling <tt>permgrp_effective_len</tt> on a <tt>NULL</tt> group.</dd>
 * </dl>
 *
 * @param[in] G Permutation group.
 *
 * @return Length of shortest nonredundant prefix of base.
 */
extern size_t permgrp_effective_len(permgrp_t G);

/**
 * @brief Return strong generating set of generators as an array.
 *
 * Return strong generating set of generators as an array.
 *
 * @param[in] G Permutation Group.
 *
 * @return Array object containing generators of group. If the group is trivial or
 * <tt>NULL</tt>, a <tt>NULL</tt> pointer is returned.
 */
extern arrays_t permgrp_sgs(permgrp_t G);

/**
 * @brief Returns stabilizer of a set of points.
 *
 * Returns stabilizer of a set of points.
 *
 * @param[in] G Pointer to permutation group.
 * @param[in] s Set being stabilized.
 * @param[in] m NUmber of <tt>setwords</tt> needed to hold the set.
 *
 * @return Subgroup of permutation group that stabilizes the set.
 */
extern permgrp_t permgrp_setstab(permgrp_t G[static 1],
                                 sets_t s[static 1], size_t m);

/**
 * @brief Swap two distinct instances of permutation groups.
 *
 * Swap two distinct instances of permutation groups.
 *
 * <dl>
 * <dt><strong>Unchecked Runtime Errors</strong></dt>
 * <dd><tt>G1</tt> and <tt>G2</tt> are aliases.</dd>
 * </dl>
 *
 * @param[in] G1 First permutation group.
 * @param[in] G2 Second permutation group.
 */
static inline
void permgrp_swap(permgrp_t *restrict G1, permgrp_t *restrict G2)
{
  volatile permgrp_t tmp = *G1;
  *G1 = *G2;
  *G2 = tmp;
}

# endif
